package server

import (
	"fmt"
	"net/http"
	"net/url"
	"strings"
	"time"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cache"
	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/uautil"
)

const sourceArch string = "source"
const minbyHashAPTVersion uint64 = 0x1000200000000 //APT version 1.2.0 as result of (1<<48 + 2<<32)

var allowedFiles = map[string]bool{
	"Release":      true,
	"Release.gpg":  true,
	"InRelease":    true,
	"Packages":     true,
	"Packages.gz":  true,
	"Packages.bz2": true,
	"Sources":      true,
	"Sources.gz":   true,
	"Sources.bz2":  true,
}

type RepoHandler struct {
	enforceSessions bool
	db              *cacus.DBClient
	cache           *cache.DistCache
}

type Session interface {
	Data() *cache.DataItem
	Bitmap() cache.RequestBitmap
	SetBitmap(cache.RequestBitmap)
}

type dummySession struct {
	DataItem      *cache.DataItem
	RequestBitmap cache.RequestBitmap
}

func (d *dummySession) Data() *cache.DataItem {
	return d.DataItem
}

func (d *dummySession) Bitmap() cache.RequestBitmap {
	return d.RequestBitmap
}

func (d *dummySession) SetBitmap(bitmap cache.RequestBitmap) {
	d.RequestBitmap = bitmap
}

func NewRepoHandler(enforceSessions bool, db *cacus.DBClient, c *cache.DistCache) RepoHandler {
	return RepoHandler{enforceSessions: enforceSessions, db: db, cache: c}
}

func (h *RepoHandler) Handle(w http.ResponseWriter, r *http.Request) {
	var enableSession bool
	file := chi.URLParam(r, "file")
	arch := chi.URLParam(r, "arch")
	bundleRequest := allowedFiles[file]
	if !bundleRequest {
		if arch != sourceArch {
			w.WriteHeader(http.StatusNotFound)
			return
		}
	}
	repo := chi.URLParam(r, "repo")
	env := chi.URLParam(r, "env")
	v, err := uautil.GetAPTVersion(r)
	if err != nil {
		enableSession = h.enforceSessions
		ua := r.Header.Get("User-Agent")
		if ua != "" && strings.HasPrefix(ua, "Debian") {
			logger.Errorf("Cannot parse User-Agent: '%s': %s", ua, err)
		}
	} else {
		if v < minbyHashAPTVersion {
			enableSession = true
		} else {
			enableSession = h.enforceSessions
		}
	}
	ctx := r.Context()

	if bundleRequest { // request is made to one of session-cached files
		if enableSession {
			sessionItem, err := h.cache.GetSessionItem(ctx, repo, env, arch, r.RemoteAddr)
			if err != nil {
				logger.Error(err)
				w.WriteHeader(http.StatusNotFound)
				return
			}
			sessionData := sessionItem.Value().(*cache.SessionItem)
			if sessionData.DataItem.Value().(*cache.DataItem).Bundle.Incomplete {
				w.Header().Set("X-Request-State", "Incomplete data served to silence APT errors")
				logger.Debugf("%s[%s]: served incomplete request!", r.RequestURI, r.RemoteAddr)
			}
			err = processSessionData(w, r, sessionData, repo, env, arch, file)
			readyFlag := h.cache.SetSessionCleanup(repo, env, arch, r.RemoteAddr)
			if sessionData.RequestBitmap&cache.BundleFetched == cache.BundleFetched || err != nil {
				readyFlag <- true
			}
			return
		} else {
			data, err := h.cache.GetDataBundle(ctx, repo, env, arch, false)
			if err != nil {
				logger.Errorf("Failed to get bundle for request %s: %s", r.RequestURI, err)
				w.WriteHeader(http.StatusNotFound)
				return
			}
			session := dummySession{DataItem: data}
			_ = processSessionData(w, r, &session, repo, env, arch, file)
			return
		}
	} else { // request is made to one of source files
		// chi uses raw escaped url during routing: https://github.com/go-chi/chi/commit/822e7b85e22b3f7a573782c3674edf8e732fb427
		// should unescape requested filename to avoid mongo special effects
		file, err := url.PathUnescape(file)
		if err != nil {
			logger.Errorf("%s[%s]: Failed to unescape file: %v\n", r.RequestURI, r.RemoteAddr, err)
			w.WriteHeader(http.StatusBadRequest)
			return
		}
		storageKey, err := h.cache.GetSourceStorageKey(ctx, repo, env, file)
		if err != nil {
			logger.Errorf("%s[%s]: error serving request: %s", r.RequestURI, r.RemoteAddr, err)
			w.WriteHeader(http.StatusInternalServerError)
			n, err := w.Write([]byte(err.Error()))
			if err != nil {
				logger.Errorf("Request GET %s interrupted after %d bytes written: %s", r.RequestURI, n, err)
			}
			return
		}
		http.Redirect(w, r, storageKey, http.StatusMovedPermanently)
		return
	}
}

func processSessionData(w http.ResponseWriter, r *http.Request, sessionData Session, repo, env, arch, file string) error {
	var ifModifiedSince time.Time
	var err error
	if r.Header.Get("If-Modified-Since") != "" {
		ifModifiedSince, err = http.ParseTime(r.Header.Get("If-Modified-Since"))
		if err != nil {
			logger.Debugf("%s[%s]: cannot parse time string '%s': %s", r.RequestURI, r.RemoteAddr,
				r.Header.Get("If-Modified-Since"), err)
		}
	}
	lastModified := sessionData.Data().UpdatedAt.Truncate(time.Second)
	w.Header().Set("Last-Modified", lastModified.Format(http.TimeFormat))
	if lastModified.Equal(ifModifiedSince) || lastModified.Before(ifModifiedSince) {
		w.WriteHeader(http.StatusNotModified)
		return nil
	}

	switch {
	case file == "Packages" || file == "Sources":
		n, err := w.Write(sessionData.Data().Bundle.Plain)
		if err != nil {
			logger.Errorf("Request GET /%s/%s/%s/%s interrupted after %d bytes written: %s", repo, env, arch, file, n, err)
			return err
		}
		sessionData.SetBitmap(sessionData.Bitmap() | cache.PlainFetched | cache.DataFetched)
	case file == "Packages.gz" || file == "Sources.gz":
		n, err := w.Write(sessionData.Data().Bundle.GZIPed)
		if err != nil {
			logger.Errorf("Request GET /%s/%s/%s/%s interrupted after %d bytes written: %s", repo, env, arch, file, n, err)
			return err
		}
		sessionData.SetBitmap(sessionData.Bitmap() | cache.GzippedFetched | cache.DataFetched)
	case file == "Packages.bz2" || file == "Sources.bz2":
		n, err := w.Write(sessionData.Data().Bundle.BZIP2ed)
		if err != nil {
			logger.Errorf("Request GET /%s/%s/%s/%s interrupted after %d bytes written: %s", repo, env, arch, file, n, err)
			return err
		}
		sessionData.SetBitmap(sessionData.Bitmap() | cache.BzippedFetched | cache.DataFetched)
	case file == "Release":
		n, err := w.Write(getReleaseFileForRequest(r, sessionData.Data()))
		if err != nil {
			logger.Errorf("Request GET /%s/%s/%s/%s interrupted after %d bytes written: %s", repo, env, arch, file, n, err)
			return err
		}
		sessionData.SetBitmap(sessionData.Bitmap() | cache.ReleaseFetched)
		if sessionData.Bitmap()&(cache.ReleaseFetched|cache.ReleaseGPGFetched) == (cache.ReleaseFetched | cache.ReleaseGPGFetched) {
			sessionData.SetBitmap(sessionData.Bitmap() | cache.SignatureFetched)
		}
	case file == "Release.gpg":
		n, err := w.Write(getReleaseGPGFileForRequest(r, sessionData.Data()))
		if err != nil {
			logger.Errorf("Request GET /%s/%s/%s/%s interrupted after %d bytes written: %s", repo, env, arch, file, n, err)
			return err
		}
		sessionData.SetBitmap(sessionData.Bitmap() | cache.ReleaseGPGFetched)
		if sessionData.Bitmap()&(cache.ReleaseFetched|cache.ReleaseGPGFetched) == (cache.ReleaseFetched | cache.ReleaseGPGFetched) {
			sessionData.SetBitmap(sessionData.Bitmap() | cache.SignatureFetched)
		}
	case file == "InRelease":
		n, err := w.Write(getInReleaseFileForRequest(r, sessionData.Data()))
		if err != nil {
			logger.Errorf("Request GET /%s/%s/%s/%s interrupted after %d bytes written: %s", repo, env, arch, file, n, err)
			return err
		}
		sessionData.SetBitmap(sessionData.Bitmap() | cache.InReleaseFetched | cache.SignatureFetched)
	default:
		w.WriteHeader(http.StatusNotFound)
		logger.Errorf("Request GET /%s/%s/%s/%s cannot be served!", repo, env, arch, file)
		return fmt.Errorf("request GET /%s/%s/%s/%s cannot be served", repo, env, arch, file)
	}
	return nil
}

func chooseFile(r *http.Request, bh bool, p, h []byte) []byte {
	if !bh {
		return p
	}
	v, err := uautil.GetAPTVersion(r)
	if err != nil {
		return h
	}
	if v >= minbyHashAPTVersion {
		return h
	} else {
		return p
	}
}

func getReleaseFileForRequest(r *http.Request, d *cache.DataItem) []byte {
	return chooseFile(r, d.Bundle.ByHash, d.Bundle.Release, d.Bundle.ReleaseByHash)
}

func getReleaseGPGFileForRequest(r *http.Request, d *cache.DataItem) []byte {
	return chooseFile(r, d.Bundle.ByHash, d.Bundle.ReleaseGPG, d.Bundle.ReleaseGPGByHash)
}

func getInReleaseFileForRequest(r *http.Request, d *cache.DataItem) []byte {
	return chooseFile(r, d.Bundle.ByHash, d.Bundle.InRelease, d.Bundle.InReleaseByHash)
}
