package server

import (
	"encoding/json"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"go.mongodb.org/mongo-driver/bson"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/internal/util"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

const (
	distURL = "http://dist.yandex.ru"
)

type searchArgs struct {
	repo        string
	pkg         string
	ver         string
	env         string
	descr       string
	lang        string
	sort        string
	offset      uint64
	offsetIsSet bool
	limit       uint64
	limitIsSet  bool
	strict      bool
	sources     bool
	urls        bool
	human       bool
}

type searchResults struct {
	Result  []searchResult `json:"result"`
	Success bool           `json:"success"`
	Total   int            `json:"total"`
}

type searchResult struct {
	Repo    string              `json:"repository"`
	Env     string              `json:"environment"`
	Source  string              `json:"source"`
	Version string              `json:"version"`
	DEBs    []searchDEBEntry    `json:"debs"`
	Sources []searchSourceEntry `json:"sources"`
}

type searchDEBEntry struct {
	Arch        string `json:"architecture"`
	Description string `json:"description"`
	Pkg         string `json:"package"`
	Storage     string `json:"storage_key"`
}

type searchSourceEntry struct {
	Name    string `json:"name"`
	Storage string `json:"storage_key"`
}

type searchPackage struct {
	repo string
	pkg  cacus.RepoDocument
}

type humanColumn struct {
	text   string
	maxLen int
}

type humanRow struct {
	columns    []humanColumn
	topLine    bool
	bottomLine bool
}

func (h *APIHandler) searchHandler(w http.ResponseWriter, r *http.Request) {
	ctx := r.Context()
	err := r.ParseForm()
	if err != nil {
		logger.Errorf("%s[%s]: failed to parse form", r.RequestURI, r.RemoteAddr)
		w.WriteHeader(500)
		return
	}
	sa, err := parseSearchArgs(r.Form)
	if err != nil {
		logger.Errorf("%s[%s]: %s", r.RequestURI, r.RemoteAddr, err)
		w.WriteHeader(500)
		_, err := fmt.Fprintf(w, "cannot parse request params: %s", err)
		util.HandleError(err)
		return
	}

	if sa.human && (sa.offsetIsSet || sa.limitIsSet || sa.sort != "") {
		logger.Errorf("%s[%s]: limit, offset or sort set for plaintext", r.RequestURI, r.RemoteAddr)
	}

	sel := generateSelector(sa)
	var repoList *[]string
	if sa.repo != "" {
		repoList = &[]string{sa.repo}
	} else {
		repoList, err = h.db.GetRepoList(ctx)
		if err != nil {
			logger.Errorf("%s[%s]: failed to get repo list from db: %s", r.RequestURI, r.RemoteAddr, err)
			w.WriteHeader(500)
			return
		}
	}
	foundPackages := make([]searchPackage, 0)
	for _, repo := range *repoList {
		packages, err := h.db.FindPackages(ctx, repo, sel)
		if err != nil {
			logger.Errorf("%s[%s]: failed to find packages in %s", r.RequestURI, r.RemoteAddr, repo)
			w.WriteHeader(500)
			return
		}
		for _, pkg := range *packages {
			foundPackages = append(foundPackages, searchPackage{repo: repo, pkg: pkg})
		}
	}
	totalPackages := len(foundPackages)
	if sa.sort != "" {
		err = sortFoundPackages(&foundPackages, sa.sort)
		if err != nil {
			logger.Errorf("%s[%s]: failed to sort packages", r.RequestURI, r.RemoteAddr)
			w.WriteHeader(500)
			_, err := fmt.Fprintf(w, "%s", err)
			util.HandleError(err)
			return
		}
	}
	if !sa.human && (sa.limitIsSet || sa.offsetIsSet) {
		err := applyLimitOffset(&foundPackages, sa)
		if err != nil {
			w.WriteHeader(500)
			_, err := fmt.Fprintf(w, "%s", err)
			util.HandleError(err)
			return
		}
	}
	results := processFoundPackages(&foundPackages, totalPackages)
	code, rendered, err := renderResults(results, sa)
	if err != nil {
		logger.Errorf("%s[%s]: failed to render results", r.RequestURI, r.RemoteAddr)
		w.WriteHeader(500)
		return
	}
	w.WriteHeader(code)
	n, err := w.Write(rendered)
	if err != nil {
		logger.Errorf("Request GET %s interrupted after %d bytes written: %s", r.RequestURI, n, err)
		return
	}
}

func applyLimitOffset(sp *[]searchPackage, sa *searchArgs) error {
	totalResults := len(*sp)
	if sa.offset > uint64(totalResults) {
		return fmt.Errorf("ERROR: offset > total results: %d > %d", sa.offset, totalResults)
	}
	endIdx := sa.offset + sa.limit
	if endIdx > uint64(totalResults) {
		endIdx = uint64(totalResults)
	}
	limited := (*sp)[sa.offset:endIdx]
	*sp = limited
	return nil
}

func sortFoundPackages(packages *[]searchPackage, sort string) error {
	order := strings.Split(sort, ",")
	orderList := make([]lessFunc, 0)
	for _, fn := range order {
		if strings.HasPrefix(fn, " ") {
			fn = fmt.Sprintf("+%s", strings.TrimSpace(fn))
		}
		if !(strings.HasPrefix(fn, "+") || strings.HasPrefix(fn, "-")) {
			fn = fmt.Sprintf("+%s", fn)
		}
		if _, ok := sortMap[fn]; ok {
			orderList = append(orderList, sortMap[fn])
		} else {
			return fmt.Errorf("sorting order %s is unknown", fn)
		}

	}
	OrderedBy(orderList...).Sort(packages)
	return nil
}

func getBool(b string) bool {
	return strings.ToLower(b) == "true"
}

func generateSelector(sa *searchArgs) bson.M {
	var sel = make(bson.M)
	if sa.pkg != "" {
		sel["$or"] = make([]bson.M, 0)
	}

	if !sa.strict {
		if sa.pkg != "" {
			sel["$or"] = append(sel["$or"].([]bson.M), bson.M{"Source": bson.M{"$regex": sa.pkg}})
			sel["$or"] = append(sel["$or"].([]bson.M), bson.M{"debs.Package": bson.M{"$regex": sa.pkg}})
			if sa.sources {
				sel["$or"] = append(sel["$or"].([]bson.M), bson.M{"sources.name": bson.M{"$regex": sa.pkg}})
			}
		}
	} else {
		if sa.pkg != "" {
			sel["$or"] = append(sel["$or"].([]bson.M), bson.M{"Source": sa.pkg})
			sel["$or"] = append(sel["$or"].([]bson.M), bson.M{"debs.Package": sa.pkg})
			if sa.sources {
				sel["$or"] = append(sel["$or"].([]bson.M), bson.M{"sources.name": sa.pkg})
			}
		}
	}

	if sa.ver != "" {
		if !sa.strict {
			sel["Version"] = bson.M{"$regex": sa.ver}
		} else {
			sel["Version"] = sa.ver
		}
	}

	if sa.env != "" {
		sel["environment"] = sa.env
	}

	if sa.descr != "" {
		if sa.lang != "" {
			sel["$text"] = bson.M{"$search": sa.descr, "$language": sa.lang}
		} else {
			sel["$text"] = bson.M{"$search": sa.descr}
		}
	}

	return sel
}

func parseSearchArgs(form url.Values) (*searchArgs, error) {
	sa := searchArgs{}
	sa.repo = form.Get("repo")
	sa.pkg = form.Get("pkg")
	sa.ver = form.Get("ver")
	sa.env = form.Get("env")
	sa.descr = form.Get("descr")
	sa.lang = form.Get("lang")
	sa.sort = form.Get("sort")

	limitString := form.Get("limit")
	if limitString != "" {
		limitValue, err := strconv.ParseUint(limitString, 10, 64)
		if err != nil {
			return nil, err
		}
		sa.limit = limitValue
		sa.limitIsSet = true
	} else {
		sa.limitIsSet = false
	}

	offsetString := form.Get("offset")
	if offsetString != "" {
		offsetValue, err := strconv.ParseUint(offsetString, 10, 64)
		if err != nil {
			return nil, err
		}
		sa.offset = offsetValue
		sa.offsetIsSet = true
	} else {
		sa.offsetIsSet = false
	}

	sa.strict = getBool(form.Get("strict"))
	sa.sources = getBool(form.Get("sources"))
	sa.urls = getBool(form.Get("withurls"))
	sa.human = getBool(form.Get("human"))
	return &sa, nil
}

func processFoundPackages(packages *[]searchPackage, totalPackages int) *searchResults {
	result := searchResults{Result: make([]searchResult, 0)}
	for _, pkg := range *packages {
		sr := searchResult{
			Repo:    pkg.repo,
			Env:     pkg.pkg.Environment,
			Source:  pkg.pkg.Source,
			Version: pkg.pkg.Version,
		}
		for _, src := range pkg.pkg.Sources {
			sr.Sources = append(sr.Sources, searchSourceEntry{
				Name:    src.Name,
				Storage: storageURL(src.StorageKey),
			})
		}
		for _, deb := range pkg.pkg.Debs {
			pkgName := deb.StorageKey[strings.LastIndex(deb.StorageKey, "/")+1:]
			sr.DEBs = append(sr.DEBs, searchDEBEntry{
				Pkg:         pkgName,
				Storage:     storageURL(deb.StorageKey),
				Arch:        deb.Architecture,
				Description: deb.Description,
			})
		}
		result.Result = append(result.Result, sr)
	}
	if len(*packages) > 0 || totalPackages > 0 {
		result.Success = true
	}
	result.Total = totalPackages
	return &result
}

func storageURL(key string) string {
	return fmt.Sprintf("%s%s", distURL, key)
}

func renderResults(res *searchResults, sa *searchArgs) (int, []byte, error) {
	var rendered []byte
	var err error
	var code int
	if !sa.human || !res.Success {
		rendered, err = json.Marshal(res)
		if res.Success {
			code = 200
		} else {
			code = 404
		}
		if err != nil {
			return 500, nil, err
		}
	} else {
		rendered = renderHumanResults(res, sa)
		code = 200
	}
	return code, rendered, nil
}

func renderHumanResults(res *searchResults, sa *searchArgs) []byte {
	table := make([]humanRow, 0)
	var maxRepoLen, maxEnvLen, maxPackageLen, maxURLLen = 4, 3, 7, 3
	headerRow := humanRow{
		topLine:    true,
		bottomLine: true,
		columns: []humanColumn{
			{
				text: "repo",
			},
			{
				text: "env",
			},
			{
				text: "package",
			},
		},
	}
	if sa.urls {
		headerRow.columns = append(headerRow.columns, humanColumn{text: "url"})
	}
	table = append(table, headerRow)
	for _, pkg := range res.Result {
		for _, deb := range pkg.DEBs {
			row := humanRow{
				columns: []humanColumn{
					{
						text: pkg.Repo,
					},
					{
						text: pkg.Env,
					},
					{
						text: deb.Pkg,
					},
				},
			}
			if sa.urls {
				row.columns = append(row.columns, humanColumn{text: deb.Storage})
				if len(deb.Storage) > maxURLLen {
					maxURLLen = len(deb.Storage)
				}
			}
			if len(pkg.Repo) > maxRepoLen {
				maxRepoLen = len(pkg.Repo)
			}
			if len(pkg.Env) > maxEnvLen {
				maxEnvLen = len(pkg.Env)
			}
			if len(deb.Pkg) > maxPackageLen {
				maxPackageLen = len(deb.Pkg)
			}
			table = append(table, row)
		}
		for _, src := range pkg.Sources {
			row := humanRow{
				columns: []humanColumn{
					{
						text: pkg.Repo,
					},
					{
						text: pkg.Env,
					},
					{
						text: src.Name,
					},
				},
			}
			if sa.urls {
				row.columns = append(row.columns, humanColumn{text: src.Storage})
				if len(src.Storage) > maxURLLen {
					maxURLLen = len(src.Storage)
				}
			}
			if len(pkg.Repo) > maxRepoLen {
				maxRepoLen = len(pkg.Repo)
			}
			if len(pkg.Env) > maxEnvLen {
				maxEnvLen = len(pkg.Env)
			}
			if len(src.Name) > maxPackageLen {
				maxPackageLen = len(src.Name)
			}
			table = append(table, row)
		}
	}
	table[len(table)-1].bottomLine = true
	for i := range table {
		table[i].columns[0].maxLen = maxRepoLen
		table[i].columns[1].maxLen = maxEnvLen
		table[i].columns[2].maxLen = maxPackageLen
		if sa.urls {
			table[i].columns[3].maxLen = maxURLLen
		}
	}
	return []byte(printTable(&table))
}

func printTable(t *[]humanRow) string {
	var str strings.Builder
	for _, row := range *t {
		if row.topLine {
			lastRow := len(row.columns) - 1
			for i, col := range row.columns {
				cs := fmt.Sprintf("+-%s-", strings.Repeat("-", col.maxLen))
				if i == lastRow {
					cs = fmt.Sprintf("%s+\n", cs)
				}
				str.WriteString(cs)
			}
		}

		lastRow := len(row.columns) - 1
		for i, col := range row.columns {
			cf := fmt.Sprintf("| %%-%ds ", col.maxLen)
			cs := fmt.Sprintf(cf, col.text)
			if i == lastRow {
				cs = fmt.Sprintf("%s|\n", cs)
			}
			str.WriteString(cs)
		}

		if row.bottomLine {
			lastRow := len(row.columns) - 1
			for i, col := range row.columns {
				cs := fmt.Sprintf("+-%s-", strings.Repeat("-", col.maxLen))
				if i == lastRow {
					cs = fmt.Sprintf("%s+\n", cs)
				}
				str.WriteString(cs)
			}
		}

	}
	return str.String()
}
