package server

import (
	"fmt"
	"sort"
	"time"
)

var sortMap = make(map[string]lessFunc)
var dirSortMap = map[string]sortFunc{
	"upload_date": byUploadDate,
	"source":      bySource,
	"version":     byVersion,
	"repo":        byRepo,
}
var sortDirections = []string{"+", "-"}

type lessFunc func(p1, p2 *searchPackage) bool
type sortFunc func(p1, p2 *searchPackage, direction string) bool

type multiSorter struct {
	packages *[]searchPackage
	less     []lessFunc
}

func (ms *multiSorter) Sort(packages *[]searchPackage) {
	ms.packages = packages
	sort.Sort(ms)
}

// OrderedBy returns a Sorter that sorts using the less functions, in order.
// Call its Sort method to sort the data.
func OrderedBy(less ...lessFunc) *multiSorter {
	return &multiSorter{
		less: less,
	}
}

// Len is part of sort.Interface.
func (ms *multiSorter) Len() int {
	return len(*ms.packages)
}

// Swap is part of sort.Interface.
func (ms *multiSorter) Swap(i, j int) {
	a, b := (*ms.packages)[j], (*ms.packages)[i]
	(*ms.packages)[i], (*ms.packages)[j] = a, b
}

func (ms *multiSorter) Less(i, j int) bool {
	p, q := &(*ms.packages)[i], &(*ms.packages)[j]
	// Try all but the last comparison.
	var k int
	for k = 0; k < len(ms.less)-1; k++ {
		less := ms.less[k]
		switch {
		case less(p, q):
			// p < q, so we have a decision.
			return true
		case less(q, p):
			// p > q, so we have a decision.
			return false
		}
		// p == q; try the next comparison.
	}
	// All comparisons to here said "equal", so just return whatever
	// the final comparison reports.
	return ms.less[k](p, q)
}

func bySource(p1, p2 *searchPackage, direction string) bool {
	if direction == "-" {
		return p1.pkg.Source > p2.pkg.Source
	} else {
		return p1.pkg.Source < p2.pkg.Source
	}
}

func byVersion(p1, p2 *searchPackage, direction string) bool {
	if direction == "-" {
		return p1.pkg.Version > p2.pkg.Version
	} else {
		return p1.pkg.Version < p2.pkg.Version
	}
}

func byUploadDate(p1, p2 *searchPackage, direction string) bool {
	var u1, u2 time.Time
	for _, record := range p1.pkg.AuditMeta {
		if record.Event == "upload" && record.Timestamp.After(u1) {
			u1 = record.Timestamp
		}
	}
	for _, record := range p2.pkg.AuditMeta {
		if record.Event == "upload" && record.Timestamp.After(u2) {
			u2 = record.Timestamp
		}
	}
	if direction == "-" {
		return u1.After(u2)
	} else {
		return u2.After(u1)
	}
}

func byRepo(p1, p2 *searchPackage, direction string) bool {
	if direction == "-" {
		return p1.repo > p2.repo
	} else {
		return p1.repo < p2.repo
	}
}

func wrapLessFunc(f sortFunc, direction string) lessFunc {
	return func(f sortFunc, direction string) lessFunc {
		return func(p1, p2 *searchPackage) bool {
			return f(p1, p2, direction)
		}
	}(f, direction)
}

func init() {
	for key, fun := range dirSortMap {
		for _, direction := range sortDirections {
			sortName := fmt.Sprintf("%s%s", direction, key)
			sortMap[sortName] = wrapLessFunc(fun, direction)
		}
	}
}
