package server

import (
	"fmt"
	"net/http"
	"time"

	"github.com/go-chi/chi/v5"
	"github.com/go-chi/chi/v5/middleware"

	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cache"
	"a.yandex-team.ru/infra/dist/repo-daemon/internal/cacus"
	"a.yandex-team.ru/infra/dist/repo-daemon/pkg/logger"
)

type RepoDaemon struct {
	param cacus.RepoDaemon
	db    *cacus.DBClient
	cache *cache.DistCache
}

func NewRepoDaemon(param cacus.RepoDaemon, db *cacus.DBClient, c *cache.DistCache) RepoDaemon {
	return RepoDaemon{param: param, db: db, cache: c}
}

func (d *RepoDaemon) StartServer(debug bool) error {
	router := chi.NewRouter()

	router.Use(middleware.RequestID)
	router.Use(middleware.RealIP)
	middleware.DefaultLogger = middleware.RequestLogger(&middleware.DefaultLogFormatter{Logger: logger.Requests, NoColor: true})
	router.Use(middleware.Logger)
	router.Use(middleware.Recoverer)

	router.Get("/ping", PingHandler)
	router.Head("/ping", PingHandler)

	repoHandler := NewRepoHandler(d.param.EnforceSessions, d.db, d.cache)
	router.Get("/{repo}/{env}/{arch}/{file}", repoHandler.Handle)
	router.Head("/{repo}/{env}/{arch}/{file}", repoHandler.Handle)

	torrentHandler := NewTorrentHandler(d.cache)
	router.Get("/storage/{couple_id}/{repo}/{deb_file}", torrentHandler.Handler)
	router.Head("/storage/{couple_id}/{repo}/{deb_file}", torrentHandler.Handler)

	byHashHandler := NewByHashHandler(d.param.EnforceSessions, d.db, d.cache)
	router.Get("/{repo}/{env}/{arch}/by-hash/{hash}", byHashHandler.Handle)
	router.Head("/{repo}/{env}/{arch}/by-hash/{hash}", byHashHandler.Handle)
	router.Get("/{repo}/{env}/{arch}/by-hash/SHA256/{hash}", byHashHandler.Handle)
	router.Head("/{repo}/{env}/{arch}/by-hash/SHA256/{hash}", byHashHandler.Handle)

	apiHandler := NewAPIHandler(d.db)
	router.Mount("/api/v1", apiHandler.APIV1Router())

	if debug {
		router.Mount("/debug", middleware.Profiler())
	}

	router.Get("/unistat", UnistatHandler)

	address := fmt.Sprintf(":%d", d.param.Port)
	s := &http.Server{
		Addr:              address,
		Handler:           router,
		ReadTimeout:       2 * time.Minute,
		ReadHeaderTimeout: 2 * time.Minute,
		WriteTimeout:      2 * time.Minute,
		IdleTimeout:       5 * time.Minute,
	}
	return s.ListenAndServe()
}
