package logger

import (
	"fmt"
	"os"
	"strings"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

var logger *zap.SugaredLogger
var Requests *RequestsLogger
var atomicLevel zap.AtomicLevel
var levelsMap = map[string]zapcore.Level{
	"debug":    zapcore.DebugLevel,
	"info":     zapcore.InfoLevel,
	"warning":  zapcore.WarnLevel,
	"warn":     zapcore.WarnLevel,
	"error":    zapcore.ErrorLevel,
	"err":      zapcore.ErrorLevel,
	"critical": zapcore.FatalLevel,
	"crit":     zapcore.FatalLevel,
}

type RequestsLogger struct {
	logger *zap.SugaredLogger
}

func (r *RequestsLogger) Print(v ...interface{}) {
	r.logger.Info(v...)
}

func newRequestsLogger(logger *zap.SugaredLogger) *RequestsLogger {
	return &RequestsLogger{logger: logger}
}

func ConfigureLogging(c Config) error {
	level := strings.ToLower(c.Level)
	if _, ok := levelsMap[level]; !ok {
		return fmt.Errorf("level %s is unknown", c.Level)
	}
	atomicLevel = zap.NewAtomicLevelAt(levelsMap[level])
	l, err := newRotatingFileLogger(c.File, atomicLevel)
	if err != nil {
		return err
	}
	logger = l
	l, err = newRotatingFileLogger(c.RequestsLog, atomicLevel)
	if err != nil {
		return err
	}
	Requests = newRequestsLogger(l)
	return nil
}

func newRotatingFileLogger(path string, level zap.AtomicLevel) (*zap.SugaredLogger, error) {
	syncer, err := newRotatingFileSyncer(path)
	if err != nil {
		return nil, err
	}
	encoderCfg := zap.NewProductionEncoderConfig()
	encoderCfg.EncodeTime = zapcore.RFC3339TimeEncoder
	l := zap.New(zapcore.NewCore(zapcore.NewConsoleEncoder(encoderCfg), syncer, level))
	return l.Sugar(), nil
}

func Debug(v ...interface{}) {
	logger.Debug(v...)
}

func Debugf(format string, v ...interface{}) {
	logger.Debugf(format, v...)
}

func Info(v ...interface{}) {
	logger.Info(v...)
}

func Infof(format string, v ...interface{}) {
	logger.Infof(format, v...)
}

func Warning(v ...interface{}) {
	logger.Warn(v...)
}

func Warningf(format string, v ...interface{}) {
	logger.Warnf(format, v...)
}

func Error(v ...interface{}) {
	logger.Error(v...)
}

func Errorf(format string, v ...interface{}) {
	logger.Errorf(format, v...)
}

func Critical(v ...interface{}) {
	logger.Fatal(v...)
}

func Criticalf(format string, v ...interface{}) {
	logger.Fatalf(format, v...)
}

func Fatal(v ...interface{}) {
	logger.Fatal(v...)
	os.Exit(-1)
}

func Fatalf(format string, v ...interface{}) {
	logger.Fatalf(format, v...)
	os.Exit(-1)
}
