package logger

import (
	"log"
	"os"
	"os/signal"
	"sync"
	"syscall"
)

type rotatingFileSyncer struct {
	path string
	f    *os.File
	mtx  sync.Mutex
}

func newRotatingFileSyncer(path string) (*rotatingFileSyncer, error) {
	sigs := make(chan os.Signal, 1)
	syncer := rotatingFileSyncer{path: path}
	err := syncer.ensureFile()
	if err != nil {
		return nil, err
	}
	signal.Notify(sigs, syscall.SIGHUP)
	go syncer.sighupHandler(sigs)
	return &syncer, nil
}

func (r *rotatingFileSyncer) Write(p []byte) (n int, err error) {
	r.mtx.Lock()
	defer r.mtx.Unlock()
	return r.f.Write(p)
}

func (r *rotatingFileSyncer) Sync() error {
	r.mtx.Lock()
	defer r.mtx.Unlock()
	return r.f.Sync()
}

func (r *rotatingFileSyncer) openFile() error {
	f, err := os.OpenFile(r.path, os.O_APPEND|os.O_CREATE|os.O_RDWR, 0644)
	if err != nil {
		return err
	}
	r.f = f
	return nil
}

func (r *rotatingFileSyncer) reopenFile() error {
	err := r.f.Close()
	if err != nil {
		log.Printf("ERROR: cannot close file %s: %s", r.path, err)
		return r.openFile()
	}
	return r.openFile()
}

func (r *rotatingFileSyncer) ensureFile() error {
	r.mtx.Lock()
	defer r.mtx.Unlock()
	if r.f == nil {
		return r.openFile()
	}
	stat, err := r.f.Stat()
	if err != nil {
		log.Printf("ERROR: cannot stat opened file %s: %s", r.f.Name(), err)
		return r.reopenFile()
	}
	osStat, err := os.Stat(r.path)
	if err != nil {
		log.Printf("ERROR: cannot stat %s: %s", r.path, err)
		return r.reopenFile()
	}
	if !os.SameFile(stat, osStat) {
		return r.reopenFile()
	}
	return nil
}

func (r *rotatingFileSyncer) sighupHandler(c chan os.Signal) {
	for range c {
		log.Print("SIGHUP received reopening files")
		err := r.ensureFile()
		if err != nil {
			log.Printf("ERROR: Cannot reopen log '%s' on sighup: %s", r.path, err)
		}
	}
}
