package uautil

import (
	"net/http"
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestGetAPTVersion(t *testing.T) {
	r := http.Request{Header: map[string][]string{"User-Agent": {"mock"}}}
	rv, re := GetAPTVersion(&r)
	assert.Equal(t, uint64(0), rv)
	assert.Equal(t, errAPTVersionParse, re)
	r = http.Request{Header: map[string][]string{"User-Agent": {"Debian APT-HTTP/1.3 (0.0.0)"}}}
	rv, re = GetAPTVersion(&r)
	assert.Equal(t, uint64(0), rv)
	assert.Equal(t, nil, re)
	r = http.Request{Header: map[string][]string{"User-Agent": {"Debian APT-HTTP/1.3 (1.2.0)"}}}
	rv, re = GetAPTVersion(&r)
	assert.Equal(t, uint64(0x0001000200000000), rv)
	assert.Equal(t, nil, re)
	r = http.Request{Header: map[string][]string{"User-Agent": {"Debian APT-HTTP/1.3 "}}}
	rv, re = GetAPTVersion(&r)
	assert.Equal(t, uint64(0), rv)
	assert.Equal(t, errAPTVersionParse, re)
	r = http.Request{Header: map[string][]string{"User-Agent": {"Debian APT-HTTP/1.3 (a.b.c.d)"}}}
	rv, re = GetAPTVersion(&r)
	assert.Equal(t, uint64(0), rv)
	assert.Equal(t, nil, re)
	r = http.Request{Header: map[string][]string{"User-Agent": {"Mozilla 4.0"}}}
	rv, re = GetAPTVersion(&r)
	assert.Equal(t, uint64(0), rv)
	assert.Equal(t, errAPTVersionParse, re)
}

func TestParseAPTVersion(t *testing.T) {
	rv, re := parseAPTVersion("Debian APT-HTTP/1.3 (0.0.0)")
	assert.Equal(t, uint64(0), rv)
	assert.Equal(t, nil, re)
	rv, re = parseAPTVersion("Debian APT-HTTP/1.3 (1.2.13asfosk~fasfu)")
	assert.Equal(t, uint64(0x0001000200000000), rv)
	assert.Equal(t, nil, re)
	rv, re = parseAPTVersion("Debian APT-HTTP/1.3 (0.1.200)")
	assert.Equal(t, uint64(0x0000000100000000), rv)
	assert.Equal(t, nil, re)
	rv, re = parseAPTVersion("Debian APT-HTTP/1.3 (1.1.9)")
	assert.Equal(t, uint64(0x0001000100000000), rv)
	assert.Equal(t, nil, re)
	rv, re = parseAPTVersion("Debian APT-HTTP/1.3 (2.3.6)")
	assert.Equal(t, uint64(0x0002000300000000), rv)
	assert.Equal(t, nil, re)
}
