import typing
from collections import defaultdict
from os.path import join as join_path
from os import rename, listdir, makedirs, getenv
from time import sleep
from pycurl import (
    Curl,
    E_COULDNT_CONNECT,
    HTTP_VERSION,
    CURL_HTTP_VERSION_1_0,
    POST,
    POSTFIELDS,
    HTTPHEADER,
    RESPONSE_CODE,
    error as CurlError,
)
from io import BytesIO

import yt.wrapper as yt
from google.protobuf.json_format import MessageToJson
from infra.dostavlyator.lib.data.spec import (
    TResource,
    TResourceSet,
)
from infra.dostavlyator.proto import main_pb2
from infra.dostavlyator.proto.tables_pb2 import (
    TBoxRequestedTable,
    TBoxAppliedTable,
)
from infra.dostavlyator.lib.misc.misc import (
    GetLogger,
    get_instance_details,
    safe_symlink,
    read_spec_file,
)
from infra.dostavlyator.lib.db.tables import (
    LookupBoxRequestedTable,
    LookupBoxAssignedTable,
    LookupBoxAppliedTable,
    InsertBoxAppliedTable,
    GetPathBoxAppliedTable,
    InsertBoxRequestedTable,
)

log = GetLogger('dostavlyator.activator')

INSTANCE_FILTER = get_instance_details()
INSTANCE_FILTER = [
    {
        "DeployPodPersistentFqdn": INSTANCE_FILTER["DeployPodPersistentFqdn"],
        "DeployBoxId": INSTANCE_FILTER["DeployBoxId"],
    }
]
ITERATION_WAIT_TIME = 5


def get_box_requested(yt_client, yt_dir):
    box_requested_list = LookupBoxRequestedTable(yt_client, yt_dir, INSTANCE_FILTER)
    if not box_requested_list:
        raise Exception(f"Can't find TBoxAssignedTable record using instance_filter: \"{INSTANCE_FILTER}\"")
    return box_requested_list[0]


def get_box_assigned(yt_client, yt_dir):
    box_assigned_list = LookupBoxAssignedTable(yt_client, yt_dir, INSTANCE_FILTER)
    if not box_assigned_list:
        raise Exception(f"Can't find TBoxAssignedTable record using instance_filter: \"{INSTANCE_FILTER}\"")
    return box_assigned_list[0]


def lock_box_applied(yt_client, yt_dir):
    yt_client.lock_rows(GetPathBoxAppliedTable(yt_dir), INSTANCE_FILTER, lock_type="exclusive")


def get_box_applied(yt_client, yt_dir) -> TBoxAppliedTable:
    box_applied_list = LookupBoxAppliedTable(yt_client, yt_dir, INSTANCE_FILTER)
    if not box_applied_list:
        raise Exception(f"Can't find TBoxAppliedTable record using instance_filter: \"{INSTANCE_FILTER}\"")
    return box_applied_list[0]


class ResourceSet:
    def __init__(self, proto: TResourceSet, is_active: bool, is_ready: bool) -> None:
        self.proto: TResourceSet = proto
        self.is_active: bool = is_active
        self.is_ready: bool = is_ready

    def __getattr__(self, attr):
        return getattr(self.proto, attr)

    @property
    def status(self) -> main_pb2.EStatus:
        if self.ManualStatus != main_pb2.EStatus.NONE:
            return self.ManualStatus
        return self.ValidationStatus


class State:
    def __init__(self, box_assigned, box_applied) -> None:

        self._resources = dict([(resource.Id, resource) for resource in box_assigned.Spec.Resource])
        self.rss_ids = set()
        self.rs_by_rs_id = dict()
        self.rss_id_by_rs_id = dict()
        self.rs_by_rss_id = defaultdict(list)

        _resources_ready = {
            resource.ResourceId for resource in box_applied.Spec.Resource if resource.FileStatus in {main_pb2.EFileStatus.READY, main_pb2.EFileStatus.READY_VERIFY}
        }

        def check_is_ready(_rs_proto):
            return all([resource_id in _resources_ready for resource_id in _rs_proto.Resource])

        for rs_proto in box_assigned.Spec.ResourceSet:
            status_proto = next(filter(lambda x: x.ResourceSetId == rs_proto.Id, box_applied.Spec.ActiveResourceSet), None)
            if status_proto:
                if rs_proto.ValidationStatus == main_pb2.EStatus.TESTING:
                    rs_proto.ValidationStatus = status_proto.ValidationStatus
            rs = ResourceSet(proto=rs_proto, is_active=bool(status_proto), is_ready=check_is_ready(rs_proto))
            self.rs_by_rs_id[rs_proto.Id] = rs
            self.rs_by_rss_id[rs_proto.ResourceSetSpecId].append(rs)
            self.rss_id_by_rs_id[rs_proto.Id] = rs_proto.ResourceSetSpecId
            self.rss_ids.add(rs_proto.ResourceSetSpecId)

    def save(self, yt_client, yt_dir) -> TBoxAppliedTable:
        with log.debug('save()'):
            with yt_client.Transaction(type="tablet"):
                lock_box_applied(yt_client, yt_dir)

                box_applied = get_box_applied(yt_client, yt_dir)
                box_applied.Spec.ClearField('ActiveResourceSet')
                box_applied.Spec.ActiveResourceSet.MergeFrom(
                    [
                        main_pb2.TBoxAppliedResourceSetSpec(
                            ResourceSetId=rs.Id,
                            ActiveFiles=[self.get_resource(resource_id).ResourceCandidateId for resource_id in rs.Resource],
                            ValidationStatus=rs.status
                        )
                        for rs in self.rs_by_rs_id.values()
                        if rs.is_active
                    ]
                )
                InsertBoxAppliedTable(yt_client, yt_dir, box_applied)
                return box_applied

    def get_resource(self, resource_id) -> TResource:
        return self._resources[resource_id]

    def get_rss_ids(self) -> typing.List[str]:
        return self.rss_ids

    def get_latest_rs(self, rss_id, rs_status, ready_only=False) -> typing.Optional[ResourceSet]:
        return max(
            filter(lambda x: (not ready_only or x.is_ready) and x.status == rs_status, self.rs_by_rss_id[rss_id]),
            key=lambda x: x.Version,
            default=None,
        )

    def get_active_rs(self, rss_id) -> typing.Optional[ResourceSet]:
        return next(filter(lambda x: x.is_active, self.rs_by_rss_id[rss_id]), None)


class Activator:
    def __init__(
        self,
        *,
        yt_cluster: str,
        yt_dir: str,
        storage_dir: str,
        resource_set_dir: str,
        output_dir: str,
        spec_fname: str,
        state_fname: str,
        notify_url: str,
        once: bool,
        shadow_mode: bool,
        no_activate: bool
    ) -> None:
        self.yt_client = yt.YtClient(yt_cluster, token=getenv("YT_TOKEN"), config={"backend": "rpc"})
        self.yt_dir = yt_dir
        self.storage_dir = storage_dir
        self.resource_set_dir = resource_set_dir
        self.output_dir = output_dir
        self.spec_fname = spec_fname
        self.state_fname = state_fname
        self.notify_url = notify_url
        self._once = once
        self._shadow_mode = shadow_mode
        self._no_activate = no_activate
        self.is_initialized = False
        self.box_requested = None
        self.box_assigned = None
        self.box_applied = None
        self.state = None
        self.resource_map = dict()
        self.resource_set_map = dict()

    def sync_requested(self) -> None:
        instance_details = get_instance_details()
        requested_spec = read_spec_file(self.spec_fname)
        self.box_requested = TBoxRequestedTable(Spec=requested_spec, **instance_details)
        log.debug(f'Store to YT TBoxRequested(SpecId={self.box_requested.Spec.SpecId})')
        InsertBoxRequestedTable(self.yt_client, self.yt_dir, self.box_requested)

    def sync_state(self):
        log.debug('Read BoxAssigned, BoxApplied')
        if not self.box_requested:
            self.box_requested = get_box_requested(self.yt_client, self.yt_dir)
        self.box_assigned = get_box_assigned(self.yt_client, self.yt_dir)
        self.box_applied = get_box_applied(self.yt_client, self.yt_dir)
        return State(self.box_assigned, self.box_applied)

    def tick(self) -> bool:
        if not self._shadow_mode:
            self.sync_requested()
        self.state = self.sync_state()
        to_activate = self.select_rs_to_initialize() if not self.is_initialized else self.select_rs_to_activate()
        if not to_activate:
            log.debug('nothing to do')
            return True
        if not self._shadow_mode:
            for resource_set in to_activate:
                if not resource_set.is_ready:
                    log.debug(f'TResourceSet(Id={resource_set.Id}) not ready. Wait...')
                    return False
        if self._no_activate:
            self.save_state()
        else:
            self.activate(to_activate)
        return True

    def run(self):
        while True:
            try:
                if self.tick() and self._once:
                    return
            except KeyboardInterrupt:
                raise
            except SystemExit:
                raise
            except:
                log.exception("Exception during iteration:")
            log.debug('sleep(1)')
            sleep(1)

    def select_best_rs(self, rss_id) -> typing.Optional[ResourceSet]:
        with log.debug('select_best_rs()'):
            log.debug(f'Search ResourceSet for ResourceSetSpec({rss_id})')
            for rs_status in [main_pb2.EStatus.VALID, main_pb2.EStatus.TESTING]:
                for ready_only in [True, False]:
                    rs = self.state.get_latest_rs(rss_id=rss_id, rs_status=rs_status, ready_only=ready_only)
                    if rs:
                        return rs
            raise Exception(
                f"Failed to find suitable ResourceSet for ResourceSetSpec {rss_id} during initialization -"
                "this problem shouldn't happen, probably there is a bug in activator code"
            )

    def select_rs_to_initialize(self) -> list:
        to_activate = []
        with log.debug('select_rs_to_initialize()'):
            for rss_id in self.state.get_rss_ids():
                rs = self.select_best_rs(rss_id)
                log.debug(f'Select ResourceSet({rs.Id}) status={main_pb2.EStatus.Name(rs.status)} is_ready={rs.is_ready}')
                to_activate.append(rs)
        return to_activate

    def select_rs_to_repair(self) -> list:
        to_activate = []
        with log.debug('select_rs_to_repair()'):
            for rss_id in self.state.get_rss_ids():
                active_rs = self.state.get_active_rs(rss_id)
                if not active_rs or active_rs.status == main_pb2.EStatus.BAD:  # TODO TESTING
                    rs = self.select_best_rs(rss_id)
                    if rs:
                        if active_rs:
                            log.debug(f'Repair ResourceSet({active_rs.Id}) -> ResourceSet({rs.Id})')
                        else:
                            log.debug(f'Select ResourceSet({rs.Id}) for ResourceSetSpec({rss_id})')
                        to_activate.append(rs)
        return to_activate

    def select_rs_to_fast_activate(self) -> list:
        to_activate = []
        with log.debug('select_rs_to_fast_activate()'):
            for rss_id in self.state.get_rss_ids():
                active_rs = self.state.get_active_rs(rss_id)
                valid_rs = self.state.get_latest_rs(rss_id, rs_status=main_pb2.EStatus.VALID, ready_only=True)
                if valid_rs and (not active_rs or active_rs.Version != valid_rs.Version):
                    log.debug(f'Select ResourceSet({valid_rs.Id})')
                    to_activate.append(valid_rs)
        return to_activate

    def select_rs_to_validate(self) -> list:
        with log.debug('select_rs_to_validate()'):
            for rss_id in self.state.get_rss_ids():
                active_rs = self.state.get_active_rs(rss_id)
                latest_testing_rs = self.state.get_latest_rs(rss_id, rs_status=main_pb2.EStatus.TESTING, ready_only=True)
                if latest_testing_rs and active_rs.Version < latest_testing_rs.Version:
                    return [latest_testing_rs]
        return []

    def select_rs_to_activate(self) -> list:
        with log.debug('select_rs_to_activate()'):
            # 1. check for BAD ResourceSet't (repair)
            to_activate = self.select_rs_to_repair()
            if to_activate:
                return to_activate

            # 2. fast switch to VALID ResourceSet't
            to_activate = self.select_rs_to_fast_activate()
            if to_activate:
                return to_activate

            # 3. activate TESTING ResourceSet (validation)
            to_activate = self.select_rs_to_validate()
            if to_activate:
                return to_activate

        # 4. nothing to activate
        return []

    def make_resource_set_dir(self, resource_set) -> str:
        return join_path(self.resource_set_dir, resource_set.Id)

    def build_resource_set_dir(self, resource_set) -> None:
        with log.debug(f'Build ResourceSet({resource_set.Id}) -> {resource_set.Path}'):
            resource_set_dir = self.make_resource_set_dir(resource_set)
            self.resource_set_map[resource_set.Path] = resource_set_dir
            makedirs(resource_set_dir, mode=0o755, exist_ok=True)
            for resource_id in resource_set.Resource:
                resource = self.state.get_resource(resource_id)
                src_dir = join_path(self.storage_dir, resource.ResourceCandidateId)
                if resource.Path == '*':
                    for f in listdir(src_dir):
                        src = join_path(src_dir, f)
                        dst = join_path(resource_set_dir, f)
                        safe_symlink(src, dst)
                        self.resource_map[join_path(resource_set.Path, f)] = src
                else:
                    dst_dir = join_path(resource_set_dir, resource.Path)
                    safe_symlink(src_dir, dst_dir)
                    self.resource_map[join_path(resource_set.Path, resource.Path)] = src_dir

    def create_symlinks(self, to_activate) -> None:
        with log.debug('create_symlinks()'):
            for resource_set in to_activate:
                self.build_resource_set_dir(resource_set)
            makedirs(self.output_dir, mode=0o755, exist_ok=True)
            for resource_set in to_activate:
                resource_set_dir = self.make_resource_set_dir(resource_set)
                resource_set_output_dir = join_path(self.output_dir, resource_set.Path)
                safe_symlink(resource_set_dir, resource_set_output_dir)
            # TODO: remove unnecessary

    def notify_process(self, to_activate, data) -> bool:
        class RetryException(Exception):
            pass

        if not self.notify_url:
            return True
        # TODO: set timeout
        try:
            c = Curl()
            c.setopt(c.URL, self.notify_url)
            c.setopt(HTTP_VERSION, CURL_HTTP_VERSION_1_0)
            c.setopt(POST, 1)
            c.setopt(HTTPHEADER, ['Expect:'])
            c.setopt(POSTFIELDS, data)
            buffer = BytesIO()
            c.setopt(c.WRITEDATA, buffer)
            log.info(f'Notify service to reload. POST "{self.notify_url}"')
            c.perform()
            response_code = c.getinfo(RESPONSE_CODE)
            # response = buffer.getvalue().decode('UTF-8')
            c.close()
            if response_code in (505, 521):
                raise RetryException(f'Got HTTP {response_code}, retry.')
            log.debug(f'Got HTTP {response_code}')
            if response_code in (200, 204):
                log.debug('Mark as VALID')
                for resource_set in to_activate:
                    resource_set.ValidationStatus = main_pb2.EStatus.VALID
                return True
        except CurlError as e:
            errcode = e.args[0]
            if errcode in {E_COULDNT_CONNECT, }:
                raise
            log.error(e)
        except RetryException:
            raise
        except Exception as e:
            log.error(e)
        for resource_set in to_activate:
            if resource_set.status == main_pb2.EStatus.TESTING:
                resource_set.ValidationStatus = main_pb2.EStatus.BAD
            else:
                raise Exception('Mixed TResource status')
            log.debug('Mark as BAD')
        return False

    def check_healt(self) -> None:
        log.error('TODO: check health')

    def dump_state(self) -> main_pb2.TBoxState:
        return main_pb2.TBoxState(
            BoxRequested=self.box_requested.Spec, BoxAssigned=self.box_assigned.Spec, BoxApplied=self.box_applied.Spec
        )

    def dump_info(self) -> main_pb2.TActivatorInfo:
        return main_pb2.TActivatorInfo(
            StorageDir=self.storage_dir,
            ResourceSetDir=self.resource_set_dir,
            OutputDir=self.output_dir,
            NotifyUrl=self.notify_url,
            StateFile=self.state_fname,
        )

    def dump_box_info(self) -> main_pb2.TBoxInfo:
        return main_pb2.TBoxInfo(
            State=self.dump_state(),
            ActivatorInfo=self.dump_info(),
            ResourceMap=self.resource_map,
            ResourceSetMap=self.resource_set_map,
        )

    def save_state(self) -> None:
        log.debug(f'save_sate(SpecId={self.box_assigned.Spec.SpecId})')
        if not self._shadow_mode:
            self.box_applied = self.state.save(self.yt_client, self.yt_dir)
        if self.state_fname is None:
            return
        box_info = self.dump_box_info()
        json_info = MessageToJson(box_info, including_default_value_fields=False, preserving_proto_field_name=True)
        fname_tmp = self.state_fname + '_tmp'
        with open(fname_tmp, 'w+') as f:
            f.write(json_info)
            f.close()
        rename(fname_tmp, self.state_fname)
        return json_info

    def switch_active(self, to_activate):
        with log.debug('switch_active()'):
            for rs in to_activate:
                active_rs = self.state.get_active_rs(self.state.rss_id_by_rs_id[rs.Id])
                if rs != active_rs:
                    with log.debug(f'activate ResourceSet({rs.Id})'):
                        if active_rs:
                            log.debug(f'deactivate ResourceSet({active_rs.Id})')
                            active_rs.is_active = False
                        rs.is_active = True
            # TODO: return diff

    def activate(self, to_activate):
        with log.debug('activate()'):
            try:
                self.switch_active(to_activate)
                self.create_symlinks(to_activate)
                json_info = self.save_state()
                self.notify_process(to_activate, data=json_info)
                self.check_healt()
                self.save_state()
                self.is_initialized = True
            except:
                log.exception("exception during activation:")
                if not self.is_initialized:
                    raise
