from datetime import datetime
from prompt_toolkit.completion import (
    Completer as BaseCompleter,
    WordCompleter,
    Completion,
)
from infra.dostavlyator.lib.misc.misc import GetLogger
from infra.dostavlyator.proto.main_pb2 import EStatus

log = GetLogger("infra.dostavlyator.lib.cli.completion")


def get_TResourceSetSpec_by_id(db, selector):
    return {db.resource_set_spec[selector]}


def get_TResourceSetSpec_by_path(db, selector):
    return set(
        resource_set_spec for resource_set_spec in db.resource_set_spec.values()
        if resource_set_spec.Path == selector
    )


def get_TResourceSetSpec(db, selector_type, selector):
    if selector_type == 'by-id':
        return get_TResourceSetSpec_by_id(db, selector)
    elif selector_type == 'by-path':
        return get_TResourceSetSpec_by_path(db, selector)
    raise Exception(f"Invalid TResourceSetSpec selector type '{selector_type}'")


class Completer(BaseCompleter):
    def __init__(self, db):
        self.db = db
        self.origin_document = None
        self.todo_completer = WordCompleter(['TODO']).get_completions
        self.TResource_by_id_completer = WordCompleter(
            sorted(set(resource_id for resource_id in self.db.resource.keys()))
        ).get_completions
        self.TResourceSet_by_path_completer = WordCompleter(
            sorted(set(resource_set.Path for resource_set in self.db.resource_set.values()))
        ).get_completions
        self.TResourceSetSpec_by_path_completer = WordCompleter(
            sorted(set(resource_set_spec.Path for resource_set_spec in self.db.resource_set_spec.values()))
        ).get_completions

    def make_completions(self, completer, variants, document, complete_event, level=0, **args):
        tokens = document.text.split(' ')
        if len(tokens) - 1 == level:
            if not completer:
                completer = WordCompleter(variants.keys()).get_completions
            yield from completer(document, complete_event)
            return None
        cmd = tokens[level]
        if cmd in variants or None in variants:
            next_step = variants[cmd] if cmd in variants else variants[None]
            if next_step:
                yield from next_step(document, complete_event, level + 1, **args)
        return None

    def get_completions(self, document, complete_event):
        variants = {
            "exit": None,
            "reload": None,
            "status": None,
            "freeze": None,
            "unfreeze": None,
            "unwipe": None,
            "TResource": self.get_completions_TResource,
            "TResourceSet": self.get_completions_TResourceSet,
            "TResourceSpec": self.get_completions_TResourceSpec,
            "TResourceSetSpec": self.get_completions_TResourceSetSpec,
            "TBoxRequested": self.todo_completer,
            "TBoxAssigned": self.todo_completer,
            "TBoxApplied": self.get_completions_TBoxApplied,
        }
        yield from self.make_completions(None, variants, document, complete_event)

    def get_completions_TResource(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "by-id": self.get_completions_TResource_by_id,
        }
        yield from self.make_completions(None, variants, document, complete_event, level, **args)

    def get_completions_TResource_by_id(self, document, complete_event, level, **args):
        variants = {None: WordCompleter(["pin", "unpin"]).get_completions}
        yield from self.make_completions(
            self.TResource_by_id_completer, variants, document, complete_event, level, **args
        )

    def get_completions_TResourceSet(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "by-id": self.get_completions_TResourceSet_by_id,
            "by-path": self.get_completions_TResourceSet_by_path,
        }
        yield from self.make_completions(None, variants, document, complete_event, level, **args)

    def get_completions_TResourceSet_by_path(self, document, complete_event, level, **args):
        yield from self.TResourceSet_by_path_completer(document, complete_event, level, **args)

    def get_completions_TResourceSet_by_id(
        self, document, complete_event, level, resource_set_spec_id=None, resource_set_spec_path=None, **args
    ):
        def get_completions(document, complete_event):
            resource_set_spec = None
            if resource_set_spec_id:
                resource_set_spec = get_TResourceSetSpec_by_id(self.db, selector=resource_set_spec_id)
            elif resource_set_spec_path:
                resource_set_spec = get_TResourceSetSpec_by_path(self.db, selector=resource_set_spec_path)

            if resource_set_spec:
                resource_set = list()
                for rss in resource_set_spec:
                    resource_set.extend(rss.resource_set)
            else:
                resource_set = self.db.resource_set.values()
            for rs in sorted(resource_set, key=lambda x: x.CreationTime, reverse=True)[:40]:
                ctime = datetime.utcfromtimestamp(rs.CreationTime)
                ctime_txt = ctime.strftime('%Y-%m-%d %H:%M:%S UTC')
                yield Completion(
                    text=rs.Id,
                    display=f'{rs.Id} CreationTime={ctime_txt} Status={EStatus.Name(rs.GetStatus())} Version={rs.Version}',
                )

        variants = {None: self.get_completions_TResourceSet_menu}
        yield from self.make_completions(
            get_completions,
            variants,
            document,
            complete_event,
            level,
            resource_set_spec_id=resource_set_spec_id,
            resource_set_spec_path=resource_set_spec_path,
            **args,
        )

        return None

    def get_completions_TResourceSpec(self, document, complete_event, level, **args):
        pass

    def get_completions_TResourceSetSpec(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "by-id": self.get_completions_TResourceSetSpec_by_id,
            "by-path": self.get_completions_TResourceSetSpec_by_path,
        }
        yield from self.make_completions(None, variants, document, complete_event, level, **args)

    def get_completions_TResourceSetSpec_by_id(self, document, complete_event, level, **args):
        variants = {None: self.get_completions_TResourceSetSpec_by_id_menu}

        def get_completions(document, complete_event):
            for resource_set_spec in sorted(self.db.resource_set_spec.values(), key=lambda rss: rss.Id):
                yield Completion(
                    text=resource_set_spec.Id, display=f'Name={resource_set_spec.Name} Path={resource_set_spec.Path}'
                )

        yield from self.make_completions(get_completions, variants, document, complete_event, level, **args)

    def get_completions_TResourceSetSpec_by_path(self, document, complete_event, level, **args):
        variants = {None: self.get_completions_TResourceSetSpec_by_path_menu}
        yield from self.make_completions(
            self.TResourceSetSpec_by_path_completer, variants, document, complete_event, level, **args
        )

    def get_completions_TResourceSetSpec_by_id_menu(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "pin": self.get_completions_TResourceSetSpec_pin,
            "pin-last": None,
            "pin-earlier": self.get_completions_TResourceSetSpec_pin_earlier,
            "unpin": None,
            "TResourceSet": self.get_completions_TResourceSetSpec_TResourceSet,
        }
        tokens = document.text.split(' ')
        resource_set_spec_id = tokens[level - 1]
        yield from self.make_completions(
            None, variants, document, complete_event, level, resource_set_spec_id=resource_set_spec_id, **args
        )

    def get_completions_TResourceSetSpec_by_path_menu(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "pin-earlier": self.get_completions_TResourceSetSpec_pin_earlier,
            "pin-last": None,
            "unpin": None,
            "TResourceSet": self.get_completions_TResourceSetSpec_TResourceSet,
        }
        tokens = document.text.split(' ')
        resource_set_spec_path = tokens[level - 1]
        yield from self.make_completions(
            None, variants, document, complete_event, level, resource_set_spec_path=resource_set_spec_path, **args
        )

    def get_completions_TResourceSetSpec_TResourceSet(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "by-id": self.get_completions_TResourceSet_by_id,
        }
        yield from self.make_completions(None, variants, document, complete_event, level, **args)

    def get_completions_TResourceSetSpec_pin_earlier(
        self, document, complete_event, level, resource_set_spec_id=None, resource_set_spec_path=None, **args
    ):
        resource_set_spec = None
        if resource_set_spec_id:
            resource_set_spec = get_TResourceSetSpec_by_id(self.db, selector=resource_set_spec_id)
        elif resource_set_spec_path:
            resource_set_spec = get_TResourceSetSpec_by_path(self.db, selector=resource_set_spec_path)
        resource_set = list()
        for rss in resource_set_spec:
            resource_set.extend(rss.resource_set)
        for rs in sorted(resource_set, key=lambda x: x.CreationTime, reverse=True):
            ctime = datetime.utcfromtimestamp(rs.CreationTime)
            yield Completion(
                text=ctime.strftime('%Y-%m-%dT%H:%M:%SZ'),
                display=ctime.strftime(
                    f'%Y-%m-%d %H:%M:%S UTC Status={EStatus.Name(rs.GetStatus())} Version={rs.Version}'
                ),
            )
        return None

    def get_completions_TResourceSetSpec_pin(
        self, document, complete_event, level, resource_set_spec_id=None, resource_set_spec_path=None, **args
    ):
        resource_set_spec = None
        if resource_set_spec_id:
            resource_set_spec = get_TResourceSetSpec_by_id(self.db, selector=resource_set_spec_id)
        elif resource_set_spec_path:
            resource_set_spec = get_TResourceSetSpec_by_path(self.db, selector=resource_set_spec_path)
        resource_set = list()
        for rss in resource_set_spec:
            resource_set.extend(rss.resource_set)
        for rs in sorted(resource_set, key=lambda x: x.CreationTime, reverse=True):
            ctime = datetime.utcfromtimestamp(rs.CreationTime)
            ctime_txt = ctime.strftime('%Y-%m-%d %H:%M:%S UTC')
            yield Completion(
                text=rs.Id,
                display=f'{rs.Id} CreationTime={ctime_txt} Status={EStatus.Name(rs.GetStatus())} Version={rs.Version}',
            )

        return None

    def get_completions_TResourceSet_menu(self, document, complete_event, level, **args):
        yield from WordCompleter(['validate']).get_completions(document, complete_event)
        return None

    def get_completions_TBoxApplied(self, document, complete_event, level, **args):
        variants = {
            "list": None,
            "validation-problems": self.todo_completer,
            "download-problems": self.todo_completer,
            "by-id": self.todo_completer,
        }
        yield from self.make_completions(None, variants, document, complete_event, level, **args)
