import inspect
from copy import deepcopy
from datetime import datetime
from dateutil.parser import isoparse

from infra.dostavlyator.proto.main_pb2 import EStatus

from infra.dostavlyator.lib.misc import misc
from infra.dostavlyator.lib.db.tables import ReadControlFlagsNode
from infra.dostavlyator.lib.db import control
from infra.dostavlyator.lib.cli.completion import get_TResourceSetSpec_by_path

log = misc.GetLogger("infra.dostavlyator.lib.cli.executor")


class Executor:
    def __init__(self, db):
        self.db = db

    def step(self, tokens, args, variants):
        log.debug(f"{inspect.stack()[1][3]}: {tokens}, {args} {[variants.keys()]}")
        if not tokens:
            raise Exception("token expected")
        cmd = tokens.pop(0)
        if cmd in variants:
            func, nargs = variants[cmd]
            if nargs < 0:
                args = args[:nargs]
            elif nargs > 0:
                args.extend(tokens[0:nargs])
                tokens = tokens[nargs:]
            return func(tokens, args)
        raise Exception(f"unknown cmd='{cmd}'")

    def execute(self, tokens):
        variants = {
            "status": (self.action_status, 0),
            "reload": (self.action_reload, 0),
            "freeze": (self.action_freeze, 0),
            "unfreeze": (self.action_unfreeze, 0),
            "fluggaenkdechioebolsen": (self.action_wipe_db, 0),
            "unwipe": (self.action_unwipe_db, 0),
            "TResourceSetSpec": (self.branch_TResourceSetSpec, 0),
            "TResourceSet": (self.branch_TResourceSet, 0),
            "TBoxApplied": (self.branch_TBoxApplied, 0),
        }
        return self.step(tokens, [], variants)

    def action_status(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 0:
            raise Exception(f"required 0 args, got {args}'")
        flags = ReadControlFlagsNode(self.db._yt_client, self.db._base_path)
        print(flags)

    def action_reload(self, tokens, args):
        self.db.ReadTables()

    def action_freeze(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 0:
            raise Exception(f"required 0 args, got {args}'")
        control.freeze(self.db._yt_client, self.db._base_path)

    def action_unfreeze(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 0:
            raise Exception(f"required 0 args, got {args}'")
        control.unfreeze(self.db._yt_client, self.db._base_path)

    def action_wipe_db(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 0:
            raise Exception(f"required 0 args, got {args}'")
        control.wipe_db(self.db._yt_client, self.db._base_path)

    def action_unwipe_db(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 0:
            raise Exception(f"required 0 args, got {args}'")
        control.unwipe_db(self.db._yt_client, self.db._base_path)

    def branch_TResourceSetSpec(self, tokens, args):
        variants = {
            "by-id": (self.branch_TResourceSetSpec_by_id, 1),
            "by-path": (self.branch_TResourceSetSpec_by_path, 1),
        }
        return self.step(tokens, args, variants)

    def branch_TResourceSet(self, tokens, args):
        variants = {
            "by-id": (self.branch_TResourceSet_by_id, 1),
        }
        return self.step(tokens, args, variants)

    def branch_TResourceSetSpec_by_id(self, tokens, args):
        variants = {
            "list": (self.action_TResourceSetSpec_by_id_list, 0),
            "pin": (self.action_TResourceSetSpec_by_id_pin, 1),
            "pin-earlier": (self.action_TResourceSetSpec_by_id_pin_earlier, 1),
            "unpin": (self.action_TResourceSetSpec_by_id_unpin, 0),
            "TResourceSet": (self.branch_TResourceSet, -1),
        }
        return self.step(tokens, args, variants)

    def branch_TResourceSetSpec_by_path(self, tokens, args):
        variants = {
            "list": (self.action_TResourceSetSpec_by_path_list, 0),
            "pin-earlier": (self.action_TResourceSetSpec_by_path_pin_earlier, 1),
            "unpin": (self.action_TResourceSetSpec_by_path_unpin, 0),
            "TResourceSet": (self.branch_TResourceSet, -1),
        }
        return self.step(tokens, args, variants)

    def branch_TResourceSet_by_id(self, tokens, args):
        variants = {
            "validate": (self.action_TResourceSet_by_id_validate, 0),
        }
        return self.step(tokens, args, variants)

    def action_TResourceSetSpec_by_path_list(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 1:
            raise Exception(f"required 1 args, got {args}'")
        path = args[0]
        resource_set_spec = get_TResourceSetSpec_by_path(self.db, path)
        rows = []
        for rss in resource_set_spec:
            rows.append((rss.Id,))
            for rs in sorted(rss.resource_set, key=lambda x: x.Version, reverse=True)[:40]:
                ctime = datetime.utcfromtimestamp(rs.CreationTime)
                rows.append((rs.Id, rs.Version, ctime, EStatus.Name(rs.GetStatus())))
        fields = ["Id", "Version", "CreationTime", "Status"]
        return fields, rows

    def action_TResourceSetSpec_by_id_list(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 1:
            raise Exception(f"required 1 args, got {args}'")
        resource_set_spec_id = args[0]
        rss = self.db.resource_set_spec[resource_set_spec_id]
        rows = []
        rows.append((rss.Id,))
        for rs in sorted(rss.resource_set, key=lambda x: x.Version, reverse=True)[:40]:
            ctime = datetime.utcfromtimestamp(rs.CreationTime)
            rows.append((rs.Id, rs.Version, ctime, EStatus.Name(rs.GetStatus())))
        fields = ["Id", "Version", "CreationTime", "Status"]
        return fields, rows

    def action_TResourceSetSpec_by_id_pin(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 2:
            raise Exception(f"required 2 args, got {args}'")
        resource_set_spec_id, resource_set_id = args
        rs = self.db.resource_set[resource_set_id]
        log.info(
            f"pin TResourceSet(Id={rs.Id}, Version={rs.Version}, CreationTime={rs.CreationTime}) for TResourceSetSpec(Id={resource_set_spec_id})"
        )
        self.db.UpdateControlPinResourceSetTable(resource_set_spec_id, resource_set_id)

    def action_TResourceSetSpec_by_path_pin_earlier(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 2:
            raise Exception(f"required 2 args, got {args}'")
        path, ctime = args
        resource_set_spec = get_TResourceSetSpec_by_path(self.db, path)
        for rss in resource_set_spec:
            self.action_TResourceSetSpec_by_id_pin_earlier([], [rss.Id, ctime])

    def action_TResourceSetSpec_by_id_pin_earlier(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 2:
            raise Exception(f"required 2 args, got {args}'")
        resource_set_spec_id, ctime = args
        barrier_ctime = int(isoparse(ctime).timestamp())
        resource_set_spec = self.db.resource_set_spec[resource_set_spec_id]
        for rs in sorted(resource_set_spec.resource_set, key=lambda x: x.CreationTime, reverse=True):
            log.debug(
                f"look TResourceSet(Id={rs.Id}, Version={rs.Version}, CreationTime={rs.CreationTime}) for TResourceSetSpec(Id={resource_set_spec_id})"
            )
            if rs.CreationTime < barrier_ctime:
                log.info(
                    f"pin TResourceSet(Id={rs.Id}, Version={rs.Version}, CreationTime={rs.CreationTime}) for TResourceSetSpec(Id={resource_set_spec_id})"
                )
                self.db.UpdateControlPinResourceSetTable(resource_set_spec_id, rs.Id)
                break

    def action_TResourceSetSpec_by_path_unpin(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 1:
            raise Exception(f"required 1 args, got {args}'")
        path = args[0]
        resource_set_spec = get_TResourceSetSpec_by_path(self.db, path)
        for rss in resource_set_spec:
            self.action_TResourceSetSpec_by_id_unpin(
                [],
                [
                    rss.Id,
                ],
            )

    def action_TResourceSetSpec_by_id_unpin(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 1:
            raise Exception(f"required 1 args, got {args}'")
        resource_set_spec_id = args[0]
        log.info(f"unpin TResourceSetSpec(Id={resource_set_spec_id})")
        self.db.DeleteControlPinResourceSetTable(resource_set_spec_id)

    def action_TResourceSet_by_id_validate(self, tokens, args):
        if tokens:
            raise Exception(f"got extra tokens '{tokens}'")
        if len(args) != 1:
            raise Exception(f"required 1 args, got {args}'")
        resource_set_id = args[0]
        resource_set_proto = deepcopy(self.db.resource_set[resource_set_id]._proto)
        resource_set_proto.ValidationStatus = EStatus.TESTING
        self.db.UpdateResourceSet(resource_set_proto)
        return True

    def branch_TBoxApplied(self, tokens, args):
        variants = {
            # "list": (self.action_, 0),
            # "validation-problems": (self.action_, 1),
            # "download-problems": (self.action_, 0),
        }
        return self.step(tokens, args, variants)
