from time import time
from copy import deepcopy
from google.protobuf.json_format import MessageToJson
import infra.dostavlyator.lib.debug.debug as debug
from infra.dostavlyator.lib.misc import misc
from infra.dostavlyator.lib.db.tables import (
    ReadControlFlagsNode,
    ReadResourceTable,
    ReadBoxAppliedTable,
    ReadBoxAssignedTable,
    ReadBoxRequestedTable,
    ReadControlPinResourceSetTable,
    ReadResourceCandidateTable,
    ReadResourceSetSpecTable,
    ReadResourceSetTable,
    ReadResourceSpecTable,
    InsertBoxAppliedTable,
    InsertBoxAssignedTable,
    InsertBoxRequestedTable,
    InsertControlPinResourceSetTable,
    InsertResourceCandidateTable,
    InsertResourceSetSpecTable,
    InsertResourceSetTable,
    InsertResourceSpecTable,
    InsertResourceTable,
    DeleteBoxAppliedTable,
    DeleteBoxAssignedTable,
    DeleteBoxRequestedTable,
    DeleteControlPinResourceSetTable,
)
from infra.dostavlyator.proto import main_pb2
from infra.dostavlyator.proto import flags_pb2

log = misc.GetLogger('dostavlyator.db')


def ValueOrThrow(value, msg):
    if not value:
        raise Exception(msg)
    return value


class TProtoBased:
    def __init__(self):
        self.update_mtime()

    @property
    def ctime(self):
        return self.CreationTime

    @property
    def mtime(self):
        return self._mtime

    def update_mtime(self):
        self._mtime = time()

    def __getattr__(self, attr):
        return getattr(self._proto, attr)

    def ToJSON(self):
        return MessageToJson(self._proto, including_default_value_fields=False, preserving_proto_field_name=True)


class TResourceSpec(TProtoBased):
    def __init__(self, proto):
        TProtoBased.__init__(self)
        self._proto = proto
        self.resource = list()
        self.resource_set_spec = set()

    def __hash__(self):
        return hash(self.Id)

    def is_used(self):
        return any(resource_set_spec.is_used() for resource_set_spec in self.resource_set_spec)

    def GetLast(self):
        return self.resource[-1] if self.resource else None

    def GetLastOrThrow(self):
        return ValueOrThrow(self.GetLast(), Exception(f'No TResoruce for TResourceSpec(Id={self.Id})'))

    def GetLastGood(self):
        for r in reversed(self.resource):
            if not r.IsBad():
                return r
        return None

    def GetLastGoodOrThrow(self):
        return ValueOrThrow(self.GetLastGood(), Exception(f'No GOOD TResource for TResourceSpec(Id={self.Id})'))

    def GetLastValid(self):
        for r in reversed(self.resource):
            if r.IsValid():
                return r
        return None

    def GetLastValidOrThow(self):
        return ValueOrThrow(self.GetLastValid(), Exception(f'No VALID TResource for TResourceSpec(Id={self.Id})'))

    def GetLastTesting(self):
        for r in reversed(self.resource):
            if r.IsTesting():
                return r
        return None

    def GetLastTestingOrThow(self):
        return ValueOrThrow(self.GetLastTesting(), Exception(f'No TESTING TResource for TResourceSpec(Id={self.Id})'))

    def GetLastVersion(self):
        return self.resource[-1].Version if len(self.resource) else 0

    @property
    def Name(self):
        return self._proto.Name if self._proto.Name else self._proto.Id


class TResourceSetSpec(TProtoBased):
    def __init__(self, proto, resource_spec: set):
        TProtoBased.__init__(self)
        self._proto = proto
        self.resource_spec = resource_spec
        self.resource_set = list()
        for rs in resource_spec:
            rs.resource_set_spec.add(self)
        self.box_requested = set()

    def __hash__(self):
        return hash(self.Id)

    def is_used(self):
        return bool(self.box_requested)

    def GetLast(self):
        return self.resource_set[-1] if self.resource_set else None

    def GetLastOrThrow(self):
        return ValueOrThrow(self.GetLast(), Exception(f'No TResoruceSet for TResourceSetSpec(Id={self.Id})'))

    def GetLastValid(self):
        for rs in reversed(self.resource_set):
            if rs.IsValid():
                return rs
        return None

    def GetLastValidOrThow(self):
        return ValueOrThrow(self.GetLastValid(), Exception(f'No VALID TResoruceSet for TResourceSetSpec(Id={self.Id})'))

    def GetLastTesting(self):
        for rs in reversed(self.resource_set):
            if rs.IsTesting():
                return rs
        return None

    def GetLastTestingOrThow(self):
        return ValueOrThrow(
            self.GetLastTesting(), Exception(f'No TESTING TResoruceSet for TResourceSetSpec(Id={self.Id})')
        )

    def GetLastVersion(self):
        return self.resource_set[-1].Version if self.resource_set else 0

    def GetRequiredValidationApprovesCount(self):
        min_approves = self.ValidateMin
        required_approves = int(self.ValidateMax * len(self.box_requested) if self.ValidateMax < 1.0 else self.ValidateMax)
        return max(min_approves, required_approves)

    @property
    def MaxDeployDurationSec(self):
        return sum(map(lambda x: x.MaxDeployDurationSec, self.resource_spec))

    @property
    def Name(self):
        return self._proto.Name if self._proto.Name else self._proto.Id


class TResourceCandidate(TProtoBased):
    def __init__(self, proto):
        TProtoBased.__init__(self)
        self._proto = proto

    def __hash__(self):
        return hash(self.Id)

    def GetStatus(self):
        return self.ManualStatus if self.ManualStatus else self.SourceStatus


class TResource(TProtoBased):
    def __init__(self, proto, resource_spec: TResourceSpec, resource_candidate: TResourceCandidate):
        TProtoBased.__init__(self)
        self._proto = proto
        self.resource_spec = resource_spec
        self.resource_candidate = resource_candidate
        self.resource_set = set()

    def __hash__(self):
        return hash(self.Id)

    def is_used(self):
        return self.resource_spec.is_used()

    def GetStatus(self):
        if self.ManualStatus:
            return self.ManualStatus
        candidate_status = self.resource_candidate.GetStatus()
        if candidate_status == main_pb2.EStatus.BAD:
            return candidate_status
        return self.ValidationStatus

    def IsBad(self):
        return self.GetStatus() == main_pb2.EStatus.BAD

    def IsValid(self):
        return self.GetStatus() == main_pb2.EStatus.VALID

    def IsTesting(self):
        return self.GetStatus() == main_pb2.EStatus.TESTING


class TResourceSet(TProtoBased):
    def __init__(self, proto, resource_set_spec: TResourceSetSpec, resource: set):
        TProtoBased.__init__(self)
        self._proto = proto
        self.resource_set_spec = resource_set_spec
        self.resource = resource
        for r in resource:
            r.resource_set.add(self)

    def __hash__(self):
        return hash(self.Id)

    def is_used(self):
        return self.resource_set_spec.is_used()

    def GetStatus(self):
        if self.ManualStatus:
            return self.ManualStatus
        for resource in self.resource:
            resource_status = resource.GetStatus()
            if resource_status == main_pb2.EStatus.BAD:
                return resource_status
        return self.ValidationStatus

    def IsBad(self):
        return self.GetStatus() == main_pb2.EStatus.BAD

    def IsValid(self):
        return self.GetStatus() == main_pb2.EStatus.VALID

    def IsTesting(self):
        return self.GetStatus() == main_pb2.EStatus.TESTING


class TBoxRequested(TProtoBased):
    def __init__(self, proto):
        TProtoBased.__init__(self)
        self._proto = proto
        self.resource_set_spec = set()

    def __hash__(self):
        return hash(self.DeployPodPersistentFqdn)


class TBoxAssigned(TProtoBased):
    def __init__(self, proto):
        TProtoBased.__init__(self)
        self._proto = proto

    def __hash__(self):
        return hash(self.DeployPodPersistentFqdn)


class TBoxApplied(TProtoBased):
    def __init__(self, proto):
        TProtoBased.__init__(self)
        self._proto = proto
        self.ready_resources = set()

    def __hash__(self):
        return hash(self.DeployPodPersistentFqdn)


class TUpdatedObjects:
    def __init__(self):
        self.resource_spec = set()
        self.resource_set_spec = set()
        self.resource_candidate = set()
        self.resource = set()
        self.resource_set = set()
        self.box_requested = set()
        self.box_assigned = set()
        self.box_applied = set()

    def merge(self, other):
        for container in [
            'resource_spec',
            'resource_set_spec',
            'resource_candidate',
            'resource',
            'resource_set',
            'box_requested',
            'box_assigned',
            'box_applied',
        ]:
            getattr(self, container).update(getattr(other, container))
        return self

    def __iadd__(self, other):
        return self.merge(other)

    def __str__(self):
        return self.dump()

    def dump(self):
        return '''{{
            resource_candidate: {resource_candidate}
            resource: {resource}
            resource_set: {resource_set}
            resource_spec: {resource_spec}
            resource_set_spec: {resource_set_spec}
            box_requested: {box_requested}
            box_assigned: {box_assigned}
            box_applied: {box_applied}
        }}'''.format(
            resource_candidate = [resource_candidate.Id for resource_candidate in self.resource_candidate],
            resource = [resource.Id for resource in self.resource],
            resource_set = [resource_set.Id for resource_set in self.resource_set],
            resource_spec = [resource_spec.Id for resource_spec in self.resource_spec],
            resource_set_spec = [resource_set_spec.Id for resource_set_spec in self.resource_set_spec],
            box_requested = [box_requested.DeployPodPersistentFqdn for box_requested in self.box_requested],
            box_assigned = [box_assigned.DeployPodPersistentFqdn for box_assigned in self.box_assigned],
            box_applied = [box_applied.DeployPodPersistentFqdn for box_applied in self.box_applied]
        )


class DB:
    def __init__(self, yt_client, base_path):
        self._yt_client = yt_client
        self._base_path = base_path
        self._retention_timestamp = None
        self.control_flags = flags_pb2.TControlFlags()
        self.control_pin_resource_set = dict()
        self.resource_spec = dict()
        self.resource_set_spec = dict()
        self.resource_candidate = dict()
        self.resource = dict()
        self.resource_set = dict()
        self.box_requested = dict()
        self.box_assigned = dict()
        self.box_applied = dict()

    def _GenerateTimestamp(self):
        return self._yt_client.generate_timestamp()

    def ReadControlFlagsNode(self) -> None:
        with log.debug('DB::ReadControlFlagsNode()'):
            self.control_flags = ReadControlFlagsNode(self._yt_client, self._base_path)

    def _ReadControlPinResourceSetTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadControlPinResourceSetTable()'):
            result = TUpdatedObjects()
            pins = deepcopy(self.control_pin_resource_set)
            self.control_pin_resource_set = ReadControlPinResourceSetTable(
                self._yt_client, self._base_path, retention_timestamp=0
            )
            pins.update(self.control_pin_resource_set)
            for resource_set_spec_id, resource_set_id in pins.items():
                if resource_set_spec_id in self.resource_set_spec:
                    result.resource_set_spec.add(self.resource_set_spec[resource_set_spec_id])
                if resource_set_id in self.resource_set:
                    result.resource_set.add(self.resource_set[resource_set_id])
            return result

    def _ReadControlTables(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadControlTables()'):
            result = TUpdatedObjects()
            result += self._ReadControlPinResourceSetTable()
            return result

    def _ReadResourceSpecTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadResourceSpecTable()'):
            result = TUpdatedObjects()
            data = ReadResourceSpecTable(self._yt_client, self._base_path, self._retention_timestamp)
            for proto in data:
                result += self.UpdateResourceSpec(proto=proto, insert=False)
            return result

    def _ReadResourceSetSpecTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadResourceSetSpecTable()'):
            result = TUpdatedObjects()
            data = ReadResourceSetSpecTable(self._yt_client, self._base_path, self._retention_timestamp)
            for proto in data:
                result += self.UpdateResourceSetSpec(proto=proto, insert=False)
            return result

    def _ReadResourceCandidateTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadResourceCandidateTable()'):
            result = TUpdatedObjects()
            data = ReadResourceCandidateTable(self._yt_client, self._base_path, self._retention_timestamp)
            for proto in data:
                result += self.UpdateResourceCandidate(proto=proto, insert=False)
            return result

    def _ReadResourceTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadResourceTable()'):
            result = TUpdatedObjects()
            data = ReadResourceTable(self._yt_client, self._base_path, self._retention_timestamp)
            data = sorted(data, key=lambda proto: proto.Version)
            for proto in data:
                result += self.UpdateResource(proto=proto, insert=False)
            return result

    def _ReadResourceSetTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadResourceSetTable()'):
            result = TUpdatedObjects()
            data = ReadResourceSetTable(self._yt_client, self._base_path, self._retention_timestamp)
            data = sorted(data, key=lambda proto: proto.Version)
            for proto in data:
                result += self.UpdateResourceSet(proto=proto, insert=False)
            return result

    def _ReadBoxRequestedTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadBoxRequestedTable()'):
            result = TUpdatedObjects()
            data = ReadBoxRequestedTable(self._yt_client, self._base_path, self._retention_timestamp)
            for proto in data:
                result += self.UpdateBoxRequested(proto=proto, insert=False)
            return result

    def _ReadBoxAssignedTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadBoxAssignedTable()'):
            result = TUpdatedObjects()
            data = ReadBoxAssignedTable(self._yt_client, self._base_path, self._retention_timestamp)
            for proto in data:
                result += self.UpdateBoxAssigned(proto=proto, insert=False)
            return result

    def _ReadBoxAppliedTable(self) -> TUpdatedObjects:
        with log.debug('DB::_ReadBoxAppliedTable()'):
            result = TUpdatedObjects()
            data = ReadBoxAppliedTable(self._yt_client, self._base_path, self._retention_timestamp)
            for proto in data:
                result += self.UpdateBoxApplied(proto=proto, insert=False)
            return result

    def ReadTables(self) -> TUpdatedObjects:
        with log.debug('DB::ReadTables()'):
            timestamp = self._GenerateTimestamp()
            result = TUpdatedObjects()
            result += self._ReadResourceSpecTable()
            result += self._ReadResourceSetSpecTable()
            result += self._ReadResourceCandidateTable()
            result += self._ReadResourceTable()
            result += self._ReadResourceSetTable()
            result += self._ReadBoxRequestedTable()
            result += self._ReadBoxAssignedTable()
            result += self._ReadBoxAppliedTable()
            result += self._ReadControlTables()
            self._retention_timestamp = timestamp
            return result

    def Cleanup(self) -> TUpdatedObjects:
        with log.debug('DB::Cleanup()'):
            result = TUpdatedObjects()
            for fqdn in set(self.box_requested.keys()):
                box_requested = self.box_requested[fqdn]
                if time() - box_requested.mtime > 10 * 60:  # TODO: magic number
                    result += self.DeleteBoxRequested(box_requested)

            for fqdn in set(self.box_assigned.keys()):
                if fqdn not in self.box_requested.keys():
                    box_assigned = self.box_assigned[fqdn]
                    result += self.DeleteBoxAssigned(box_assigned)

            for fqdn in set(self.box_applied.keys()):
                if fqdn not in self.box_requested.keys():
                    box_applied = self.box_applied[fqdn]
                    result += self.DeleteBoxApplied(box_applied)
            return result

    def UpdateControlPinResourceSetTable(self, resource_set_spec_id, resource_set_id):
        log.debug('DB::UpdateControlPinResourceSetTable()')
        InsertControlPinResourceSetTable(self._yt_client, self._base_path, resource_set_spec_id, resource_set_id)

    def UpdateBoxRequested(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        fqdn = proto.DeployPodPersistentFqdn
        if fqdn in self.box_requested:
            box_requested = self.box_requested[fqdn]
            box_requested.update_mtime()
            if proto == box_requested._proto:
                log.debug(f'skip TBoxRequested(SpecId={proto.Spec.SpecId}) from {fqdn}')
                # debug check
                debug.check_box_requested_consistency(box_requested_proto=proto, box_requested=box_requested, db=self)
                return result
            log.debug(f'update TBoxRequested(SpecId={proto.Spec.SpecId}) from {fqdn}')
            box_requested._proto = proto
            set_spec_ids = set([set_spec_proto.Id for set_spec_proto in proto.Spec.ResourceSet])

            specs_to_delete = []
            for resource_set_spec in box_requested.resource_set_spec:
                if resource_set_spec.Id not in set_spec_ids:
                    resource_set_spec.box_requested.remove(box_requested)
                    specs_to_delete.append(resource_set_spec)
            for resource_set_spec in specs_to_delete:
                box_requested.resource_set_spec.remove(resource_set_spec)
        else:
            log.debug(f'new TBoxRequested(SpecId={proto.Spec.SpecId}) from {fqdn}')
            box_requested = TBoxRequested(proto=proto)
            self.box_requested[fqdn] = box_requested
        if insert:
            InsertBoxRequestedTable(self._yt_client, self._base_path, proto)

        for resource_spec_proto in proto.Spec.Resource:
            if resource_spec_proto.Id not in self.resource_spec:
                result += self.UpdateResourceSpec(proto=resource_spec_proto)
        for resource_set_spec_proto in proto.Spec.ResourceSet:
            if resource_set_spec_proto.Id not in self.resource_set_spec:
                result += self.UpdateResourceSetSpec(proto=resource_set_spec_proto)
            resource_set_spec = self.resource_set_spec[resource_set_spec_proto.Id]
            resource_set_spec.box_requested.add(box_requested)
            # log.debug(f'link TBoxRequested({fqdn}) to TResourceSetSpec(Id={resource_set_spec.Id})')
            box_requested.resource_set_spec.add(resource_set_spec)
        result.box_requested.add(box_requested)
        return result

    def UpdateBoxAssigned(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        fqdn = proto.DeployPodPersistentFqdn
        if fqdn in self.box_assigned:
            box_assigned = self.box_assigned[fqdn]
            box_assigned.update_mtime()
            if proto == box_assigned._proto:
                log.debug(f'skip TBoxAssigned(SpecId={proto.Spec.SpecId}) for {fqdn}')
                return result
            log.debug(f'update TBoxAssigned(SpecId={proto.Spec.SpecId}) for {fqdn}')
            box_assigned._proto = proto
        else:
            log.debug(f'new TBoxAssigned(SpecId={proto.Spec.SpecId}) for {fqdn}')
            box_assigned = TBoxAssigned(proto=proto)
            self.box_assigned[fqdn] = box_assigned
        if insert:
            InsertBoxAssignedTable(self._yt_client, self._base_path, proto)
        result.box_assigned.add(box_assigned)
        return result

    def UpdateBoxApplied(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        fqdn = proto.DeployPodPersistentFqdn
        if fqdn in self.box_applied:
            box_applied = self.box_applied[fqdn]
            box_applied.update_mtime()
            if proto == box_applied._proto:
                log.debug(f'skip TBoxApplied() for {fqdn}')
                return result
            log.debug(f'update TBoxApplied() for {fqdn}')
            box_applied._proto = proto
        else:
            log.debug(f'new TBoxApplied() for {fqdn}')
            box_applied = TBoxApplied(proto=proto)
            self.box_applied[fqdn] = box_applied
        box_applied.ready_resources = set(resource.ResourceId for resource in box_applied.Spec.Resource if resource.FileStatus in {main_pb2.EFileStatus.READY, main_pb2.EFileStatus.READY_VERIFY})
        if insert:
            InsertBoxAppliedTable(self._yt_client, self._base_path, proto)
        result.box_applied.add(box_applied)
        return result

    def UpdateResourceSpec(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        if not proto.CreationTime:
            proto.CreationTime = int(time())
            insert = True
        if proto.Id in self.resource_spec:
            self.resource_spec[proto.Id].update_mtime()
            log.debug(f'skip TResourceSpec(Id={proto.Id})')
            return result
        log.debug(f'new TResourceSpec(Id={proto.Id})')
        if insert:
            InsertResourceSpecTable(self._yt_client, self._base_path, proto)
        resource_spec = TResourceSpec(proto=proto)
        self.resource_spec[proto.Id] = resource_spec
        result.resource_spec.add(resource_spec)
        return result

    def UpdateResourceSetSpec(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        proto = deepcopy(proto)
        proto.ClearField('Validator')
        proto.ClearField('Prefetch')
        if proto.Id in self.resource_set_spec:
            self.resource_set_spec[proto.Id].update_mtime()
            log.debug(f'skip TResourceSetSpec(Id={proto.Id})')
            return result
        log.debug(f'new TResourceSetSpec(Id={proto.Id})')
        if not proto.CreationTime:
            proto.CreationTime = int(time())
            insert = True
        if insert:
            InsertResourceSetSpecTable(self._yt_client, self._base_path, proto)
        resource_set_spec = TResourceSetSpec(
            proto=proto, resource_spec=set(self.resource_spec[resource_spec_id] for resource_spec_id in proto.Resource)
        )
        self.resource_set_spec[proto.Id] = resource_set_spec
        result.resource_set_spec.add(resource_set_spec)
        return result

    def UpdateResourceCandidate(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        if not proto.CreationTime:
            proto.CreationTime = int(time())
            insert = True
        if proto.Id in self.resource_candidate:
            self.resource_candidate[proto.Id].update_mtime()
            if proto == self.resource_candidate[proto.Id]._proto:
                log.debug(f'skip TResourceCandidate(Id={proto.Id})')
                return result
            log.debug(f'update TResourceCandidate(Id={proto.Id})')
            if insert:
                InsertResourceCandidateTable(self._yt_client, self._base_path, proto)
            resource_candidate = self.resource_candidate[proto.Id]
            resource_candidate._proto = proto
            result.resource_candidate.add(resource_candidate)
            return result
        log.debug(f'new TResourceCandidate(Id={proto.Id})')
        if insert:
            InsertResourceCandidateTable(self._yt_client, self._base_path, proto)
        resource_candidate = TResourceCandidate(proto)
        self.resource_candidate[proto.Id] = resource_candidate
        result.resource_candidate.add(resource_candidate)
        return result

    def UpdateResource(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        if not proto.CreationTime:
            proto.CreationTime = int(time())
            insert = True
        if proto.Id in self.resource:
            self.resource[proto.Id].update_mtime()
            if proto == self.resource[proto.Id]._proto:
                log.debug(f'skip TResource(Id={proto.Id}, Version={proto.Version})')
                return result
            log.debug(f'update TResource(Id={proto.Id}, Version={proto.Version})')
            if insert:
                InsertResourceTable(self._yt_client, self._base_path, proto)
            resource = self.resource[proto.Id]
            resource._proto = proto
            result.resource.add(resource)
            return result
        log.debug(f'new TResource(Id={proto.Id}, Version={proto.Version})')
        if insert:
            InsertResourceTable(self._yt_client, self._base_path, proto)
        resource_candidate = self.resource_candidate[proto.ResourceCandidateId]
        resource_spec = self.resource_spec[proto.ResourceSpecId]
        resource = TResource(proto=proto, resource_spec=resource_spec, resource_candidate=resource_candidate)
        self.resource[proto.Id] = resource
        resource_spec.resource.append(resource)
        result.resource.add(resource)
        return result

    def UpdateResourceSet(self, proto, insert=True) -> TUpdatedObjects:
        result = TUpdatedObjects()
        if not proto.CreationTime:
            proto.CreationTime = int(time())
            insert = True
        if proto.Id in self.resource_set:
            self.resource_set[proto.Id].update_mtime()
            if proto == self.resource_set[proto.Id]._proto:
                log.debug(f'skip TResourceSet(Id={proto.Id}, Version={proto.Version})')
                return result
            log.debug(f'update TResourceSet(Id={proto.Id}, Version={proto.Version})')
            if insert:
                InsertResourceSetTable(self._yt_client, self._base_path, proto)
            resource_set = self.resource_set[proto.Id]
            resource_set._proto = proto
            result.resource_set.add(resource_set)
            return result
        log.debug(f'new TResourceSet(Id={proto.Id}, Version={proto.Version})')
        if insert:
            InsertResourceSetTable(self._yt_client, self._base_path, proto)
        resource_set_spec = self.resource_set_spec[proto.ResourceSetSpecId]
        resource_set = TResourceSet(
            proto=proto,
            resource_set_spec=resource_set_spec,
            resource=set(self.resource[resource_id] for resource_id in proto.Resource)
        )
        self.resource_set[proto.Id] = resource_set
        resource_set_spec.resource_set.append(resource_set)
        result.resource_set.add(resource_set)
        return result

    def DeleteControlPinResourceSetTable(self, resource_set_spec_id) -> None:
        with log.debug(f'DB::DeleteControlPinResourceSetTable({resource_set_spec_id})'):
            DeleteControlPinResourceSetTable(self._yt_client, self._base_path, resource_set_spec_id)

    def DeleteBoxRequested(self, box_requested) -> TUpdatedObjects:
        fqdn = box_requested.DeployPodPersistentFqdn
        with log.debug(f'DB::DeleteBoxRequested({fqdn})'):
            result = TUpdatedObjects()
            DeleteBoxRequestedTable(self._yt_client, self._base_path, box_requested._proto)
            for resource_set_spec in box_requested.resource_set_spec:
                result.resource_set_spec.add(resource_set_spec)
                resource_set_spec.box_requested.remove(box_requested)
            if fqdn in self.box_requested:
                del self.box_requested[fqdn]
            return result

    def DeleteBoxAssigned(self, box_assigned) -> TUpdatedObjects:
        fqdn = box_assigned.DeployPodPersistentFqdn
        with log.debug(f'DB::DeleteBoxAssigned({fqdn})'):
            result = TUpdatedObjects()
            DeleteBoxAssignedTable(self._yt_client, self._base_path, box_assigned._proto)
            if fqdn in self.box_assigned:
                del self.box_assigned[fqdn]
            result.box_assigned.add(box_assigned)
            return result

    def DeleteBoxApplied(self, box_applied) -> TUpdatedObjects:
        fqdn = box_applied.DeployPodPersistentFqdn
        with log.debug(f'DB::DeleteBoxApplied({fqdn})'):
            result = TUpdatedObjects()
            DeleteBoxAppliedTable(self._yt_client, self._base_path, box_applied._proto)
            if fqdn in self.box_applied:
                del self.box_applied[fqdn]
            result.box_applied.add(box_applied)
            return result
