from copy import deepcopy
from yt.wrapper.errors import YtHttpResponseError, YtResponseError
from infra.dostavlyator.lib.misc import misc
from infra.dostavlyator.lib.data.spec import (
    TResource,
    TResourceCandidate,
    TResourceSet,
    TResourceSetSpec,
    TResourceSpec,
)
from google.protobuf.json_format import MessageToJson, Parse as JsonToMessage
from infra.dostavlyator.proto import flags_pb2
from infra.dostavlyator.proto.main_pb2 import (
    TBoxRequestedSpec,
    TBoxAppliedSpec,
    TBoxAssignedSpec,
)
from infra.dostavlyator.proto.tables_pb2 import (
    TBoxAppliedTable,
    TBoxAssignedTable,
    TBoxRequestedTable,
)

log = misc.GetLogger('dostavlyator.lib.tables')

default_schema = [
    {'name': 'Id', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
    {'name': 'Spec', 'type': 'string', 'required': 'true'},
]

default_attributes = {
    'schema': default_schema,
    'dynamic': True,
    'in_memory_mode': 'none',
    'atomicity': 'full',
    'merge_rows_on_flush': True,
}


def GetPathControl(base_path):
    return base_path + '/control'


def GetPathControlFlagsNode(base_path):
    return GetPathControl(base_path) + '/flags'


def GetPathControlPinResourceSetTable(base_path):
    return GetPathControl(base_path) + '/pin_resource_set'


def GetPathResourceSpecTable(base_path):
    return base_path + '/resource_spec'


def GetPathResourceSetSpecTable(base_path):
    return base_path + '/resource_set_spec'


def GetPathResourceCandidateTable(base_path):
    return base_path + '/resource_candidate'


def GetPathResourceTable(base_path):
    return base_path + '/resource'


def GetPathResourceSetTable(base_path):
    return base_path + '/resource_set'


def GetPathBoxRequestedTable(base_path):
    return base_path + '/box_requested'


def GetPathBoxAssignedTable(base_path):
    return base_path + '/box_assigned'


def GetPathBoxAppliedTable(base_path):
    return base_path + '/box_applied'


def CreateAllTables(yt_client, base_path, **attrs):
    for f in [
        CreateControlDirectory,
        CreateControlPinFlagsNode,
        CreateControlPinResourceSetTable,
        CreateResourceSpecTable,
        CreateResourceSetSpecTable,
        CreateResourceCandidateTable,
        CreateResourceTable,
        CreateResourceSetTable,
        CreateBoxRequestedTable,
        CreateBoxAssignedTable,
        CreateBoxAppliedTable,
    ]:
        try:
            f(yt_client=yt_client, base_path=base_path, **attrs)
        except (YtHttpResponseError, YtResponseError) as e:
            if not e.contains_code(501):  # 501 - allreay exists
                raise
            log.debug(e.error['message'])


def RemoveAllTables(yt_client, base_path):
    for path_func in [
        GetPathControlFlagsNode,
        GetPathControlPinResourceSetTable,
        GetPathResourceSpecTable,
        GetPathResourceSetSpecTable,
        GetPathResourceCandidateTable,
        GetPathResourceTable,
        GetPathResourceSetTable,
        GetPathBoxRequestedTable,
        GetPathBoxAssignedTable,
        GetPathBoxAppliedTable,
    ]:
        try:
            yt_client.remove(path_func(base_path))
        except (YtHttpResponseError, YtResponseError) as e:
            log.debug(e.error['message'])


def CreateControlDirectory(yt_client, base_path, **attrs):
    attributes = {}
    attributes.update(attrs)
    yt_client.create(type='map_node', path=GetPathControl(base_path), attributes=attributes)


def CreateControlPinFlagsNode(yt_client, base_path, **attrs):
    attributes = {}
    attributes.update(attrs)
    yt_client.create(type='string_node', path=GetPathControlFlagsNode(base_path), attributes=attributes)


def CreateControlPinResourceSetTable(yt_client, base_path, **attrs):
    attributes = deepcopy(default_attributes)
    attributes.update(attrs)
    schema = [
        {'name': 'ResourceSetSpecId', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'ResourceSetId', 'type': 'string', 'required': 'true'},
    ]
    attributes['schema'] = schema
    yt_client.create(type='table', path=GetPathControlPinResourceSetTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathControlPinResourceSetTable(base_path))


def CreateResourceSpecTable(yt_client, base_path, **attrs):
    attributes = deepcopy(default_attributes)
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathResourceSpecTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathResourceSpecTable(base_path))


def CreateResourceSetSpecTable(yt_client, base_path, **attrs):
    attributes = deepcopy(default_attributes)
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathResourceSetSpecTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathResourceSetSpecTable(base_path))


def CreateResourceCandidateTable(yt_client, base_path, **attrs):
    attributes = deepcopy(default_attributes)
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathResourceCandidateTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathResourceCandidateTable(base_path))


def CreateResourceTable(yt_client, base_path, **attrs):
    attributes = deepcopy(default_attributes)
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathResourceTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathResourceTable(base_path))


def CreateResourceSetTable(yt_client, base_path, **attrs):
    attributes = deepcopy(default_attributes)
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathResourceSetTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathResourceSetTable(base_path))


def CreateBoxRequestedTable(yt_client, base_path, **attrs):
    schema = [
        {'name': 'DeployPodPersistentFqdn', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'DeployBoxId', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'DeployProjectId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployStageId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployUnitId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployPodId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployNodeDC', 'type': 'string', 'required': 'true'},
        {'name': 'Spec', 'type': 'string', 'required': 'true'},
    ]
    attributes = deepcopy(default_attributes)
    attributes['schema'] = schema
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathBoxRequestedTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathBoxRequestedTable(base_path))


def CreateBoxAssignedTable(yt_client, base_path, **attrs):
    schema = [
        {'name': 'DeployPodPersistentFqdn', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'DeployBoxId', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'DeployProjectId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployStageId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployUnitId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployPodId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployNodeDC', 'type': 'string', 'required': 'true'},
        {'name': 'Spec', 'type': 'string', 'required': 'true'},
    ]
    attributes = deepcopy(default_attributes)
    attributes['schema'] = schema
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathBoxAssignedTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathBoxAssignedTable(base_path))


def CreateBoxAppliedTable(yt_client, base_path, **attrs):
    schema = [
        {'name': 'DeployPodPersistentFqdn', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'DeployBoxId', 'type': 'string', 'required': 'true', 'sort_order': 'ascending'},
        {'name': 'DeployProjectId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployStageId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployUnitId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployPodId', 'type': 'string', 'required': 'true'},
        {'name': 'DeployNodeDC', 'type': 'string', 'required': 'true'},
        {'name': 'Spec', 'type': 'string', 'required': 'true'},
    ]
    attributes = deepcopy(default_attributes)
    attributes['schema'] = schema
    attributes.update(attrs)
    yt_client.create(type='table', path=GetPathBoxAppliedTable(base_path), attributes=attributes)
    yt_client.mount_table(GetPathBoxAppliedTable(base_path))


def ReadResourceSpecTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathResourceSpecTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResourceSpec())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def ReadResourceSetSpecTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathResourceSetSpecTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResourceSetSpec())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def ReadResourceCandidateTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathResourceCandidateTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResourceCandidate())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def LookupResourceCandidateTable(yt_client, base_path, records_filter):
    rows = yt_client.lookup_rows(GetPathResourceCandidateTable(base_path), records_filter)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResourceCandidate())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def LockResourceCandidateTable(yt_client, base_path, records_filter):
    yt_client.lock_rows(GetPathResourceCandidateTable(base_path), records_filter, lock_type="exclusive")


def ReadResourceTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathResourceTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResource())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def LookupResourceTable(yt_client, base_path, records_filter):
    rows = yt_client.lookup_rows(GetPathResourceTable(base_path), records_filter)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResource())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def LockResourceTable(yt_client, base_path, records_filter):
    yt_client.lock_rows(GetPathResourceTable(base_path), records_filter, lock_type="exclusive")


def ReadResourceSetTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathResourceSetTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResourceSet())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def LookupResourceSetTable(yt_client, base_path, records_filter):
    rows = yt_client.lookup_rows(GetPathResourceSetTable(base_path), records_filter)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TResourceSet())
        assert spec.IsInitialized()
        result.append(spec)
    return result


def LockResourceSetTable(yt_client, base_path, records_filter):
    yt_client.lock_rows(GetPathResourceSetTable(base_path), records_filter, lock_type="exclusive")


def LockControlFlagsNode(yt_client, base_path):
    yt_client.lock(GetPathControlFlagsNode(base_path), mode="exclusive")


def UnLockControlFlagsNode(yt_client, base_path):
    yt_client.lock(GetPathControlFlagsNode(base_path))


def ReadControlFlagsNode(yt_client, base_path):
    str_data = yt_client.get(GetPathControlFlagsNode(base_path))
    if not str_data:
        return flags_pb2.TControlFlags()
    return JsonToMessage(str_data, flags_pb2.TControlFlags())


def ReadControlPinResourceSetTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathControlPinResourceSetTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    return dict((row["ResourceSetSpecId"], row["ResourceSetId"]) for row in rows)


def ReadBoxRequestedTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathBoxRequestedTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        fqdn = row['DeployPodPersistentFqdn']
        try:
            spec = JsonToMessage(row['Spec'], TBoxRequestedSpec())
            box_requested = TBoxRequestedTable(
                DeployPodPersistentFqdn=fqdn,
                DeployPodId=row['DeployPodId'],
                DeployNodeDC=row['DeployNodeDC'],
                DeployProjectId=row['DeployProjectId'],
                DeployStageId=row['DeployStageId'],
                DeployUnitId=row['DeployUnitId'],
                DeployBoxId=row['DeployBoxId'],
                Spec=spec,
            )
            assert box_requested.IsInitialized()
            result.append(box_requested)
        except Exception as e:
            log.warning(f'Failed to load TBoxRequested(fqdn={fqdn}), skip: {e}')
    return result


def ReadBoxAssignedTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathBoxAssignedTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TBoxAssignedSpec())
        box_assigned = TBoxAssignedTable(
            DeployPodPersistentFqdn=row['DeployPodPersistentFqdn'],
            DeployPodId=row['DeployPodId'],
            DeployNodeDC=row['DeployNodeDC'],
            DeployProjectId=row['DeployProjectId'],
            DeployStageId=row['DeployStageId'],
            DeployUnitId=row['DeployUnitId'],
            DeployBoxId=row['DeployBoxId'],
            Spec=spec,
        )
        assert box_assigned.IsInitialized()
        result.append(box_assigned)
    return result


def ReadBoxAppliedTable(yt_client, base_path, retention_timestamp):
    query = '* FROM [{}]'.format(GetPathBoxAppliedTable(base_path))
    rows = yt_client.select_rows(query, allow_full_scan=True, retention_timestamp=retention_timestamp)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TBoxAppliedSpec())
        box_applied = TBoxAppliedTable(
            DeployPodPersistentFqdn=row['DeployPodPersistentFqdn'],
            DeployPodId=row['DeployPodId'],
            DeployNodeDC=row['DeployNodeDC'],
            DeployProjectId=row['DeployProjectId'],
            DeployStageId=row['DeployStageId'],
            DeployUnitId=row['DeployUnitId'],
            DeployBoxId=row['DeployBoxId'],
            Spec=spec,
        )
        assert box_applied.IsInitialized()
        result.append(box_applied)
    return result


def LookupBoxRequestedTable(yt_client, base_path, records_filter):
    rows = yt_client.lookup_rows(GetPathBoxRequestedTable(base_path), records_filter)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TBoxRequestedSpec())
        box_requested = TBoxRequestedTable(
            DeployPodPersistentFqdn=row['DeployPodPersistentFqdn'],
            DeployPodId=row['DeployPodId'],
            DeployNodeDC=row['DeployNodeDC'],
            DeployProjectId=row['DeployProjectId'],
            DeployStageId=row['DeployStageId'],
            DeployUnitId=row['DeployUnitId'],
            DeployBoxId=row['DeployBoxId'],
            Spec=spec,
        )
        assert box_requested.IsInitialized()
        result.append(box_requested)
    return result


def LookupBoxAssignedTable(yt_client, base_path, records_filter):
    rows = yt_client.lookup_rows(GetPathBoxAssignedTable(base_path), records_filter)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TBoxAssignedSpec())
        box_assigned = TBoxAssignedTable(
            DeployPodPersistentFqdn=row['DeployPodPersistentFqdn'],
            DeployPodId=row['DeployPodId'],
            DeployNodeDC=row['DeployNodeDC'],
            DeployProjectId=row['DeployProjectId'],
            DeployStageId=row['DeployStageId'],
            DeployUnitId=row['DeployUnitId'],
            DeployBoxId=row['DeployBoxId'],
            Spec=spec,
        )
        assert box_assigned.IsInitialized()
        result.append(box_assigned)
    return result


def LookupBoxAppliedTable(yt_client, base_path, records_filter) -> TBoxAppliedTable:
    rows = yt_client.lookup_rows(GetPathBoxAppliedTable(base_path), records_filter)
    result = list()
    for row in rows:
        spec = JsonToMessage(row['Spec'], TBoxAppliedSpec())
        box_assigned = TBoxAppliedTable(
            DeployPodPersistentFqdn=row['DeployPodPersistentFqdn'],
            DeployPodId=row['DeployPodId'],
            DeployNodeDC=row['DeployNodeDC'],
            DeployProjectId=row['DeployProjectId'],
            DeployStageId=row['DeployStageId'],
            DeployUnitId=row['DeployUnitId'],
            DeployBoxId=row['DeployBoxId'],
            Spec=spec,
        )
        assert box_assigned.IsInitialized()
        result.append(box_assigned)
    return result


def InsertControlFlagsNode(yt_client, base_path, proto):
    data = MessageToJson(proto, including_default_value_fields=False, preserving_proto_field_name=True)
    yt_client.set(GetPathControlFlagsNode(base_path), data)


def InsertControlPinResourceSetTable(yt_client, base_path, resource_set_spec_id, resource_set_id):
    data = [{'ResourceSetSpecId': resource_set_spec_id, 'ResourceSetId': resource_set_id}]
    return yt_client.insert_rows(GetPathControlPinResourceSetTable(base_path), data, require_sync_replica=True)


def InsertResourceSpecTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [{'Id': proto.Id, 'Spec': spec}]
    return yt_client.insert_rows(GetPathResourceSpecTable(base_path), data, require_sync_replica=True)


def InsertResourceSetSpecTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [{'Id': proto.Id, 'Spec': spec}]
    return yt_client.insert_rows(GetPathResourceSetSpecTable(base_path), data, require_sync_replica=True)


def InsertResourceCandidateTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [{'Id': proto.Id, 'Spec': spec}]
    return yt_client.insert_rows(GetPathResourceCandidateTable(base_path), data, require_sync_replica=True)


def InsertResourceTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [{'Id': proto.Id, 'Spec': spec}]
    return yt_client.insert_rows(GetPathResourceTable(base_path), data, update=True, require_sync_replica=True)


def InsertResourceSetTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [{'Id': proto.Id, 'Spec': spec}]
    return yt_client.insert_rows(GetPathResourceSetTable(base_path), data, require_sync_replica=True)


def InsertBoxRequestedTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto.Spec, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [
        {
            'DeployPodPersistentFqdn': proto.DeployPodPersistentFqdn,
            'DeployPodId': proto.DeployPodId,
            'DeployNodeDC': proto.DeployNodeDC,
            'DeployProjectId': proto.DeployProjectId,
            'DeployStageId': proto.DeployStageId,
            'DeployUnitId': proto.DeployUnitId,
            'DeployBoxId': proto.DeployBoxId,
            'Spec': spec,
        }
    ]
    return yt_client.insert_rows(GetPathBoxRequestedTable(base_path), data, require_sync_replica=True)


def InsertBoxAssignedTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto.Spec, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [
        {
            'DeployPodPersistentFqdn': proto.DeployPodPersistentFqdn,
            'DeployPodId': proto.DeployPodId,
            'DeployNodeDC': proto.DeployNodeDC,
            'DeployProjectId': proto.DeployProjectId,
            'DeployStageId': proto.DeployStageId,
            'DeployUnitId': proto.DeployUnitId,
            'DeployBoxId': proto.DeployBoxId,
            'Spec': spec,
        }
    ]
    return yt_client.insert_rows(GetPathBoxAssignedTable(base_path), data, require_sync_replica=True)


def InsertBoxAppliedTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    spec = MessageToJson(proto.Spec, including_default_value_fields=False, preserving_proto_field_name=True)
    data = [
        {
            'DeployPodPersistentFqdn': proto.DeployPodPersistentFqdn,
            'DeployPodId': proto.DeployPodId,
            'DeployNodeDC': proto.DeployNodeDC,
            'DeployProjectId': proto.DeployProjectId,
            'DeployStageId': proto.DeployStageId,
            'DeployUnitId': proto.DeployUnitId,
            'DeployBoxId': proto.DeployBoxId,
            'Spec': spec,
        }
    ]
    return yt_client.insert_rows(GetPathBoxAppliedTable(base_path), data, require_sync_replica=True)


def DeleteControlPinResourceSetTable(yt_client, base_path, resource_set_spec_id):
    data = [{"ResourceSetSpecId": resource_set_spec_id}]
    return yt_client.delete_rows(GetPathControlPinResourceSetTable(base_path), data, require_sync_replica=True)


def DeleteBoxRequestedTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    data = [
        {
            'DeployPodPersistentFqdn': proto.DeployPodPersistentFqdn,
            'DeployBoxId': proto.DeployBoxId,
        }
    ]
    return yt_client.delete_rows(GetPathBoxRequestedTable(base_path), data, require_sync_replica=True)


def DeleteBoxAssignedTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    data = [
        {
            'DeployPodPersistentFqdn': proto.DeployPodPersistentFqdn,
            'DeployBoxId': proto.DeployBoxId,
        }
    ]
    return yt_client.delete_rows(GetPathBoxAssignedTable(base_path), data, require_sync_replica=True)


def DeleteBoxAppliedTable(yt_client, base_path, proto):
    assert proto.IsInitialized()
    data = [
        {
            'DeployPodPersistentFqdn': proto.DeployPodPersistentFqdn,
            'DeployBoxId': proto.DeployBoxId,
        }
    ]
    return yt_client.delete_rows(GetPathBoxAppliedTable(base_path), data, require_sync_replica=True)
