import humanfriendly
import signal
import re
from os import unlink, symlink, rename, makedirs, getenv
from os.path import dirname
from sys import exc_info, exit
import pathlib
import collections
import logging
import shutil
import hashlib
import json
from infra.dostavlyator.lib.data import spec
from infra.dostavlyator.proto.main_pb2 import TBoxRequestedSpec, TDownloadSpec

GLOBAL_LOGGER_INDENT = 0


def GetLogger(name):
    class MyLogger(logging.LoggerAdapter):
        def indent(self):
            global GLOBAL_LOGGER_INDENT
            return ' ' * (2 * GLOBAL_LOGGER_INDENT) if GLOBAL_LOGGER_INDENT > 0 else ''

        def genmsg(self, msg):
            return self.indent() + str(msg)

        def debug(self, msg, **kwargs):
            (_type, _value, _traceback) = exc_info()
            if not ('exc_info' in kwargs or _type is None):
                kwargs['exc_info'] = True
            self.logger.debug(self.genmsg(msg), **kwargs)
            return self

        def info(self, msg, **kwargs):
            (_type, _value, _traceback) = exc_info()
            if not ('exc_info' in kwargs or _type is None):
                kwargs['exc_info'] = True
            self.logger.info(self.genmsg(msg), **kwargs)
            return self

        def warning(self, msg, **kwargs):
            (_type, _value, _traceback) = exc_info()
            if not ('exc_info' in kwargs or _type is None):
                kwargs['exc_info'] = True
            self.logger.warning(self.genmsg(msg), **kwargs)
            return self

        def error(self, msg, **kwargs):
            (_type, _value, _traceback) = exc_info()
            if not ('exc_info' in kwargs or _type is None):
                kwargs['exc_info'] = True
            self.logger.error(self.genmsg(msg), **kwargs)
            return self

        def __enter__(self):
            global GLOBAL_LOGGER_INDENT
            GLOBAL_LOGGER_INDENT += 1
            return self

        def __exit__(self, exc_type, exc_value, traceback):
            global GLOBAL_LOGGER_INDENT
            GLOBAL_LOGGER_INDENT -= 1
            return False

    logger = MyLogger(logging.getLogger(__name__), extra=None)
    return logger


log = GetLogger('dostavlyator.lib.misc')


TURI = collections.namedtuple(typename="TURI", field_names=["scheme", "authority", "path", "query", "fragment"])
uri_regex = re.compile(
    r"^(([^:/?#]+):)?(//([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?"
)  # https://datatracker.ietf.org/doc/html/rfc3986#appendix-B


LOG_FORMAT = "%(asctime)s [%(levelname)s] %(message)s"
DATE_FORMAT = "%Y/%m/%d %H:%M:%S"


def signal_handler(signo, frame):
    exit()


def register_signal_handlers():
    signal.signal(signal.SIGTERM, signal_handler)
    signal.signal(signal.SIGQUIT, signal_handler)


def uriparse(uri):
    match = uri_regex.match(uri)
    return TURI(
        scheme=match.group(2),
        authority=match.group(4),
        path=match.group(5),
        query=match.group(7),
        fragment=match.group(9),
    )


def safe_stat(path: pathlib.Path):
    try:
        path_stat = path.stat()
        if path.is_dir():
            x = [safe_stat(f) for f in path.iterdir()]
            if not x:
                return dict(size=0, mtime=0)
            size = sum(i['size'] for i in x)
            mtime = max(i['mtime'] for i in x)
            return dict(size=size, mtime=mtime)
        elif path.is_file():
            size = path_stat.st_size
            mtime = path_stat.st_mtime
            return dict(size=size, mtime=mtime)
        else:
            raise Exception("Unsupported path type")
    except:
        log.exception("Exception on \"safe_stat\" call for path = \"{path}\", details:")
        raise  # TODO: debug


def safe_unlink(path) -> None:
    try:
        unlink(path)
    except FileNotFoundError:
        pass


def safe_symlink(src, dst):
    tmp_link = dst + '_tmp'
    try:
        dst_dir = dirname(dst)
        if dst_dir:
            log.debug(f'makedirs("{dst_dir}", mode=0o755, exist_ok=True)')
            makedirs(dst_dir, mode=0o755, exist_ok=True)
        safe_unlink(tmp_link)
        log.debug(f'create symlink "{src}"->"{dst}"')
        symlink(src, tmp_link)
        rename(tmp_link, dst)
    except OSError:
        safe_unlink(tmp_link)
        raise


def safe_file_md5(path: pathlib.Path):
    file_hash = hashlib.md5()
    try:
        with open(str(path), "rb") as f:
            while True:
                chunk = f.read(8192)
                if not chunk:
                    break
                file_hash.update(chunk)
    except:
        log.exception("Exception on \"safe_file_md5\" call for path = \"{path}\", details:")
        raise  # TODO: debug
        return None
    return file_hash


def safe_remove(path: pathlib.Path) -> bool:
    try:
        if not path.exists():
            return
        if path.is_dir():
            shutil.rmtree(path)
        elif path.is_file() or path.is_symlink():
            log.warning(f"Unexpected object found in storage directory: {path}, trying to remove ...")
            path.unlink()
        else:
            raise Exception("Unsupported path type")
        return True
    except:
        log.exception(f"Exception on \"safe_remove\" call for path = \"{path}\", details:")
        raise  # TODO: debug
        return False


def get_instance_details():
    return {
        "DeployPodPersistentFqdn": getenv('DEPLOY_POD_PERSISTENT_FQDN', 'localhost'),
        "DeployProjectId": getenv('DEPLOY_PROJECT_ID', 'dostavlyator_local_project'),
        "DeployStageId": getenv('DEPLOY_STAGE_ID', 'dostavlyator_local_stage'),
        "DeployUnitId": getenv('DEPLOY_UNIT_ID', 'dostavlyator_local_unit'),
        "DeployBoxId": getenv('DEPLOY_BOX_ID', 'dostavlyator_local_box'),
        "DeployPodId": getenv('DEPLOY_POD_ID', 'localhost'),
        "DeployNodeDC": getenv('DEPLOY_NODE_DC', 'dostavlyator_local_dc'),
    }


def mkjson(**kwargs):
    return json.dumps(kwargs)


def configure_logging(verbose):
    root_log = logging.getLogger("infra.dostavlyator")
    root_log.setLevel(level=logging.DEBUG if verbose else logging.INFO)
    console_handler = logging.StreamHandler()
    console_handler.setLevel(logging.DEBUG)
    formatter = logging.Formatter(fmt=LOG_FORMAT, datefmt=DATE_FORMAT)
    console_handler.setFormatter(formatter)
    root_log.addHandler(console_handler)


def read_spec_file(filename: str) -> TBoxRequestedSpec:
    with open(filename, 'rb') as f:
        file_bytes = f.read()
        return from_json_box_requested_spec(
            json.loads(file_bytes.decode('utf-8')), hashlib.sha256(file_bytes).hexdigest()
        )


def from_json_box_requested_spec(box_requested_spec: dict, box_requested_spec_hash: str) -> TBoxRequestedSpec:
    resource = list()
    resource_set = list()

    resource_defaults = box_requested_spec.get('ResourceDefaults', {})
    resource_set_defaults = box_requested_spec.get('ResourceSetDefaults', {})

    for resource_spec in box_requested_spec.get('Resource', []):
        resource.append(from_json_resource_spec(resource_spec, resource_defaults))

    for resource_set_spec in box_requested_spec.get('ResourceSet', []):
        x = from_json_resource_set_spec(resource_set_spec, resource_defaults, resource_set_defaults)
        resource_set.append(x[0])
        resource.extend(x[1::])

    download_spec = from_json_download_spec(box_requested_spec.get('DownloadSpec', {}))
    return TBoxRequestedSpec(
        Resource=resource, ResourceSet=resource_set, DownloadSpec=download_spec, SpecId=box_requested_spec_hash
    )


def from_json_resource_spec(resource_spec: dict, resource_defaults: dict) -> spec.TResourceSpec:
    resource_spec = resource_defaults | resource_spec
    if not isinstance(resource_spec['GatherUriOpt'], str):
        resource_spec['GatherUriOpt'] = json.dumps(resource_spec['GatherUriOpt'])
    if isinstance(resource_spec.get('GatherPeriodSec'), str):
        resource_spec['GatherPeriodSec'] = int(humanfriendly.parse_timespan(resource_spec['GatherPeriodSec']))
    if isinstance(resource_spec.get('AlertGatherPeriodSec'), str):
        resource_spec['AlertGatherPeriodSec'] = int(humanfriendly.parse_timespan(resource_spec['AlertGatherPeriodSec']))
    if isinstance(resource_spec.get('AlertSourceAgeSec'), str):
        resource_spec['AlertSourceAgeSec'] = int(humanfriendly.parse_timespan(resource_spec['AlertSourceAgeSec']))
    if isinstance(resource_spec.get('MaxDeployDurationSec'), str):
        resource_spec['MaxDeployDurationSec'] = int(humanfriendly.parse_timespan(resource_spec['MaxDeployDurationSec']))
    return spec.TResourceSpec(**resource_spec)


def from_json_resource_set_spec(resource_set_spec: dict, resource_defaults: dict, resource_set_defaults: dict) -> list:
    resource = list()
    resource_set_spec = resource_set_defaults | resource_set_spec
    for i, r in enumerate(resource_set_spec['Resource']):
        if isinstance(r, dict):
            resource.append(from_json_resource_spec(r, resource_defaults))
            resource_set_spec['Resource'][i] = resource[-1].Id

    resource_set = spec.TResourceSetSpec(**resource_set_spec)
    return [resource_set] + resource


def from_json_download_spec(download_spec: dict) -> TDownloadSpec:
    return TDownloadSpec(**download_spec)
