import os
import sys
import signal
import atexit
import flask
from time import time, sleep

from library.python.monlib.metric_registry import MetricRegistry
from library.python.monlib.encoder import dumps

from infra.dostavlyator.lib.misc.misc import GetLogger
from infra.dostavlyator.lib.data import spec
from infra.dostavlyator.proto import main_pb2

log = GetLogger('dostavlyator.lib.metrics')
CONTENT_TYPE_SPACK = 'application/x-solomon-spack'
CONTENT_TYPE_JSON = 'application/json'
server_pid = None
start_time = time()
app = flask.Flask(__name__)
__DB = None


def stop_server():
    global server_pid
    if server_pid:
        log.debug('stop metrics server')
        print("stop metrics server", file=sys.stderr)
        os.kill(server_pid, signal.SIGTERM)
        sleep(1)
        os.kill(server_pid, signal.SIGKILL)
        os.waitpid(server_pid, 0)
        server_pid = None


def start_server(host, port, db):
    global server_pid
    global __DB
    stop_server()
    log.debug('start metrics server')
    server_pid = os.fork()
    if server_pid:
        atexit.unregister(stop_server)
        atexit.register(stop_server)
    else:  # child
        try:
            __DB = db
            app.run(host=host, port=port)
        except SystemExit:
            pass
        os._exit(os.EX_OK)


@app.route('/metrics')
def metrics():
    metrics = MetricRegistry()
    # uptime
    metric_uptime = metrics.int_gauge({'sensor': 'sys.uptime_sec'})
    metric_uptime.set(time() - start_time)
    # box_requested
    for box_requested in __DB.box_requested.values():
        metric_box_requested_age = metrics.int_gauge(
            {'host': 'leader', 'sensor': 'box_requested.age', 'fqdn': box_requested.DeployPodPersistentFqdn}
        )
        metric_box_requested_age.set(time() - box_requested.mtime)
    # resource_set_spec
    for resource_set_spec in __DB.resource_set_spec.values():
        resource_set_spec_labels = {
            'host': 'leader',
            'id': resource_set_spec.Id,
            'path': resource_set_spec.Path,
            'name': resource_set_spec.Name,
        }
        metric_resource_set_spec_age = metrics.int_gauge(resource_set_spec_labels | {'sensor': 'resource_set_spec.age'})
        metric_resource_set_spec_age.set(time() - resource_set_spec.ctime)
        metric_resource_set_spec_usage = metrics.int_gauge(
            resource_set_spec_labels | {'sensor': 'resource_set_spec.usage'}
        )
        metric_resource_set_spec_usage.set(len(resource_set_spec.box_requested))
        last_resource_set = resource_set_spec.GetLast()
        if last_resource_set:
            metric_resource_set_spec_last_resource_set_age = metrics.int_gauge(
                resource_set_spec_labels | {'sensor': 'resource_set_spec.last_resource_set.age'}
            )
            metric_resource_set_spec_last_resource_set_age.set(time() - last_resource_set.ctime)
            metric_resource_set_spec_last_resource_set_min_source_age = metrics.int_gauge(
                resource_set_spec_labels | {'sensor': 'resource_set_spec.last_resource_set.min_source_age'}
            )
            min_source_creation_time = min(x.resource_candidate.SourceCreationTime for x in last_resource_set.resource)
            metric_resource_set_spec_last_resource_set_min_source_age.set(time() - min_source_creation_time)

            add_validation_status_metrics(
                metrics=metrics, last_resource_set=last_resource_set, resource_set_spec_labels=resource_set_spec_labels
            )

        last_valid_resource_set = resource_set_spec.GetLastValid()
        if last_valid_resource_set:
            metric_resource_set_spec_last_valid_resource_set_age = metrics.int_gauge(
                resource_set_spec_labels | {'sensor': 'resource_set_spec.last_valid_resource_set.age'}
            )
            metric_resource_set_spec_last_valid_resource_set_age.set(time() - last_valid_resource_set.ctime)

    # resource_spec
    for resource_spec in __DB.resource_spec.values():
        resource_spec_labels = {
            'host': 'leader',
            'id': resource_spec.Id,
            'path': resource_spec.Path,
            'name': resource_spec.Name,
        }
        metric_resource_spec_age = metrics.int_gauge(resource_spec_labels | {'sensor': 'resource_spec.age'})
        metric_resource_spec_age.set(time() - resource_spec.ctime)
        last_resource = resource_spec.GetLast()
        if last_resource:
            metric_resource_spec_last_resource_age = metrics.int_gauge(
                resource_spec_labels | {'sensor': 'resource_spec.last_resource.age'}
            )
            metric_resource_spec_last_resource_age.set(time() - last_resource.ctime)
            metric_resource_spec_last_resource_source_age = metrics.int_gauge(
                resource_spec_labels | {'sensor': 'resource_spec.last_resource.source_age'}
            )
            metric_resource_spec_last_resource_source_age.set(
                time() - last_resource.resource_candidate.SourceCreationTime
            )
        last_valid_resource = resource_spec.GetLastValid()
        if last_valid_resource:
            metric_resource_spec_last_valid_resource_age = metrics.int_gauge(
                resource_spec_labels | {'sensor': 'resource_spec.last_valid_resource.age'}
            )
            metric_resource_spec_last_valid_resource_age.set(time() - last_valid_resource.ctime)
            metric_resource_spec_last_valid_resource_source_age = metrics.int_gauge(
                resource_spec_labels | {'sensor': 'resource_spec.last_valid_resource.source_age'}
            )
            metric_resource_spec_last_valid_resource_source_age.set(
                time() - last_valid_resource.resource_candidate.SourceCreationTime
            )

    # box_assigned
    assigned_resources = dict()
    for box_assigned in __DB.box_assigned.values():
        for resource in box_assigned.Spec.Resource:
            if resource.Id in assigned_resources:
                assigned_resources[resource.Id] += 1
            else:
                assigned_resources[resource.Id] = 0

    # box_applied
    applied_resources = dict()
    for box_applied in __DB.box_applied.values():
        for resource in box_applied.Spec.Resource:
            if resource.ResourceId in applied_resources:
                applied_resources[resource.ResourceId] += 1
            else:
                applied_resources[resource.ResourceId] = 0
        for active_resource_set_proto in box_applied.Spec.ActiveResourceSet:
            resource_set = __DB.resource_set[active_resource_set_proto.ResourceSetId]
            resource_set_spec = resource_set.resource_set_spec
            metric_resource_set_spec_resource_set_min_source_age = metrics.int_gauge(
                {
                    'host': 'leader',
                    'fqdn': box_applied.DeployPodPersistentFqdn,
                    'sensor': 'box_applied.resource_set_spec.resource_set.min_source_age',
                    'id': resource_set_spec.Id,
                    'path': resource_set_spec.Path,
                    'name': resource_set_spec.Name,
                }
            )
            min_source_creation_time = min(x.resource_candidate.SourceCreationTime for x in resource_set.resource)
            metric_resource_set_spec_resource_set_min_source_age.set(time() - min_source_creation_time)
            for resource in resource_set.resource:
                resource_spec = resource.resource_spec
                metric_box_applied_resource_spec_active_resource_source_age = metrics.int_gauge(
                    {
                        'host': 'leader',
                        'fqdn': box_applied.DeployPodPersistentFqdn,
                        'sensor': 'box_applied.resource_spec.active_resource.source_age',
                        'id': resource_spec.Id,
                        'path': resource_spec.Path,
                        'name': resource_spec.Name,
                    }
                )
                metric_box_applied_resource_spec_active_resource_source_age.set(
                    time() - resource.resource_candidate.SourceCreationTime
                )

    # resource
    for resource in __DB.resource.values():
        if resource not in assigned_resources.keys() and resource not in applied_resources.keys():
            continue
        resource_labels = {
            'host': 'leader',
            'id': resource.Id,
            'path': resource.resource_spec.Path,
            'name': resource.resource_spec.Name,
        }
        metric_resource_assigned = metrics.int_gauge(resource_labels | {'sensor': 'resource.assigned'})
        metric_resource_assigned.set(assigned_resources[resource.Id])
        metric_resource_applied = metrics.int_gauge(resource_labels | {'sensor': 'resource.applied'})
        metric_resource_applied.set(applied_resources[resource.Id])

    # solomon fetcher will set the Accept header to application/x-solomon-spack,
    # which is more efficient than JSON
    if flask.request.headers.get('accept', None) == CONTENT_TYPE_SPACK:
        return flask.Response(dumps(metrics), mimetype=CONTENT_TYPE_SPACK)

    # but it's a good idea to leave ability to visually inspect JSON
    return flask.Response(dumps(metrics, format='json'), mimetype=CONTENT_TYPE_JSON)


def add_validation_status_metrics(*, metrics, last_resource_set, resource_set_spec_labels):
    statuses = main_pb2.EStatus.values()
    last_resource_set_status = last_resource_set.GetStatus()
    for status in statuses:
        metric_resource_set_spec_validation_status = metrics.int_gauge(
            resource_set_spec_labels | {'sensor': validation_status_sensor_name(status)}
        )
        assert type(status) == type(last_resource_set_status)
        value = 1 if status == last_resource_set_status else 0
        metric_resource_set_spec_validation_status.set(value)


def validation_status_sensor_name(status):
    return 'resource_set_spec.last_resource_set.validation_status.{}'.format(spec.ValidationStatusName(status))
