import datetime
from collections import namedtuple

from juggler_sdk import Check, Child

from library.python.monitoring.solo.objects.solomon.v2 import (
    Project,
    Cluster,
    YpCluster,
    Service,
    Shard,
    Channel,
    Method,
    Juggler,
    Alert,
    Type,
    Threshold,
    PredicateRule,
    Expression,
    MultiAlert,
)
from library.python.monitoring.solo.objects.solomon.sensor import Sensor
from library.python.monitoring.solo.helpers.cli import Controller, find_tokens, SolomonEndpoint, JugglerEndpoint
from library.python.monitoring.solo.util.text import prepare_text

from infra.dostavlyator.lib.solo.metrics import validation_status_sensor_name
from infra.dostavlyator.lib.data import spec
from infra.dostavlyator.lib.misc.misc import GetLogger, get_instance_details
import infra.dostavlyator.proto as dostavlyator_proto

log = GetLogger('dostavlyator.lib.solo')

SoloCfg = namedtuple('SoloCfg', ['abc_service', 'project_id', 'cluster_id', 'endpoint_set_id', 'port'])
controller = None


def makecfg(abc_service: str, port: int) -> SoloCfg:
    i = get_instance_details()
    cfg = SoloCfg(
        abc_service=abc_service,
        project_id=i['DeployProjectId'],
        cluster_id=i['DeployStageId'],
        endpoint_set_id=f"{i['DeployStageId']}.{i['DeployUnitId']}",
        port=port,
    )
    return cfg


def init(yt_cluster: str, yt_base_path: str):
    yt_path = yt_base_path + '/solo_state'
    tokens = {'SOLOMON_TOKEN': None, 'JUGGLER_TOKEN': None, 'YT_TOKEN': None}

    find_tokens(tokens, tokens_secret_id=None)  # TODO: tokens_secret_id
    global controller
    controller = Controller(
        save=True,
        delete_untracked=True,
        clear_untracked_in_state=True,
        state_yt_cluster=yt_cluster,
        state_locke_path=yt_path,
        state_locke_token=tokens["YT_TOKEN"],
        solomon_endpoint=SolomonEndpoint.PRODUCTION,
        solomon_token=tokens["SOLOMON_TOKEN"],
        juggler_endpoint=JugglerEndpoint.PRODUCTION,
        juggler_token=tokens["JUGGLER_TOKEN"],
        juggler_mark=None,  # TODO: juggler_mark
    )


def register(abc_service: str, port: int, db):
    log.debug('solo.register()')
    cfg = makecfg(abc_service, port)

    project = Project(
        id=cfg.project_id,
        name=cfg.project_id,
        abc_service=cfg.abc_service,
        only_auth_push=False,
        only_sensor_name_shards=False,
        only_new_format_writes=False,
        only_new_format_reads=False,
    )

    cluster = Cluster(
        id=cfg.cluster_id,
        project_id=project.id,
        name=cfg.cluster_id,
        use_fqdn=True,
        sensors_ttl_days=30,
        yp_clusters={
            YpCluster(pod_set_id=cfg.endpoint_set_id, cluster='sas'),
            YpCluster(pod_set_id=cfg.endpoint_set_id, cluster='myt'),
            YpCluster(pod_set_id=cfg.endpoint_set_id, cluster='iva'),
            YpCluster(pod_set_id=cfg.endpoint_set_id, cluster='vla'),
            YpCluster(pod_set_id=cfg.endpoint_set_id, cluster='man'),
        },
        port=cfg.port,
    )

    service = Service(
        id="upravlyator", project_id=project.id, name="upravlyator", port=cfg.port, path="/metrics", interval=60, grid=0
    )

    shard = Shard(
        id=f"{project.id}.{cluster.id}.{service.id}",
        project_id=project.id,
        cluster_id=cluster.id,
        service_id=service.id,
        sensors_ttl_days=cluster.sensors_ttl_days,
        sensor_name_label="sensor",
    )
    solo_objects = [project, cluster, service, shard]

    """
    cos_graph = Graph(
        id=f"{cos.name}_graph",
        project_id=solo_example_project.id,
        name=f"{cos.name.capitalize()} sensor Graph",
        min="-1",
        max="1",
        transform="NONE",
        hide_no_data=False,
        normalize=False,
        ignore_inf=False,
        max_points=0,
        downsampling="BY_POINTS",
        scale="NATURAL",
        over_lines_transform="NONE",
        elements={
            Element(
                title=f"{cos.name.capitalize()}",
                selectors={Selector(name="sensor", value="{}".format(cos.name))}
            )
        }
    )
    """

    juggler_channel = Channel(
        id=f"{cluster.id}_jc",
        project_id=project.id,
        name="Juggler",
        method=Method(
            juggler=Juggler(
                host=make_host_selector(project.id, cluster.id),
                service='{{{annotations.service}}}',
                description=""
            )
        ),
        notify_about_statuses={"OK", "ERROR", "ALARM", "WARN"},
        repeat_notify_delay_millis=5 * 60 * 1000,
    )
    solo_objects.append(juggler_channel)

    multialert_juggler_channel = Channel(
        id=f"{cluster.id}_mjc",
        project_id=project.id,
        name="multialert juggler channel",
        method=Method(
            juggler=Juggler(
                host=make_host_selector(project.id, cluster.id),
                service='{{{annotations.service}}}',
                instance="{{{alert.id}}}",
                description="",
            )
        ),
        notify_about_statuses={"OK", "ERROR", "ALARM", "WARN"},
        repeat_notify_delay_millis=5 * 60 * 1000,
    )
    solo_objects.append(multialert_juggler_channel)
    prepare_text(
        """
        Alert: {{{alert.name}}}
        MultiAlert: {{{{alert.parent.id}}}_{{{alert.id}}},
        URL: {{{url}}}
        {{#annotationsList}}
        {{{key}}}: {{{value}}}
        {{/annotationsList}}
    """
    )
    # host_selector = make_host_selector(project.id, cluster.id)

    for resource_spec in db.resource_spec.values():
        if not resource_spec.AlertGatherPeriodSec:
            continue

        if not resource_spec.is_used():
            continue

        metric_resource_spec_last_resource_source_age = Sensor(
            project=project.id,
            cluster=cluster.name,
            service=service.name,
            host='leader',
            sensor='resource_spec.last_resource.source_age',
            id=resource_spec.Id,
            kind="IGAUGE",
        )

        resource_gather_id = make_alert_id(cluster.id, "resource_gather", resource_spec.Id)
        resource_gather_selector = make_service_selector("resource_gather", resource_spec.Id)
        resource_gather_alert = Alert(
            id=resource_gather_id[64:],
            name=f"Resource {resource_spec.Name} gather timeout",
            project_id=project.id,
            type=Type(
                threshold=Threshold(
                    selectors=metric_resource_spec_last_resource_source_age.selectors,
                    time_aggregation="LAST_NON_NAN",
                    predicate="GT",
                    threshold=resource_spec.AlertGatherPeriodSec,
                    predicate_rules=[
                        PredicateRule(
                            threshold_type="LAST_NON_NAN", comparison="GT", threshold=resource_spec.AlertGatherPeriodSec
                        ),
                    ],
                )
            ),
            window_secs=int(datetime.timedelta(minutes=5).total_seconds()),
            notification_channels={juggler_channel.id},
            annotations={'service': resource_gather_selector},
        )
        solo_objects.append(resource_gather_alert)

        metric_box_applied_resource_source_age = Sensor(
            project=project.id,
            cluster=cluster.name,
            service=service.name,
            host='leader',
            sensor='box_applied.resource_spec.active_resource.source_age',
            id=resource_spec.Id,
            kind="IGAUGE",
        )
        box_applied_resource_source_age_id = make_alert_id(cluster.id, "box_resource_source_age", resource_spec.Id)
        box_applied_resource_source_age_service = make_service_selector("box_resource_source_age", resource_spec.Id)
        box_applied_resource_source_age_alert = MultiAlert(
            id=box_applied_resource_source_age_id[64:],
            name=f"Resource {resource_spec.Name} too old",
            project_id=project.id,
            type=Type(
                threshold=Threshold(
                    selectors=metric_box_applied_resource_source_age.selectors,
                    time_aggregation="LAST_NON_NAN",
                    predicate="GT",
                    threshold=resource_spec.AlertSourceAgeSec,
                    predicate_rules=[
                        PredicateRule(
                            threshold_type="LAST_NON_NAN",
                            comparison="GT",
                            threshold=resource_spec.AlertSourceAgeSec,
                        ),
                    ],
                )
            ),
            window_secs=int(datetime.timedelta(minutes=5).total_seconds()),
            notification_channels={multialert_juggler_channel.id},
            annotations={"service": box_applied_resource_source_age_service},
            group_by_labels={"fqdn"},
        )
        solo_objects.append(box_applied_resource_source_age_alert)

    for resource_set_spec in db.resource_set_spec.values():
        if not resource_set_spec.is_used():
            continue

        add_resource_set_gather_check(
            solo_objects=solo_objects,
            project=project,
            cluster=cluster,
            service=service,
            resource_set_spec=resource_set_spec,
        )
        add_resource_set_source_age_check(
            solo_objects=solo_objects,
            project=project,
            cluster=cluster,
            service=service,
            resource_set_spec=resource_set_spec,
        )
        add_resource_set_validation_alert_with_check(
            solo_objects=solo_objects,
            project=project,
            cluster=cluster,
            service=service,
            resource_set_spec=resource_set_spec,
            juggler_channel=juggler_channel,
            program_func=alert_last_greater_than_theshold,
            status=dostavlyator_proto.main_pb2.EStatus.BAD,
            alert_description='validation failed',
            window=datetime.timedelta(minutes=5),
        )
        add_resource_set_validation_alert_with_check(
            solo_objects=solo_objects,
            project=project,
            cluster=cluster,
            service=service,
            resource_set_spec=resource_set_spec,
            juggler_channel=juggler_channel,
            program_func=alert_min_greater_than_theshold,
            status=dostavlyator_proto.main_pb2.EStatus.TESTING,
            alert_description='has been TESTING for more than 2 hours',
            window=datetime.timedelta(hours=2),
        )

    try:
        controller.process(solo_objects)
    except Exception as e:
        log.error(e)


def add_resource_set_gather_check(
    *,
    solo_objects,
    project,
    cluster,
    service,
    resource_set_spec,
):
    return add_resource_set_aggregated_check(
        solo_objects = solo_objects,
        project = project,
        cluster = cluster,
        service = service,
        check_name = "resource_set_gather",
        child_check_name = "resource_gather",
        resource_set_spec = resource_set_spec
    )


def add_resource_set_source_age_check(
    *,
    solo_objects,
    project,
    cluster,
    service,
    resource_set_spec,
):
    resource_set_gather_depend = make_unreach_depend(
        host = make_host_selector(project.id, cluster.id),
        service = make_service_selector('resource_set_gather', resource_set_spec.Id)
    )
    unreach_hold = min(180*60, resource_set_spec.MaxDeployDurationSec)
    return add_resource_set_aggregated_check(
        solo_objects = solo_objects,
        project = project,
        cluster = cluster,
        service = service,
        check_name = "box_resource_set_source_age",
        child_check_name = "box_resource_source_age",
        resource_set_spec = resource_set_spec,
        unreach_service = [{'check': resource_set_gather_depend, 'hold': unreach_hold}],
    )


def add_resource_set_aggregated_check(
    *,
    solo_objects,
    project,
    cluster,
    service,
    check_name,
    child_check_name,
    resource_set_spec,
    unreach_checks = None,
    unreach_service = None
):
    resource_selectors = [make_service_selector(child_check_name, resource_spec.Id) for resource_spec in resource_set_spec.resource_spec if resource_spec.AlertGatherPeriodSec]
    if not resource_selectors:
        return
    child_service_selector = '|'.join([f'service={selector}' for selector in resource_selectors])
    host_selector = make_host_selector(project.id, cluster.id)
    service_selector = make_service_selector(check_name, resource_set_spec.Id)
    aggregator_kwargs = {'nodata_mode': 'skip'}
    if unreach_checks:
        aggregator_kwargs['unreach_checks'] = unreach_checks
    if unreach_service:
        aggregator_kwargs['unreach_mode'] = 'force_ok'
        aggregator_kwargs['unreach_service'] = unreach_service
    check = Check(
        namespace = f"{project.id}",
        host = host_selector,
        service = service_selector,
        refresh_time = 60,
        ttl = 600,
        aggregator = "logic_or",
        aggregator_kwargs = aggregator_kwargs,
        children = [
            Child(
                host = f"(host={host_selector}) & ({child_service_selector})",
                service = "all",
                instance = "all",
                group_type = "EVENTS",
            )
        ],
    )
    solo_objects.append(check)


def add_resource_set_validation_alert_with_check(
    *,
    solo_objects,
    project,
    cluster,
    service,
    resource_set_spec,
    juggler_channel,
    program_func,
    status,
    alert_description,
    window,
):
    status_name = spec.ValidationStatusName(status)
    validation_failed_sensor = Sensor(
        project=project.id,
        cluster=cluster.name,
        service=service.name,
        host='leader',
        sensor=validation_status_sensor_name(status),
        id=resource_set_spec.Id,
        kind="IGAUGE",
    )

    resource_set_validation_service = make_service_selector(f"validation.{status_name}", resource_set_spec.Id)
    resource_set_validation_id = make_alert_id(cluster.id, f"validation.{status_name}", resource_set_spec.Id)
    alert = Alert(
        id=resource_set_validation_id[:64],
        name=f"{cluster.id} ResourceSetSpec {resource_set_spec.Name} {alert_description}",
        project_id=project.id,
        type=Type(
            expression=Expression(program=program_func(validation_failed_sensor, threshold=0))
        ),
        window_secs=int(window.total_seconds()),
        notification_channels={juggler_channel.id},
        annotations={'service': resource_set_validation_service},
    )
    check = Check(
        namespace=f"{project.id}",
        host=make_host_selector(project.id, cluster.id),
        service=resource_set_validation_service,
        refresh_time=60,
        ttl=600,
        aggregator="logic_or",
    )
    solo_objects.append(alert)
    solo_objects.append(check)


def alert_last_greater_than_theshold(sensor, threshold):
    return """
        let metric = {sensor};
        alarm_if(last(metric) > {threshold});
    """.format(
        sensor=sensor, threshold=threshold
    )


def alert_min_greater_than_theshold(sensor, threshold):
    return """
        let metric = {sensor};
        alarm_if(min(metric) > {threshold});
    """.format(
        sensor=sensor, threshold=threshold
    )


def make_host_selector(project_id, cluster_id):
    return f"{project_id}.{cluster_id}"


def make_service_selector(alert_name, object_id):
    return f"upravlyator.{alert_name}.{object_id}"


def make_alert_id(cluster_id, alert_name, object_id):
    return f"{cluster_id}.{alert_name}.{object_id}"


def make_unreach_depend(*, host, service):
    return f"{host}:{service}"
