from copy import deepcopy
from time import sleep
from json import dumps as json_dumps
from yt.wrapper.config import get_config, set_command_param, del_command_param
import yt.wrapper as yt
from yt.common import YtResponseError

from infra.dostavlyator.lib.misc.misc import GetLogger, get_instance_details

log = GetLogger('dostavlyator.upravlyator.leader')


class WithTransaction:
    def __init__(self, yt_tx, abort_on_exception=True):
        self._yt_tx = yt_tx
        self._abort_on_exception = abort_on_exception

    def __enter__(self) -> None:
        self._hack(self._yt_tx._client, self._yt_tx.transaction_id, self._yt_tx._ping_ancestor_transactions)

    def __exit__(self, type, value, traceback) -> None:
        transaction_id, ping_ancestor_transactions = self._yt_tx._stack.get()
        self._hack(self._yt_tx._client, transaction_id, ping_ancestor_transactions)
        if type:
            self._yt_tx.abort()

    def _hack(self, yt_client, transaction_id, ping_ancestor_transactions) -> None:
        if transaction_id is None:
            del_command_param("transaction_id", yt_client)
        else:
            set_command_param("transaction_id", transaction_id, yt_client)
        if ping_ancestor_transactions is None:
            del_command_param("ping_ancestor_transactions", yt_client)
        else:
            set_command_param("ping_ancestor_transactions", ping_ancestor_transactions, yt_client)


class TLeaderElection:
    def __init__(self, yt_client, yt_base_path) -> None:
        # self._yt_client = yt.YtClient(proxy.yt_cluster.proxy, token=yt_cluster.token, config=config=deepcopy(get_config(yt_client)))
        self._yt_client = yt.YtClient(config=deepcopy(get_config(yt_client)))
        self._yt_leader_path = yt_base_path + '/leader'
        self._yt_tx = None
        self._is_leader = False

    def __del__(self) -> None:
        self.abort()

    def abort(self) -> None:
        if self._yt_tx:
            try:
                log.debug('Remove leader lock')
                self._yt_tx.abort()
            except:
                log.error('exception while yt_tx.abort()')
        self._yt_tx = None
        self._is_leader = False

    def is_leader(self) -> bool:
        if self._is_leader and not self._yt_tx.is_pinger_alive():
            self.abort()
        return self._is_leader

    def check(self) -> None:
        if not self.is_leader():
            raise Exception('not a leader')  # TODO: exception class

    def become_a_leader(self) -> None:
        if self.is_leader():
            raise Exception('allready leader')  # TODO: exception class
        try:
            self._yt_client.create(type='string_node', path=self._yt_leader_path, attributes={})
        except YtResponseError as e:
            if e.inner_errors[0]['code'] not in {
                501,
            }:  # skip error 501='Node already exists'
                raise
        if self._yt_tx:
            self.abort()
        self._yt_tx = self._yt_client.Transaction()  # TODO: timeout
        with WithTransaction(self._yt_tx):
            lock_id = self._yt_client.lock(path=self._yt_leader_path, mode='exclusive', waitable=True)['lock_id']
            while True:
                lock_state = self._yt_client.get('//sys/locks/' + lock_id + '/@state')
                if lock_state == 'acquired':
                    instance_details = get_instance_details()
                    self._yt_client.set(path=self._yt_leader_path, value=json_dumps(instance_details, sort_keys=True))
                    self._is_leader = True
                    return
                log.info('Other leader exist')
                sleep(1)
