from concurrent.futures import ThreadPoolExecutor
from collections import defaultdict
from time import time

from infra.dostavlyator.lib.data import spec
from infra.dostavlyator.lib.misc.misc import GetLogger
from infra.dostavlyator.lib.db.db import TUpdatedObjects
from infra.dostavlyator.lib.upravlyator.sources.register import GetResourceCandidate

log = GetLogger('dostavlyator.lib.resource_collector')


class TResourceCollector:
    def __init__(self):
        self.executor = ThreadPoolExecutor(max_workers=100)  # TODO: magic number
        self.gather_tasks = dict()  # (gather_uri, gather_opt) -> Future
        self.gather_timestamps = dict()  # (gather_uri, gather_opt) -> imestamp: float
        self.resource_candidates = defaultdict(list)

    def Gather(self, db) -> TUpdatedObjects:
        result = TUpdatedObjects()
        with log.debug('TResourceCollector::Gather()'):
            # 1. GatherResourceCandidates
            resource_spec_candidates = self.GatherResourceCandidate(db)
            for resource_spec, resource_candidates_proto in resource_spec_candidates.items():
                # 2. RegisterResourceCandidate
                resource_candidates = self.RegisterResourceCandidate(
                    db=db, resource_spec=resource_spec, resource_candidates_proto=resource_candidates_proto
                )
                # 3. RegisterResource
                result += self.RegisterResource(
                    db=db, resource_spec=resource_spec, resource_candidates=resource_candidates
                )
            return result

    def GatherResourceCandidate(self, db) -> dict:
        with log.debug('TResourceCollector::GatherResourceCandidate()'):
            now = time()
            # 1. Start gathering
            for resource_spec in db.resource_spec.values():
                uri = resource_spec.GatherUri
                opt = resource_spec.GatherUriOpt
                key = (uri, opt)
                if key in self.gather_tasks:
                    log.debug(f'TResourceSpec(Id={resource_spec.Id}): Skip. (In progress)')
                elif resource_spec.GatherPeriodSec == 0 and bool(resource_spec.resource):
                    log.debug(f'TResourceSpec(Id={resource_spec.Id}): Skip. (GatherPeriodSec = 0)')
                elif now - self.gather_timestamps.get(key, 0) < resource_spec.GatherPeriodSec:
                    log.debug(f'TResourceSpec(Id={resource_spec.Id}): Skip. (GatherPeriodSec)')
                else:
                    log.debug(f'TResourceSpec(Id={resource_spec.Id}): Start.')
                    self.gather_tasks[(uri, opt)] = self.executor.submit(
                        GetResourceCandidate, gather_uri=uri, gather_uri_opt=opt
                    )
            # 2. Get result from ready tasks
            ready_tasks = [key for key, future in self.gather_tasks.items() if future.done()]
            for key in ready_tasks:
                try:
                    future = self.gather_tasks.pop(key)
                    resource_candidates_proto = future.result()
                    self.gather_timestamps[key] = now
                    self.resource_candidates[key].extend(resource_candidates_proto)
                except Exception as e:
                    log.error(f'Gather error for {key}: {e}')

            # 3. Generate and return result
            result = dict()
            for key, resource_candidates_proto in self.resource_candidates.items():
                uri, opt = key
                for resource_spec in db.resource_spec.values():
                    if resource_spec.GatherUri == uri and resource_spec.GatherUriOpt == opt:
                        result[resource_spec] = resource_candidates_proto
            return result

    def Clear(self):
        self.resource_candidates.clear()

    def RegisterResourceCandidate(self, db, resource_spec, resource_candidates_proto) -> list:
        result = list()
        with log.debug(f'TResourceCollector::RegisterResourceCandidate(TResourceSpec(Id={resource_spec.Id}))'):
            if not resource_candidates_proto:
                log.error(f'No TResourceCandidate(s) found for TResourceSpec(Id={resource_spec.Id})')
            for resource_candidate_proto in resource_candidates_proto:
                if resource_candidate_proto.Id in db.resource_candidate:
                    resource_candidate_proto.ManualStatus = db.resource_candidate[resource_candidate_proto.Id].ManualStatus
                    resource_candidate_proto.CreationTime = db.resource_candidate[resource_candidate_proto.Id].CreationTime
                db.UpdateResourceCandidate(proto=resource_candidate_proto)
                result.append(db.resource_candidate[resource_candidate_proto.Id])  # we should create TResource from TResoucreCandidate even if TResoucreCandidate allready exists
            return result

    def RegisterResource(self, db, resource_spec, resource_candidates) -> TUpdatedObjects:
        result = TUpdatedObjects()
        with log.debug(f'TResourceCollector::RegisterResource(TResourceSpec(Id={resource_spec.Id}))'):
            for resource_candidate in resource_candidates:
                proto = spec.TResource(
                    ResourceSpecId=resource_spec.Id,
                    ResourceCandidateId=resource_candidate.Id,
                    Version=resource_spec.GetLastVersion() + 1,
                )
                if proto.Id not in db.resource:
                    result += db.UpdateResource(proto=proto)
                else:
                    log.debug(f'skip TResource(Id={proto.Id})')
            if not resource_candidates:
                log.debug('done')
            return result
