
from infra.dostavlyator.proto import main_pb2
from infra.dostavlyator.lib.solo.solo import init as solo_init, register as solo_register
from infra.dostavlyator.lib.solo.metrics import start_server as solo_start_server
from infra.dostavlyator.lib.data import spec
from infra.dostavlyator.lib.db.db import DB, TUpdatedObjects
from infra.dostavlyator.lib.db import tables, control
from infra.dostavlyator.lib.misc.misc import GetLogger
from infra.dostavlyator.lib.upravlyator.leader import TLeaderElection
from infra.dostavlyator.lib.upravlyator.resource_collector import TResourceCollector

from yt.wrapper.errors import YtCypressTransactionLockConflict
from time import sleep
from copy import copy
from collections import defaultdict

log = GetLogger('dostavlyator.lib.upravlyator')
EStatus = main_pb2.EStatus


class TUpravlyator:
    def __init__(self, abc_service, yt_client, base_path):
        self.abc_service = abc_service
        self._yt_client = yt_client
        self._yt_base_path = base_path
        self._leader_election = TLeaderElection(self._yt_client, self._yt_base_path)
        self._db = None
        self._resource_collector = TResourceCollector()

    def become_a_leader(self):
        while True:
            try:
                if self._leader_election.is_leader():
                    return
                self._db = DB(yt_client=self._yt_client, base_path=self._yt_base_path)
                solo_start_server(host='::', port=8080, db=self._db)
                self._leader_election.become_a_leader()
                return
            except YtCypressTransactionLockConflict:
                log.debug('other leader detected')
            except Exception:
                self._leader_election.abort()
                log.debug('exception while TUpravlyator::become_a_leader()')
            sleep(1)  # TODO: timeout

    def Run(self):
        solo_init(yt_cluster=self._yt_client.config['proxy']['url'], yt_base_path=self._yt_base_path)
        while True:
            try:
                self.become_a_leader()
                self.Tick()
            except Exception as e:
                log.error('Critical error in TUpravlyator::Tick()')
                log.error(e)
                self._leader_election.abort()
                self._db = None
            tick_timeout = 10
            log.info(f'Sleep {tick_timeout} seconds...')
            sleep(tick_timeout)

    def Tick(self):
        with log.debug('TUpravlyator::Tick'):
            self._leader_election.check()
            self._db.ReadControlFlagsNode()
            if self._db.control_flags.freeze:
                log.info("Freeze!")
                return
            if self._db.control_flags.wipe:
                tables.RemoveAllTables(self._yt_client, self._yt_base_path)
                tables.CreateAllTables(self._yt_client, self._yt_base_path)
                control.freeze(self._yt_client, self._yt_base_path)
                self._db = DB(yt_client=self._yt_client, base_path=self._yt_base_path)
                solo_start_server(host='::', port=8080, db=self._db)
                return
            with self._yt_client.Transaction(type="tablet"):
                # 1. Read updates from DB
                updated_objects = self._db.ReadTables()

                # 2. Aggregate validation statuses
                updated_objects += self.AggregateValidation()

                # 3. Cleanup unused objects
                self._db.Cleanup()

                # 4. Gather new TResourceCandidate, TResource
                updated_objects += self._resource_collector.Gather(self._db)

                # 5. Expand tree of objects
                for resource in updated_objects.resource:
                    updated_objects.resource_set.update(resource.resource_set)

                for resource in updated_objects.resource:
                    updated_objects.resource_spec.add(resource.resource_spec)

                for resource_set in updated_objects.resource_set:
                    updated_objects.resource_set_spec.add(resource_set.resource_set_spec)

                for resource_spec in updated_objects.resource_spec:
                    updated_objects.resource_set_spec.update(resource_spec.resource_set_spec)

                # 6. Generate TResourceSet
                for resource_set_spec in updated_objects.resource_set_spec:
                    try:
                        updated_objects += self.GenerateResourceSet(resource_set_spec.Id)
                    except Exception as e:
                        log.error(e)

                # 7. Mark boxes to regenerate assignements
                for resource_set_spec in updated_objects.resource_set_spec:
                    # fqdns = [box_requested.DeployPodPersistentFqdn for box_requested in resource_set_spec.box_requested]
                    # log.debug(f'regenerate TBoxRequested({fqdns}) by TResourceSetSpec(Id={resource_set_spec.Id})')
                    updated_objects.box_requested.update(resource_set_spec.box_requested)
                updated_objects.box_requested.update(self._db.box_requested[box_applied.DeployPodPersistentFqdn] for box_applied in updated_objects.box_applied if box_applied.DeployPodPersistentFqdn in self._db.box_requested)

                with log.debug('Generate BoxAssigned:'):
                    for box_requested in updated_objects.box_requested:
                        self.GenerateBoxAssigned(box_requested)
                    log.debug('done')

                # i'm a leader?
                self._leader_election.check()
            # transaction commited and closed
            self._resource_collector.Clear()

            solo_register(self.abc_service, port=8080, db=self._db)
            solo_start_server(host='::', port=8080, db=self._db)

    def AggregateValidation(self) -> TUpdatedObjects:
        with log.debug('TUpravlyator::AggregateValidation()'):
            result = TUpdatedObjects()
            validation_result = defaultdict(lambda: defaultdict(set))
            for box_applied in self._db.box_applied.values():
                for resource_set in box_applied.Spec.ActiveResourceSet:
                    resource_set_id = resource_set.ResourceSetId
                    resource_set_status = resource_set.ValidationStatus
                    validation_result[resource_set_id][resource_set_status].add(box_applied.DeployPodPersistentFqdn)

            for resource_set_id, stat in validation_result.items():
                if resource_set_id not in self._db.resource_set:
                    continue
                resource_set = self._db.resource_set[resource_set_id]
                if resource_set.ValidationStatus != EStatus.TESTING:
                    continue
                resource_set_spec = resource_set.resource_set_spec

                if EStatus.BAD in stat:
                    log.info(
                        f'TResourceSet(Id={resource_set_id}): Validation failed. Mark BAD. Pods: {stat[EStatus.BAD]}'
                    )
                    proto = copy(resource_set._proto)
                    proto.ValidationStatus = EStatus.BAD
                    self._db.UpdateResourceSet(proto)
                    result.resource_set.add(resource_set)
                    continue

                approves = len(stat[EStatus.VALID])
                required_approves = resource_set_spec.GetRequiredValidationApprovesCount()
                if approves >= required_approves:
                    log.debug(f'TResourceSet(Id={resource_set_id}): {approves}/{required_approves} mark GOOD')
                    proto = copy(resource_set._proto)
                    proto.ValidationStatus = EStatus.VALID
                    self._db.UpdateResourceSet(proto)
                    result.resource_set.add(resource_set)
                    continue
                log.debug(f'TResourceSet(Id={resource_set_id}): {approves}/{required_approves}')
            return result

    def GenerateResourceSet(self, resource_set_spec_id) -> TUpdatedObjects:
        with log.debug(f'TUpravlyator::GenerateResourceSet(TResourceSetSpec(Id={resource_set_spec_id})'):
            resource_set_spec = self._db.resource_set_spec[resource_set_spec_id]
            resources = [resource_spec.GetLastGoodOrThrow() for resource_spec in resource_set_spec.resource_spec]
            resource_ids = [resource.Id for resource in resources]
            proto = spec.TResourceSet(
                ResourceSetSpecId=resource_set_spec.Id,
                Path=resource_set_spec.Path,
                Version=resource_set_spec.GetLastVersion() + 1,
                Resource=resource_ids,
                ValidationStatus=main_pb2.EStatus.TESTING,
            )
            last = resource_set_spec.GetLast()
            if last and proto.Id == last.Id:
                result = TUpdatedObjects()
                result.resource_set.add(last)
                return result
            return self._db.UpdateResourceSet(proto=proto)

    def GenerateBoxAssigned(self, box_requested):
        with log.debug(f'TUpravlyator::GenerateBoxAssigned({box_requested.DeployPodPersistentFqdn}, SpecId={box_requested.Spec.SpecId})'):
            fqdn = box_requested.DeployPodPersistentFqdn
            is_validator = True
            selected_to_download = set()
            selected_to_activate = set()
            for resource_set_spec_proto in box_requested.Spec.ResourceSet:
                resource_set_spec = self._db.resource_set_spec[resource_set_spec_proto.Id]
                with log.debug(f'Select TResourceSet for TResourceSetSpec(Id={resource_set_spec.Id})'):
                    try:
                        is_validator = resource_set_spec_proto.Validator
                        is_prefetch = resource_set_spec_proto.Prefetch
                        if resource_set_spec.Id in self._db.control_pin_resource_set:
                            resource_set_id = self._db.control_pin_resource_set[resource_set_spec.Id]
                            if resource_set_id not in self._db.resource_set:
                                raise Exception(f"Do not have TResourceSet(Id={resource_set_id}) for TResourceSetSpec(Id={resource_set_spec.Id}). Abort")
                            log.debug(f'Append pinned TResourceSet(Id={resource_set_id})')
                            selected_to_activate.add(self._db.resource_set[resource_set_id])
                            continue
                        valid_resource_set = resource_set_spec.GetLastValid()
                        testing_resource_set = resource_set_spec.GetLastTesting()
                        if not valid_resource_set and not testing_resource_set:
                            raise Exception(f'no GOOD TResourceSet for TResourceSetSpec(Id={resource_set_spec.Id})')
                        if valid_resource_set:  # we always append valid TResourceSet
                            log.debug(f'Append VALID TResourceSet(Id={valid_resource_set.Id})')
                            selected_to_activate.add(valid_resource_set)
                        if not testing_resource_set or (valid_resource_set and testing_resource_set.Version < valid_resource_set.Version):  # if testing TResourceSet version older than valid then nothing to do
                            continue
                        if is_validator or is_prefetch:
                            log.debug(f'Append TResource to download for TESTING TResourceSet(Id={testing_resource_set.Id})')
                            selected_to_download.add(testing_resource_set)
                        if is_validator:
                            is_resources_ready = fqdn in self._db.box_applied and self._db.box_applied[fqdn].ready_resources >= set(resource.Id for resource in testing_resource_set.resource)
                            if not is_resources_ready:
                                log.debug(f'Not all resources are ready')
                                continue
                            log.debug(f'All TResource are ready for TESTING TResourceSet(Id={testing_resource_set.Id})')
                            validators_required = resource_set_spec.GetRequiredValidationApprovesCount()
                            validators_count = 0
                            for box_assigned in self._db.box_assigned.values():
                                if box_assigned.DeployPodPersistentFqdn == fqdn:
                                    continue  # don't count self
                                if testing_resource_set.Id in set(resource_set.Id for resource_set in box_assigned.Spec.ResourceSet):
                                    validators_count += 1
                            log.debug(f'Current validators_count={validators_count}, validators_required={validators_required}')
                            is_more_validators_required = not valid_resource_set or (validators_count < validators_required)
                            if is_more_validators_required:
                                log.debug(f'Append TESTING TResourceSet(Id={testing_resource_set.Id}) for validation')
                                selected_to_activate.add(testing_resource_set)
                    except Exception as e:
                        log.error(e)

            selected_to_download.update(selected_to_activate)
            selected_to_download = sorted(selected_to_download, key=lambda x: x.CreationTime)
            resources = set()

            # expand resources to resources_proto
            resources_proto = set()
            for resoucre_set in selected_to_download:
                for resource in resoucre_set.resource:
                    if resource in resources:
                        continue
                    resources.add(resource)
                    r_proto = spec.TResource(
                        Id=resource.Id,
                        Path=resource.resource_spec.Path,
                        Name=resource.resource_spec.Name,
                        ResourceSpecId=resource.ResourceSpecId,
                        ResourceCandidateId=resource.ResourceCandidateId,
                        Version=resource.Version,
                        Source=resource.resource_candidate.Source,
                    )
                    resources_proto.add(r_proto)

            # create YT record
            box_assigned_spec = main_pb2.TBoxAssignedSpec(
                Resource=list(resources_proto),
                ResourceSet=[resource_set._proto for resource_set in selected_to_activate],
                SpecId=box_requested.Spec.SpecId,
            )
            box_assigned = tables.TBoxAssignedTable(
                DeployPodPersistentFqdn = fqdn,
                DeployPodId=box_requested.DeployPodId,
                DeployNodeDC=box_requested.DeployNodeDC,
                DeployProjectId=box_requested.DeployProjectId,
                DeployStageId=box_requested.DeployStageId,
                DeployUnitId=box_requested.DeployUnitId,
                DeployBoxId=box_requested.DeployBoxId,
                Spec=box_assigned_spec,
            )
            self._db.UpdateBoxAssigned(box_assigned)
