import logging
import itertools
from datetime import datetime, timedelta

import requests
import pytest


logging.captureWarnings(True)


@pytest.fixture
def sess():
    return requests.session()


@pytest.fixture(scope='session')
def now():
    return datetime.utcnow() - timedelta(minutes=10)  # without shift ydb may append new records with reordering


@pytest.fixture(scope='session')
def awacs_end_date():
    return datetime.fromisoformat('2021-08-21 23:59:59')


@pytest.mark.parametrize('key_type', [
    'LOG_LEVEL',
    'HOST',
    'POD',
    'BOX',
    'WORKLOAD',
    'CONTAINER_ID',
    'LOGGER_NAME',
    'POD_TRANSIENT_FQDN',
    'POD_PERSISTENT_FQDN',
    'NODE_FQDN',
    'THREAD_NAME',
    'REQUEST_ID',
])
@pytest.mark.parametrize('full_search', [False, True])
def test_suggest(sess, key_type, full_search, now):
    d = {
        'request': {
            'project_id': 'maps-front-maps',
            'stage_id': 'maps-front-maps_production',
            'deploy_unit_id': 'app',
            'timestamp_range': {
                'begin': (now - timedelta(hours=3)).isoformat() + 'Z',
                'end': now.isoformat() + 'Z'
            },
            'limit': 50,
            'order': 'DESC',
        },
    }

    url = 'http://localhost:8080/api/logs/GetSearchQuerySuggests/?_snake_case=true'

    d['key_type'] = key_type
    d['full_search'] = full_search

    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp = r.json()

    assert rsp['candidates']


def test_search(sess, now):
    d = {
        'project_id': 'maps-front-maps',
        'stage_id': 'maps-front-maps_production',
        'deploy_unit_id': 'app',
        'timestamp_range': {
            'begin': (now - timedelta(hours=3)).isoformat() + 'Z',
            'end': now.isoformat() + 'Z'
        },
        'limit': 50,
        'order': 'DESC',
    }

    url = 'http://localhost:8080/api/logs/SearchLogEntries/?_snake_case=true'

    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp = r.json()
    assert rsp['log_entries']
    assert rsp['continuation_tokens']
    assert rsp['continuation_tokens']['forward']
    assert rsp['continuation_tokens']['backward']
    assert rsp['continuation_tokens']['current']

    d['continuation_token'] = rsp['continuation_tokens']['current']
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()

    assert rsp2['log_entries']
    for entry1, entry2 in zip(rsp['log_entries'], rsp2['log_entries']):
        assert entry1 == entry2
    assert rsp2['continuation_tokens']['forward'] == rsp['continuation_tokens']['forward']
    assert rsp2['continuation_tokens']['backward'] == rsp['continuation_tokens']['backward']
    assert rsp2['continuation_tokens']['current'] == rsp['continuation_tokens']['current']

    d['continuation_token'] = rsp['continuation_tokens']['forward']
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()

    assert rsp2['log_entries']
    e1 = rsp['log_entries'][-1]
    e1ts = datetime.fromisoformat(e1['timestamp'].rstrip('Z')).timestamp()
    e2 = rsp2['log_entries'][0]
    e2ts = datetime.fromisoformat(e2['timestamp'].rstrip('Z')).timestamp()
    assert (-e1ts, e1['container_id'], e1['seq']) < (-e2ts, e2['container_id'], e2['seq'])

    del d['continuation_token']
    d['search_pattern_list'] = {
        'select_type': 'INCLUDE',
        'values': ['beuwbewobewiovweniovbewio'],
    }
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()
    assert not rsp2['log_entries']

    d['search_pattern_list']['values'] = ['tskv']
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()
    assert rsp2['log_entries']

    d['continuation_token'] = rsp2['continuation_tokens']['forward']
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()
    assert rsp2['log_entries']


def test_awacs_search(sess, awacs_end_date):
    d = {
        'timestamp_range': {
            'begin': (awacs_end_date - timedelta(days=2)).isoformat() + 'Z',
            'end': awacs_end_date.isoformat() + 'Z'
        },
        'limit': 1,
        'order': 'DESC',
        'namespace': 'anttsov-test-balancer-ns',
        'domain_list':
        {
            'select_type': 'EXCLUDE',
            'values': ['1234']
        },
        'header_fields_list': [
            {
                'path': 'X-TEST',
                'values': ['test1', 'test2'],
            }
        ],
        'process_time_list': {
            'values': [0., 1.],
        },
        'request_list': {
            'select_type': 'EXCLUDE',
            'values': ['1234'],
            'op_type':'GREP'
        }
    }

    url = 'http://[::1]:8080/api/v2/awacs/logs/SearchLogEntries/?_snake_case=true'

    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp = r.json()
    assert rsp['log_entries']
    assert rsp['continuation_tokens']
    assert rsp['continuation_tokens']['forward']
    assert rsp['continuation_tokens']['backward']
    assert rsp['continuation_tokens']['current']

    d['continuation_token'] = rsp['continuation_tokens']['current']
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()

    assert rsp2['log_entries']
    for entry1, entry2 in zip(rsp['log_entries'], rsp2['log_entries']):
        assert entry1 == entry2
    assert rsp2['continuation_tokens']['forward'] == rsp['continuation_tokens']['forward']
    assert rsp2['continuation_tokens']['backward'] == rsp['continuation_tokens']['backward']
    assert rsp2['continuation_tokens']['current'] == rsp['continuation_tokens']['current']

    d['continuation_token'] = rsp['continuation_tokens']['forward']
    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp2 = r.json()

    assert rsp2['log_entries']
    e1 = rsp['log_entries'][-1]
    e1ts = datetime.fromisoformat(e1['timestamp'].rstrip('Z')).timestamp()
    e2 = rsp2['log_entries'][0]
    e2ts = datetime.fromisoformat(e2['timestamp'].rstrip('Z')).timestamp()
    assert (-e1ts, e1['hostname'], e1['pushclient_row_id']) < (-e2ts, e2['hostname'], e2['pushclient_row_id'])


@pytest.mark.parametrize('key_type', [
    'DOMAIN_',
    'UPSTREAM',
    'CLIENT_IP',
    'CLIENT_PORT',
    'HOSTNAME',
    # 'COOKIES',
    # 'HEADERS',
    'YANDEXUID',
    'METHOD',
    'PROCESS_TIME',
    'REASON',
    # 'REPORTS_UUIDS',
    'REQUEST',
    'REQUEST_ID',
    'STATUS',
    # 'WORKFLOW',
])
@pytest.mark.parametrize('full_search', [False, True])
def test_awacs_suggest(sess, key_type, full_search, awacs_end_date):
    d = {
        'request': {
            'namespace': 'anttsov-test-balancer-ns',
            'timestamp_range': {
                'begin': (awacs_end_date - timedelta(days=2)).isoformat() + 'Z',
                'end': awacs_end_date.isoformat() + 'Z'
            },
            'limit': 5,
            'order': 'DESC',
        },
    }

    url = 'http://localhost:8080/api/v2/awacs/logs/GetSearchQuerySuggests/?_snake_case=true'

    d['key_type'] = key_type
    d['full_search'] = full_search

    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp = r.json()

    assert rsp['candidates']


@pytest.mark.parametrize('key_type,key_prefix', [
    ('COOKIES', 'yandex'),
    ('HEADERS', 'x-fo'),
])
def test_json_keys_suggests(sess, key_type, key_prefix, awacs_end_date):
    d = {
        'request': {
            'namespace': 'anttsov-test-balancer-ns',
            'timestamp_range': {
                'begin': (awacs_end_date - timedelta(days=2)).isoformat() + 'Z',
                'end': awacs_end_date.isoformat() + 'Z'
            },
            'limit': 5,
            'order': 'DESC',
        },
        "key_type": key_type,
        "key_prefix": key_prefix,
    }

    url = 'http://localhost:8080/api/v2/awacs/logs/GetQueryContextKeys/?_snake_case=true'

    r = sess.post(url=url, json=d, verify=False)
    r.raise_for_status()
    rsp = r.json()

    assert rsp['candidates']
