from __future__ import unicode_literals
import logging
import socket

from flask import request as flask_request
from flask import g
from six.moves import urllib

import library.python.vault_client as vault_client
from sepelib.core import config
from infra.swatlib.rpc import exceptions
from infra.swatlib.rpc import request_limiter
from infra.dproxy.proto import dproxy_pb2
from infra.dproxy.src.lib.rpc import blueprint
from infra.dproxy.src.api import error_utils


MAX_PAGE_SIZE = 200

yav_service_blueprint = blueprint.HttpRpcBlueprint('rpc.yav_service',
                                                   __name__,
                                                   '/api/yav')


log = logging.getLogger(__name__)


def get_user_ip_host():
    if config.get_value('run.spoof_user_ip', False):  # for local developer laptop
        u = urllib.parse.urlparse(config.get_value('env_cfg.web_url'))
        h = socket.getfqdn(u.netloc)
        _, _, _, _, a = socket.getaddrinfo(socket.getfqdn(u.netloc), 80)[0]
        return a[0], h
    if flask_request.access_route:
        return flask_request.access_route[0], flask_request.host
    return '127.0.0.1', flask_request.host


def get_vault_client(auth_subject):
    tp = g.ctx.yav_thread_pool
    session_id = flask_request.cookies.get('Session_id')
    if not session_id:
        raise exceptions.BadRequestError("No 'Session_id' cookie specified.")

    log.info("Got session id: user=%s", auth_subject.login)
    with g.ctx.metrics_registry.get_counter('passport_tvm_fail').count_errors():
        bb_ticket = g.ctx.tvm_client.ticket_to(config.get_value('passport.tvm_client_id'))

    user_ip, host = get_user_ip_host()
    log.info("Getting user ticket: user=%s, user_ip=%s, host=%s",
             auth_subject.login, user_ip, host)
    with g.ctx.metrics_registry.get_counter('passport_user_ticket_fail').count_errors():
        user_ticket = g.ctx.blackbox_client.get_user_ticket(
            session_id=session_id,
            service_ticket=bb_ticket,
            user_ip=user_ip,
            host=host,
        )

    with g.ctx.metrics_registry.get_counter('yav_tvm_fail').count_errors():
        yav_ticket = g.ctx.tvm_client.ticket_to(config.get_value('yav.tvm_client_id'))

    log.info("Got user ticket: user=%s", auth_subject.login)

    with g.ctx.metrics_registry.get_counter('yav_create_vault_client_fail').count_errors():
        return tp.apply(
            vault_client.VaultClient,
            kwds=dict(
                host=config.get_value('yav.host'),
                check_status=False,
                user_ticket=user_ticket,
                service_ticket=yav_ticket,
            ),
        )


def check_delegation_token_to_renew(vault_client, token, secret_uuid, signature):
    reasons = dproxy_pb2.RenewDelegationTokenResponse
    yp_tvm_id = config.get_value('yp.tvm_client_id')
    tp = g.ctx.yav_thread_pool

    with g.ctx.metrics_registry.get_counter('yav_get_token_info_fail').count_errors():
        resp = tp.apply(vault_client.get_token_info, kwds=dict(token=token))

    token_info = resp.get('token_info')
    if token_info.get('state_name') != 'normal':
        return reasons.INVALID_STATE_NAME
    if str(token_info.get('tvm_client_id')) != str(yp_tvm_id):
        return reasons.INVALID_TVM_ID

    with g.ctx.metrics_registry.get_counter('yav_list_tokens_fail').count_errors():
        all_tokens = tp.apply(vault_client.list_tokens, kwds=dict(secret_uuid=secret_uuid))

    token_uuid = token_info.get('token_uuid')
    for t in all_tokens:
        if token_uuid and token_uuid == t.get('token_uuid') and t.get('signature') == signature:
            return reasons.NONE
    return reasons.NOT_FOUND


@yav_service_blueprint.method('ListSecrets',
                              request_type=dproxy_pb2.ListSecretsRequest,
                              response_type=dproxy_pb2.ListSecretsResponse,
                              need_authentication=True)
@error_utils.translate_app_errors
def list_secrets(req, auth_subject):
    c = get_vault_client(auth_subject)
    tp = g.ctx.yav_thread_pool
    if req.page_size > MAX_PAGE_SIZE:
        raise exceptions.BadRequestError("page_size: max: {}, got: {}".format(MAX_PAGE_SIZE, req.page_size))
    page_size = req.page_size
    if not page_size:
        page_size = config.get_value('yav.list_secrets_default_limit', 50)

    with g.ctx.metrics_registry.get_counter('yav_list_secrets_fail').count_errors():
        secrets = tp.apply(
            c.list_secrets,
            kwds=dict(
                order_by="name",
                asc=True,
                query_type="infix",
                page=max(req.page-1, 0),
                query=req.query or None,  # Empty query is non-valid
                page_size=page_size,
            ),
        )

    log.info("Got %d secrets: user=%s", len(secrets), auth_subject.login)
    resp = dproxy_pb2.ListSecretsResponse()
    for s in secrets:
        s_pb = resp.secrets.add()
        s_pb.uuid = s['uuid']
        s_pb.name = s['name']
        s_pb.created_at.seconds = int(s['created_at'])
        s_pb.created_by = s['creator_login']
    return resp


@yav_service_blueprint.method('ListSecretVersions',
                              request_type=dproxy_pb2.ListSecretVersionsRequest,
                              response_type=dproxy_pb2.ListSecretVersionsResponse,
                              need_authentication=True)
@error_utils.translate_app_errors
def list_secret_versions(req, auth_subject):
    secret_uuid = req.secret_uuid
    if not secret_uuid:
        raise exceptions.BadRequestError("No 'secret_uuid' parameter specified in request.")

    c = get_vault_client(auth_subject)
    tp = g.ctx.yav_thread_pool

    with g.ctx.metrics_registry.get_counter('yav_get_secret_fail').count_errors():
        secret = tp.apply(c.get_secret, args=(secret_uuid,))

    log.info("Got secret '%s': user=%s", secret_uuid, auth_subject.login)
    resp = dproxy_pb2.ListSecretVersionsResponse()
    for v in secret['secret_versions']:
        v_pb = resp.versions.add()
        v_pb.version = v['version']
        v_pb.keys.extend(v['keys'])
        v_pb.created_at.seconds = int(v['created_at'])
        v_pb.created_by = v['creator_login']
    return resp


@yav_service_blueprint.method('ListDelegationTokens',
                              request_type=dproxy_pb2.ListDelegationTokensRequest,
                              response_type=dproxy_pb2.ListDelegationTokensResponse,
                              need_authentication=True)
@error_utils.translate_app_errors
def list_delegation_tokens(req, auth_subject):
    secret_uuid = req.secret_uuid
    if not secret_uuid:
        raise exceptions.BadRequestError("No 'secret_uuid' parameter specified in request.")

    c = get_vault_client(auth_subject)
    tp = g.ctx.yav_thread_pool

    with g.ctx.metrics_registry.get_counter('yav_list_tokens_fail').count_errors():
        tokens = tp.apply(c.list_tokens, kwds=dict(secret_uuid=secret_uuid))

    log.info("Got %d tokens: secret=%s, user=%s",
             len(tokens), secret_uuid, auth_subject.login)
    resp = dproxy_pb2.ListDelegationTokensResponse()
    for t in tokens:
        t_pb = resp.tokens.add()
        t_pb.uuid = t['token_uuid']
    return resp


@yav_service_blueprint.method('CreateDelegationToken',
                              request_type=dproxy_pb2.CreateDelegationTokenRequest,
                              response_type=dproxy_pb2.CreateDelegationTokenResponse,
                              need_authentication=True)
@request_limiter.limit(30)
@error_utils.translate_app_errors
def create_delegation_token(req, auth_subject):
    secret_uuid = req.secret_uuid
    signature = req.signature
    if not secret_uuid:
        raise exceptions.BadRequestError("No 'secret_uuid' parameter specified in request.")
    if not signature:
        raise exceptions.BadRequestError("No 'signature' parameter specified in request.")

    c = get_vault_client(auth_subject)
    tp = g.ctx.yav_thread_pool

    with g.ctx.metrics_registry.get_counter('yav_create_token_fail').count_errors():
        token, token_uuid = tp.apply(
            c.create_token,
            kwds=dict(
                secret_uuid=secret_uuid,
                tvm_client_id=config.get_value('yp.tvm_client_id'),
                signature=signature,
            ),
        )

    log.info("Created token: user=%s", auth_subject.login)
    resp = dproxy_pb2.CreateDelegationTokenResponse()
    resp.token.uuid = token_uuid
    resp.token.token = token
    return resp


@yav_service_blueprint.method('RenewDelegationToken',
                              request_type=dproxy_pb2.RenewDelegationTokenRequest,
                              response_type=dproxy_pb2.RenewDelegationTokenResponse,
                              need_authentication=True)
@request_limiter.limit(30)
@error_utils.translate_app_errors
def renew_delegation_token(req, auth_subject):
    token = req.token.token
    secret_uuid = req.secret_uuid
    signature = req.signature

    if not token:
        raise exceptions.BadRequestError("No 'token' parameter specified in request.")
    if not secret_uuid:
        raise exceptions.BadRequestError("No 'secret_uuid' parameter specified in request.")
    if not signature:
        raise exceptions.BadRequestError("No 'signature' parameter specified in request.")

    c = get_vault_client(auth_subject)
    tp = g.ctx.yav_thread_pool
    reason = check_delegation_token_to_renew(vault_client=c,
                                             token=token,
                                             secret_uuid=secret_uuid,
                                             signature=signature)
    if reason != dproxy_pb2.RenewDelegationTokenResponse.NONE:
        with g.ctx.metrics_registry.get_counter('yav_create_token_fail').count_errors():
            token, token_uuid = tp.apply(
                c.create_token,
                kwds=dict(
                    secret_uuid=secret_uuid,
                    tvm_client_id=config.get_value('yp.tvm_client_id'),
                    signature=signature,
                ),
            )

        log.info("Renewed token: user=%s, reason=%s",
                 auth_subject.login,
                 dproxy_pb2.RenewDelegationTokenResponse.RenewalReason.Name(reason))
    else:
        token_uuid = req.token.uuid
        log.info("Token is valid, did not renew: user=%s", auth_subject.login)

    resp = dproxy_pb2.RenewDelegationTokenResponse()
    resp.token.token = token
    resp.token.uuid = token_uuid
    resp.reason = reason
    return resp
