try:
    import sys
    import errno
    import socket
    import logging
    import argparse

    from infra.swatlib import cmdutil
    from infra.dproxy.src.server import dproxy
    from infra.dproxy.src.lib import logs
except KeyboardInterrupt:
    # don't bother user with python stack trace
    # if interrupted (by Ctrl+C) during imports
    raise SystemExit(1)


log = logging.getLogger('main')


def init_arg_parser():
    p = argparse.ArgumentParser(description='dproxy')
    p.add_argument('-v', '--version',
                   action='version',
                   version=dproxy.VERSION)
    p.add_argument('-c', '--cfg',
                   default=cmdutil.DEFAULT_CONFIG_PATH,
                   action='store', help='path to service cfg file')
    # --console is used to dump logs during debugging with IDE
    # when stdout isn't terminal but we don't want to log to file
    p.add_argument('--console',
                   default=False,
                   action='store_true',
                   help='redirect log to stdout (overrides config)')
    p.add_argument('-d', '--debug',
                   default=False,
                   action='store_true',
                   help='do NOT daemonize, do NOT change working dir')
    return p


def main():
    # We have to mimic cmdutil.main() here to some extent.
    # This is done because swatlib cmdutil is highly dependent on nanny-style
    # logging configuration and patterns, and thus cannot be easily modified
    # for another log format.
    application = None
    try:
        with logs.temporary_stdout_logger():
            args = cmdutil.parse_args(init_arg_parser(), sys.argv[1:])
            config_context = cmdutil.get_config_context(args, env_prefix='DPROXY')

            cmdutil.config.load(args.cfg, config_context=config_context)

        logs.setup_logging(cmdutil.config.get_value('logging'), args.console)

        application = dproxy.Application('%s@%s' % (dproxy.Application.name, socket.gethostname()))
        application.run()
    except SystemExit:
        raise
    except KeyboardInterrupt:
        log.info("exiting on user request")
        exit_code = 0
    except socket.error as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on socket error: code='{}' msg='{}'".format(errorcode, e.strerror))
        exit_code = 1
    except IOError as e:
        errorcode = errno.errorcode.get(e.errno, e.errno)
        log.exception("exiting on io error: file='{}' code='{}' msg='{}'".format(
            e.filename, errorcode, e.strerror))
        exit_code = 1
    except Exception as e:
        log.exception(cmdutil.exc.format_exc("exiting on error", e))
        exit_code = 1
    else:
        exit_code = 0
    finally:
        if application is not None:
            application.stop()

    raise SystemExit(exit_code)


if __name__ == '__main__':
    main()
