import sys
import logging
from contextlib import contextmanager

import six


DEFAULT_FORMAT = "%(process)s %(asctime)s.%(msecs)003d %(levelname)s [%(name)s] %(message)s"
DATE_FMT = "%Y-%m-%d %H:%M:%S"
_level_name_aliases = {
    # LEVEL:     (1char, 3 chars, 4 chars, full)
    'DEBUG':    ('D',   'DBG',   'DEBG',  'DEBUG'),
    'INFO':     ('I',   'INF',   'INFO',  'NORMAL'),
    'WARNING':  ('W',   'WRN',   'WARN',  'WARNING'),
    'ERROR':    ('E',   'ERR',   'ERRR',  'ERROR'),
    'CRITICAL': ('C',   'CRI',   'CRIT',  'CRITICAL'),
    'NOTSET':   ('-',   '---',   '----',  'NOTSET'),
}


def rename_levels(chars=None):
    assert chars in (None, 1, 3, 4), 'We only support unlimited and 1-3-4 char themes'

    if chars == 1:
        idx = 0
    elif chars == 3:
        idx = 1
    elif chars == 4:
        idx = 2
    else:
        idx = 3

    for level, aliases in list(_level_name_aliases.items()):
        selectedAlias = aliases[idx]

        levelno = logging.getLevelName(level)
        logging.addLevelName(levelno, selectedAlias)


@contextmanager
def temporary_stdout_logger():
    formatter = logging.Formatter(DEFAULT_FORMAT, datefmt=DATE_FMT)
    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(logging.DEBUG)
    handler.setFormatter(formatter)
    logger = logging.getLogger()
    old_level = logger.level

    logger.addHandler(handler)
    try:
        yield
    finally:
        logger.removeHandler(handler)
        logger.setLevel(old_level)


def find_object(name):
    if hasattr(logging, name):
        return getattr(logging, name)

    if '.' not in name:
        raise Exception('Cant find object {0!r}'.format(name))

    module, class_name = name.rsplit('.', 1)
    return getattr(__import__(module, fromlist=['']), class_name)


def create_formatters(conf, console=False):
    formatters = {}
    for name, props in six.iteritems(conf):
        # if user requires console mode, always use default formatter
        # and do not display logs in json and so on
        if console:
            cls = logging.Formatter
            props = dict(fmt=DEFAULT_FORMAT, datefmt=DATE_FMT)
        elif '_class' in props:
            cls_mod, cls_name = props.pop('_class').rsplit('.', 1)
            mod = __import__(cls_mod, fromlist=[cls_name])
            cls = getattr(mod, cls_name)
        else:
            cls = logging.Formatter      # pylint: disable=E1101
        formatters[name] = cls(**props)  # pylint: disable=W0142
    return formatters


def is_console_handler(handler):
    return (
        isinstance(handler, logging.StreamHandler)
        and handler.stream in (sys.stdout, sys.stderr)
    )


def create_handlers(conf, formatters, console=False):
    handlers = {}
    for name, props in six.iteritems(conf):
        handler_class_name = props['class']

        handler_class = find_object(handler_class_name)

        if 'args' in props:
            args = []
            for arg in props['args']:
                args.append(find_object(arg))
        else:
            args = []

        kwargs = props.pop('kwargs', {})

        handler = handler_class(*args, **kwargs)  # pylint: disable=W0142

        if 'level' in props:
            handler.setLevel(getattr(logging, props['level']))

        if 'formatter' in props:
            formatter = formatters[props['formatter']]
        else:
            formatter = logging.Formatter(DEFAULT_FORMAT, datefmt=DATE_FMT)
        handler.setFormatter(formatter)

        handlers[name] = handler

    if console:
        console_handler = next(
            (
                name
                for name, handler in six.iteritems(conf)
                if is_console_handler(handler)
            ),
            None
        )
        if console_handler is None:
            handlers['_console'] = logging.StreamHandler(sys.stdout)
            handlers['_console'].setFormatter(logging.Formatter(DEFAULT_FORMAT, datefmt=DATE_FMT))
        else:
            handlers['_console'] = handlers[console_handler]

    return handlers


def create_loggers(conf, handlers, console=False):
    loggers = {}
    console_enabled = False

    for name, props in six.iteritems(conf):
        logger = logging.getLogger(name)  # pylint: disable=E1101
        for handler_name in props.get('handlers', ()):
            logger.addHandler(handlers[handler_name])
            if is_console_handler(handlers[handler_name]):
                console_enabled = True
        if 'level' in props:
            logger.setLevel(getattr(logging, props['level']))

        loggers[name] = logger

    if console and not console_enabled:
        logger = loggers.setdefault('', logging.getLogger(''))
        logger.addHandler(handlers['_console'])
        logger.setLevel(logging.DEBUG)

    return loggers


def setup_logging(log_config, console=False):
    level_name_scheme = log_config.get('level_nane_scheme', 1 if console else None)
    if level_name_scheme is not None:
        rename_levels(level_name_scheme)

    formatters = create_formatters(log_config['formatters'], console=console)
    handlers = create_handlers(log_config['handlers'], formatters, console=console)

    create_loggers(log_config['loggers'], handlers, console=console)
