from __future__ import unicode_literals
import datetime
import logging

import ujson
import gevent
from gevent import threadpool
from google.protobuf import json_format

import ydb
from infra.dproxy.proto import dproxy_pb2
from infra.dproxy.proto import awacs_pb2
from infra.dproxy.src.ydb_logs import ydbutil, queries


class YdbLogsControllerBase(object):
    FAIL_COUNTER = None
    TIMEOUT_COUNTER = None
    TOTAL_TIMEOUT_COUNTER = None

    def __init__(self, scheme_client, table_client, metrics_registry, history_tables_count, pool_size, request_timeout):
        self.scheme_client = scheme_client
        self.table_client = table_client
        self.metrics_registry = metrics_registry
        self.log = logging.getLogger('ydb-logs-ctl')
        self.history_tables_count = history_tables_count
        self.tp = threadpool.ThreadPool(maxsize=pool_size)
        self.request_timeout = request_timeout

    def calc_min_max_possible_dates(self, today=None, now=None):
        now = now or datetime.datetime.utcnow()
        today = today or now.replace(hour=0, minute=0, second=0, microsecond=0)

        min_possible = today - datetime.timedelta(days=self.history_tables_count)
        max_possible = now

        return min_possible, max_possible

    def get_tables_generator_class(self):
        return ydbutil.DateTablesGenerator

    def make_tables_generator(self, timestamp_range, order, table_path, query_builder, today=None, now=None):
        # TODO ask about min possible date for 'one table for all' in olap
        min_possible, max_possible = self.calc_min_max_possible_dates(today=today, now=now)

        begin, end = timestamp_range
        max_date = end and end.ToDatetime() or max_possible
        min_date = begin and begin.ToDatetime() or min_possible
        # TODO get dt from continuation token too
        dt = min_date if order == dproxy_pb2.ASC else max_date

        self.log.debug('generating tables from %s <= %s <= %s', min_date, dt, max_date)
        tables_generator_class = self.get_tables_generator_class()
        return tables_generator_class(
            log=self.log,
            scheme_client=self.scheme_client,
            min_date=min_date,
            dt=dt,
            max_date=max_date,
            table_path=table_path,
            query_builder=query_builder,
            order=order,
        )

    def _execute(self, q, params, param_types, limit):
        self.log.info('executing YQL-query %s with params %s and param_types %s', q, params, param_types)
        scan_query = ydb.ScanQuery(q, param_types)
        scan_query_settings = ydb.ScanQuerySettings().with_collect_stats(ydb.QueryStatsCollectionMode.FULL)
        r = self.table_client.scan_query(scan_query, params, settings=scan_query_settings)
        rv = []
        last_row_set = None
        for row_set in r:
            last_row_set = row_set
            if len(rv) < limit:
                rv.extend(row_set.result_set.rows[:limit - len(rv)])

        if last_row_set is not None:
            self.log.info("scan query %r statistics: %s", q, json_format.MessageToDict(last_row_set.query_stats))

        return rv

    def _fetch(self, tables_gen, limit, fetch_timeout=None):
        rv = []

        total_timeout = gevent.Timeout(fetch_timeout)
        try:
            with total_timeout:
                for q, params, param_types in tables_gen:
                    request_timeout = gevent.Timeout(self.request_timeout)
                    try:
                        with request_timeout:
                            rv.extend(self.tp.apply(ydb.retry_operation_sync, args=(self._execute, None, q, params, param_types, limit)))
                            # r = self.tp.apply(self.session_pool.retry_operation_sync, args=(self._execute,), kwds=kwargs)
                    except ydb.issues.SchemeError:
                        self.log.debug("Table not found")
                        continue
                    except gevent.Timeout as e:
                        if e is request_timeout:
                            self.metrics_registry.get_counter(self.FAIL_COUNTER).inc()
                            self.metrics_registry.get_counter(self.TIMEOUT_COUNTER).inc()
                        raise
                    except Exception:
                        self.metrics_registry.get_counter(self.FAIL_COUNTER).inc()
                        raise

                    if len(rv) >= limit:
                        return rv
        except gevent.Timeout as e:
            if e is total_timeout:
                self.metrics_registry.get_counter(self.TOTAL_TIMEOUT_COUNTER).inc()
            else:
                raise

        return rv


class YdbLogsController(YdbLogsControllerBase):
    FAIL_COUNTER = 'ydb_query_fail'
    TIMEOUT_COUNTER = 'ydb_query_timeout_fail'
    TOTAL_TIMEOUT_COUNTER = 'ydb_query_total_timeout'

    def search(
        self,
        table_path,
        search_patterns,
        timestamp_range,
        log_levels,
        log_levels_int,
        hosts,
        pods,
        boxes,
        workloads,
        containers,
        logger_names,
        pod_transient_fqdns,
        pod_persistent_fqdns,
        node_fqdns,
        thread_names,
        request_ids,
        stack_traces,
        user_fields,
        continuation_token,
        limit,
        order,
        query_timeout=None,
    ):
        query_builder = (
            queries.SearchLogsQueryBuilder()
            .with_table_path(table_path)
            .with_log_levels(log_levels)
            .with_messages(search_patterns)
            .with_hosts(hosts)
            .with_pods(pods)
            .with_boxes(boxes)
            .with_workloads(workloads)
            .with_containers(containers)
            .with_logger_names(logger_names)
            .with_log_levels_int(log_levels_int)
            .with_pod_transient_fqdns(pod_transient_fqdns)
            .with_pod_persistent_fqdns(pod_persistent_fqdns)
            .with_node_fqdns(node_fqdns)
            .with_thread_names(thread_names)
            .with_request_ids(request_ids)
            .with_stack_traces(stack_traces)
            .with_user_fields(user_fields)
            .with_continuation_token(continuation_token)
            .order(order)
            .limit(limit)
        )

        tables_g = self.make_tables_generator(
            timestamp_range=timestamp_range,
            order=order,
            table_path=table_path,
            query_builder=query_builder,
        )

        time_hgram = 'ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        resp = dproxy_pb2.SearchLogEntriesResponse()
        for r in rows:
            e = resp.log_entries.add()
            e.timestamp.FromMicroseconds(abs(r['timestamp']))
            e.container_id = r['container_id']
            e.host = r['host']
            e.pod = r['pod']
            e.box = r['box']
            e.workload = r['workload']
            e.logger_name = r['logger_name']
            e.user_id = r['user_id']
            e.request_id = r['request_id']
            e.message = r['message']
            e.log_level = r['log_level']
            e.seq = r['seq']
            e.context = r['context'] or ''
            e.pod_transient_fqdn = r['pod_transient_fqdn']
            e.pod_persistent_fqdn = r['pod_persistent_fqdn']
            e.node_fqdn = r['node_fqdn']
            e.stack_trace = r['stack_trace']
            e.thread_name = r['thread_name']
            e.log_level_int = r['log_level_int']

        if resp.log_entries:
            forward_row = resp.log_entries[-1]
            backward_row = resp.log_entries[0]
            if order == dproxy_pb2.ASC:
                forward_direction = queries.ContinuationToken.RDIRECTIONS['<']
                backward_direction = queries.ContinuationToken.RDIRECTIONS['>']
                current_direction = queries.ContinuationToken.RDIRECTIONS['<=']
            else:
                forward_direction = queries.ContinuationToken.RDIRECTIONS['>']
                backward_direction = queries.ContinuationToken.RDIRECTIONS['<']
                current_direction = queries.ContinuationToken.RDIRECTIONS['>=']

            resp.continuation_tokens.backward = str(queries.ContinuationToken.from_row(backward_row, backward_direction))
            resp.continuation_tokens.forward = str(queries.ContinuationToken.from_row(forward_row, forward_direction))
            resp.continuation_tokens.current = str(queries.ContinuationToken.from_row(backward_row, current_direction))

        return resp

    def get_context_keys(self, known_args, key_prefix, query_timeout=None):
        limit = 500

        query_builder = (
            queries.ContextKeysQueryBuilder()
            .with_table_path(known_args['table_path'])
            .with_log_levels(known_args['log_levels'])
            .with_log_levels_int(known_args['log_levels_int'])
            .with_messages(known_args['search_patterns'])
            .with_hosts(known_args['hosts'])
            .with_pods(known_args['pods'])
            .with_boxes(known_args['boxes'])
            .with_workloads(known_args['workloads'])
            .with_containers(known_args['containers'])
            .with_logger_names(known_args['logger_names'])
            .with_pod_transient_fqdns(known_args['pod_transient_fqdns'])
            .with_pod_persistent_fqdns(known_args['pod_persistent_fqdns'])
            .with_node_fqdns(known_args['node_fqdns'])
            .with_thread_names(known_args['thread_names'])
            .with_request_ids(known_args['request_ids'])
            .with_stack_traces(known_args['stack_traces'])
            .with_user_fields(known_args['user_fields'])
            # .with_key_prefix(key_prefix)
            .with_continuation_token(known_args['continuation_token'])
            .order(known_args['order'])
            .limit(limit)
        )

        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        suggestions = set()
        for r in rows:
            try:
                data = ujson.loads(r['context'])
                for key in ydbutil.keys_of_dict(data, query_prefix=key_prefix):
                    suggestions.add(key)
                    if len(suggestions) > 20:
                        return list(suggestions)
            except Exception:
                continue

        return list(suggestions)

    def get_search_suggests(self, known_args, key_type, value_prefix, full_search=False, query_timeout=None):
        limit = 20

        builder = queries.SearchSuggestsQueryBuilder() if full_search else queries.FastSearchSuggestsQueryBuilder()
        query_builder = (
            builder
            .with_table_path(known_args['table_path'])
            .with_log_levels(known_args['log_levels'])
            .with_log_levels_int(known_args['log_levels_int'])
            .with_messages(known_args['search_patterns'])
            .with_hosts(known_args['hosts'])
            .with_pods(known_args['pods'])
            .with_boxes(known_args['boxes'])
            .with_workloads(known_args['workloads'])
            .with_containers(known_args['containers'])
            .with_logger_names(known_args['logger_names'])
            .with_pod_transient_fqdns(known_args['pod_transient_fqdns'])
            .with_pod_persistent_fqdns(known_args['pod_persistent_fqdns'])
            .with_node_fqdns(known_args['node_fqdns'])
            .with_thread_names(known_args['thread_names'])
            .with_request_ids(known_args['request_ids'])
            .with_stack_traces(known_args['stack_traces'])
            .with_user_fields(known_args['user_fields'])
            .with_value_prefix(key_type, value_prefix)
            .with_continuation_token(known_args['continuation_token'])
            .order(known_args['order'])
            .limit(limit)
        )
        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        return {r.get(key_type) for r in rows}


class OlapYdbLogsController(YdbLogsControllerBase):
    FAIL_COUNTER = 'olap_ydb_query_fail'
    TIMEOUT_COUNTER = 'olap_ydb_query_timeout_fail'
    TOTAL_TIMEOUT_COUNTER = 'olap_ydb_query_total_timeout'

    def get_tables_generator_class(self):
        return ydbutil.OlapTablesGenerator

    def search(
        self,
        table_path,
        search_patterns,
        timestamp_range,
        log_levels,
        log_levels_int,
        hosts,
        pods,
        boxes,
        workloads,
        containers,
        logger_names,
        pod_transient_fqdns,
        pod_persistent_fqdns,
        node_fqdns,
        thread_names,
        request_ids,
        stack_traces,
        user_fields,
        continuation_token,
        limit,
        order,
        query_timeout=None,
    ):
        query_builder = (
            queries.OlapSearchLogsQueryBuilder()
            .with_table_path(table_path)
            .with_log_levels(log_levels)
            .with_messages(search_patterns)
            .with_hosts(hosts)
            .with_pods(pods)
            .with_boxes(boxes)
            .with_workloads(workloads)
            .with_containers(containers)
            .with_logger_names(logger_names)
            .with_log_levels_int(log_levels_int)
            .with_pod_transient_fqdns(pod_transient_fqdns)
            .with_pod_persistent_fqdns(pod_persistent_fqdns)
            .with_node_fqdns(node_fqdns)
            .with_thread_names(thread_names)
            .with_request_ids(request_ids)
            .with_stack_traces(stack_traces)
            .with_user_fields(user_fields)
            .with_continuation_token(continuation_token)
            .order(order)
            .limit(limit)
        )

        tables_g = self.make_tables_generator(
            timestamp_range=timestamp_range,
            order=order,
            table_path=table_path,
            query_builder=query_builder,
        )

        time_hgram = 'olap_ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        resp = dproxy_pb2.SearchLogEntriesResponse()
        for r in rows:
            e = resp.log_entries.add()
            e.timestamp.FromMicroseconds(r['timestamp'])
            e.container_id = r['container_id']
            e.host = r['host']
            e.pod = r['pod']
            e.box = r['box']
            e.workload = r['workload']
            e.logger_name = r['logger_name']
            e.user_id = r['user_id']
            e.request_id = r['request_id']
            e.message = r['message']
            e.log_level = r['log_level']
            e.seq = r['seq']
            e.context = r['context'] or ''
            e.pod_transient_fqdn = r['pod_transient_fqdn']
            e.pod_persistent_fqdn = r['pod_persistent_fqdn']
            e.node_fqdn = r['node_fqdn']
            e.stack_trace = r['stack_trace']
            e.thread_name = r['thread_name']
            e.log_level_int = r['log_level_int']

        if resp.log_entries:
            forward_row = resp.log_entries[-1]
            backward_row = resp.log_entries[0]
            if order == dproxy_pb2.ASC:
                forward_direction = queries.OlapContinuationToken.RDIRECTIONS['>']
                backward_direction = queries.OlapContinuationToken.RDIRECTIONS['<']
                current_direction = queries.OlapContinuationToken.RDIRECTIONS['>=']
            else:
                forward_direction = queries.OlapContinuationToken.RDIRECTIONS['<']
                backward_direction = queries.OlapContinuationToken.RDIRECTIONS['>']
                current_direction = queries.OlapContinuationToken.RDIRECTIONS['<=']

            resp.continuation_tokens.backward = str(queries.OlapContinuationToken.from_row(backward_row, backward_direction))
            resp.continuation_tokens.forward = str(queries.OlapContinuationToken.from_row(forward_row, forward_direction))
            resp.continuation_tokens.current = str(queries.OlapContinuationToken.from_row(backward_row, current_direction))

        return resp

    def get_context_keys(self, known_args, key_prefix, query_timeout=None):
        limit = 500

        query_builder = (
            queries.OlapContextKeysQueryBuilder()
            .with_table_path(known_args['table_path'])
            .with_log_levels(known_args['log_levels'])
            .with_log_levels_int(known_args['log_levels_int'])
            .with_messages(known_args['search_patterns'])
            .with_hosts(known_args['hosts'])
            .with_pods(known_args['pods'])
            .with_boxes(known_args['boxes'])
            .with_workloads(known_args['workloads'])
            .with_containers(known_args['containers'])
            .with_logger_names(known_args['logger_names'])
            .with_pod_transient_fqdns(known_args['pod_transient_fqdns'])
            .with_pod_persistent_fqdns(known_args['pod_persistent_fqdns'])
            .with_node_fqdns(known_args['node_fqdns'])
            .with_thread_names(known_args['thread_names'])
            .with_request_ids(known_args['request_ids'])
            .with_stack_traces(known_args['stack_traces'])
            .with_user_fields(known_args['user_fields'])
            # .with_key_prefix(key_prefix)
            .with_continuation_token(known_args['continuation_token'])
            .order(known_args['order'])
            .limit(limit)
        )

        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'olap_ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        suggestions = set()
        for r in rows:
            try:
                data = ujson.loads(r['context'])
                for key in ydbutil.keys_of_dict(data, query_prefix=key_prefix):
                    suggestions.add(key)
                    if len(suggestions) > 20:
                        return list(suggestions)
            except Exception:
                continue

        return list(suggestions)

    def get_search_suggests(self, known_args, key_type, value_prefix, full_search=False, query_timeout=None):
        limit = 20

        builder = queries.OlapSearchSuggestsQueryBuilder() if full_search else queries.OlapFastSearchSuggestsQueryBuilder()
        query_builder = (
            builder
            .with_table_path(known_args['table_path'])
            .with_log_levels(known_args['log_levels'])
            .with_log_levels_int(known_args['log_levels_int'])
            .with_messages(known_args['search_patterns'])
            .with_hosts(known_args['hosts'])
            .with_pods(known_args['pods'])
            .with_boxes(known_args['boxes'])
            .with_workloads(known_args['workloads'])
            .with_containers(known_args['containers'])
            .with_logger_names(known_args['logger_names'])
            .with_pod_transient_fqdns(known_args['pod_transient_fqdns'])
            .with_pod_persistent_fqdns(known_args['pod_persistent_fqdns'])
            .with_node_fqdns(known_args['node_fqdns'])
            .with_thread_names(known_args['thread_names'])
            .with_request_ids(known_args['request_ids'])
            .with_stack_traces(known_args['stack_traces'])
            .with_user_fields(known_args['user_fields'])
            .with_value_prefix(key_type, value_prefix)
            .with_continuation_token(known_args['continuation_token'])
            .order(known_args['order'])
            .limit(limit)
        )
        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'olap_ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        return {r.get(key_type) for r in rows}


class AwacsYdbLogsController(YdbLogsControllerBase):
    FAIL_COUNTER = 'awacs_ydb_query_fail'
    TIMEOUT_COUNTER = 'awacs_ydb_query_timeout_fail'
    TOTAL_TIMEOUT_COUNTER = 'awacs_ydb_query_total_timeout'

    @staticmethod
    def _prepare_builder(builder, known_args, limit):
        return (
            builder
            .with_table_path(known_args['table_path'])
            .with_env_types(known_args['env_types'])
            .with_domains(known_args['domains'])
            .with_upstreams(known_args['upstreams'])
            .with_client_ips(known_args['client_ips'])
            .with_client_ports(known_args['client_ports'])
            .with_hostnames(known_args['hostnames'])
            .with_yandexuids(known_args['yandexuids'])
            .with_cookie_fields(known_args['cookies'])
            .with_header_fields(known_args['headers'])
            .with_methods(known_args['methods'])
            .with_reasons(known_args['reasons'])
            .with_request_ids(known_args['request_ids'])
            .with_requests(known_args['requests'])
            .with_statuses(known_args['statuses'])
            .with_process_time_list(known_args['process_time_list'])
            .with_continuation_token(known_args['continuation_token'])
            .order(known_args['order'])
            .limit(limit)
        )

    def awacs_search(self, known_args, limit, query_timeout=None):
        query_builder = self._prepare_builder(queries.AwacsSearchLogsQueryBuilder(), known_args, limit)

        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'awacs_ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        resp = awacs_pb2.SearchLogEntriesResponse()

        def set_field(e, row, field):
            value = row[field]
            if value is not None:
                setattr(e, field, value)

        for r in rows:
            e = resp.log_entries.add()
            e.timestamp.FromMicroseconds(abs(r['timestamp_us']))
            set_field(e, r, 'env_type')
            set_field(e, r, 'domain')
            set_field(e, r, 'upstream')
            set_field(e, r, 'client_ip')
            set_field(e, r, 'client_port')
            set_field(e, r, 'hostname')
            set_field(e, r, 'cookies')
            set_field(e, r, 'headers')
            set_field(e, r, 'yandexuid')
            set_field(e, r, 'method')
            set_field(e, r, 'process_time')
            set_field(e, r, 'reason')
            set_field(e, r, 'request')
            set_field(e, r, 'request_id')
            set_field(e, r, 'status')
            set_field(e, r, 'workflow')
            set_field(e, r, 'pushclient_row_id')

        if resp.log_entries:
            def token_data_from_row(i):
                r = rows[i]
                return abs(r['timestamp_us']), r['pushclient_row_id'], r['hostname']

            backward_row_token_data = token_data_from_row(0)
            forward_row_token_data = token_data_from_row(-1)

            if known_args['order'] == dproxy_pb2.ASC:
                forward_direction = queries.AwacsContinuationToken.RDIRECTIONS['<']
                backward_direction = queries.AwacsContinuationToken.RDIRECTIONS['>']
                current_direction = queries.AwacsContinuationToken.RDIRECTIONS['<=']
            else:
                forward_direction = queries.AwacsContinuationToken.RDIRECTIONS['>']
                backward_direction = queries.AwacsContinuationToken.RDIRECTIONS['<']
                current_direction = queries.AwacsContinuationToken.RDIRECTIONS['>=']

            def make_token(direction, token_data):
                return str(queries.AwacsContinuationToken(direction, *token_data))

            resp.continuation_tokens.backward = make_token(backward_direction, backward_row_token_data)
            resp.continuation_tokens.forward = make_token(forward_direction, forward_row_token_data)
            resp.continuation_tokens.current = make_token(current_direction, backward_row_token_data)

        return resp

    def awacs_get_search_suggests(self, known_args, field, value_prefix, full_search=False, query_timeout=None):
        limit = 20

        builder = queries.AwacsSearchSuggestsQueryBuilder() if full_search else queries.AwacsFastSearchSuggestsQueryBuilder()
        query_builder = self._prepare_builder(builder, known_args, limit).with_value_prefix(field, value_prefix)
        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'awacs_ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))

        rv = set()
        for r in rows:
            v = r.get(field)
            if v is None:
                continue
            if isinstance(v, int) or isinstance(v, float):
                v = str(v)
            rv.add(v)

        return rv

    def awacs_get_context_keys(self, known_args, field, key_prefix, query_timeout=None):
        limit = 500

        query_builder = queries.AwacsContextKeysQueryBuilder()
        query_builder = self._prepare_builder(query_builder, known_args, limit).with_field(field)

        tables_g = self.make_tables_generator(
            timestamp_range=known_args['timestamp_range'],
            order=known_args['order'],
            table_path=known_args['table_path'],
            query_builder=query_builder,
        )

        time_hgram = 'awacs_ydb_query_execution_time'
        with self.metrics_registry.get_histogram(time_hgram).timer():
            rows = self._fetch(tables_gen=tables_g,
                               limit=limit,
                               fetch_timeout=query_timeout,
                               )
        self.log.info('fetched %d rows', len(rows))
        suggestions = set()
        for r in rows:
            try:
                data = ujson.loads(r[field])
                for key in ydbutil.keys_of_dict(data, query_prefix=key_prefix):
                    suggestions.add(key)
                    if len(suggestions) > 20:
                        return list(suggestions)
            except Exception:
                continue

        return list(suggestions)
