from __future__ import unicode_literals
import re
import time
import struct
import datetime
import textwrap

from infra.dproxy.proto import dproxy_pb2
from infra.dproxy.src.ydb_logs.timeutil import dt_to_microseconds, microseconds_to_dt, dt_to_str

import ydb


DECLARE_TPL = "DECLARE {} AS {};"
ASSIGN_TPL = "{} = {};"

TABLE_PATH_RE = re.compile(r"^[a-zA-Z0-9_\-\.\/]+$")

# we have to wrap user query into \Q...\E, so we should escape \E
RE_QUOTE = re.compile('(?<!\\\\)(\\\\\\\\)*\\\\E', re.UNICODE)

ASC_STR = 'ASC'
DESC_STR = 'DESC'

ORDER_PB_TO_STR = {
    dproxy_pb2.DESC : DESC_STR,
    dproxy_pb2.ASC : ASC_STR,
}

REV_ORDER_PB_TO_STR = {
    dproxy_pb2.DESC : ASC_STR,
    dproxy_pb2.ASC : DESC_STR,
}


class ContinuationTokenBase(object):
    DIRECTIONS = {
        b'>': '>',
        b'<': '<',
        b')': '>=',
        b'(': '<=',
    }
    RDIRECTIONS = dict(tuple(reversed(item)) for item in DIRECTIONS.items())
    STRICT_DIRECTIONS = {
        b'>': '>',
        b'<': '<',
        b')': '>',
        b'(': '<',
    }

    @property
    def operator(self):
        return self.DIRECTIONS[self.direction]

    @property
    def strict_operator(self):
        return self.STRICT_DIRECTIONS[self.direction]


class ContinuationToken(ContinuationTokenBase):
    CONTINUATION_TOKEN = struct.Struct('<cqQ')

    @classmethod
    def from_str(cls, token):
        try:
            direction, ts, seq = cls.CONTINUATION_TOKEN.unpack_from(token, 0)
            container_id = token[cls.CONTINUATION_TOKEN.size:]

            return cls(
                direction=direction,
                ts=ts,
                seq=seq,
                container_id=container_id,
            )

        except Exception as e:
            raise ValueError("Continuation token parse error: {}".format(e))

    @classmethod
    def from_row(cls, row, direction):
        return cls(
            direction=direction,
            ts=row.timestamp.ToMicroseconds(),
            seq=row.seq,
            container_id=row.container_id,
        )

    def __init__(self, direction, ts, seq, container_id):
        assert direction in self.DIRECTIONS

        self.direction = direction
        self.ts = ts
        self.seq = seq
        self.container_id = container_id

    def __str__(self):
        header = self.CONTINUATION_TOKEN.pack(self.direction, self.ts, self.seq)
        return header + self.container_id


class OlapContinuationToken(ContinuationTokenBase):
    CONTINUATION_TOKEN = struct.Struct('<cqQ')

    @classmethod
    def from_str(cls, token):
        try:
            direction, ts, seq = cls.CONTINUATION_TOKEN.unpack_from(token, 0)
            host = token[cls.CONTINUATION_TOKEN.size:]

            return cls(
                direction=direction,
                ts=ts,
                seq=seq,
                host=host,
            )

        except Exception as e:
            raise ValueError("Continuation token parse error: {}".format(e))

    @classmethod
    def from_row(cls, row, direction):
        return cls(
            direction=direction,
            ts=row.timestamp.ToMicroseconds(),
            seq=row.seq,
            host=row.host,
        )

    def __init__(self, direction, ts, seq, host):
        assert direction in self.DIRECTIONS

        self.direction = direction
        self.ts = ts
        self.seq = seq
        self.host = host

    def __str__(self):
        header = self.CONTINUATION_TOKEN.pack(self.direction, self.ts, self.seq)
        return header + self.host


class AwacsContinuationToken(ContinuationTokenBase):
    CONTINUATION_TOKEN = struct.Struct('<cqq')

    @classmethod
    def from_str(cls, token):
        try:
            direction, ts, pushclient_row_id = cls.CONTINUATION_TOKEN.unpack_from(token, 0)
            hostname = token[cls.CONTINUATION_TOKEN.size:]

            return cls(
                direction=direction,
                ts=ts,
                pushclient_row_id=pushclient_row_id,
                hostname=hostname,
            )

        except Exception as e:
            raise ValueError("Continuation token parse error: {}".format(e))

    @classmethod
    def from_row(cls, row, direction):
        return cls(
            direction=direction,
            ts=row.timestamp.ToMicroseconds(),
            pushclient_row_id=row.pushclient_row_id,
            hostname=row.hostname,
        )

    def __init__(self, direction, ts, pushclient_row_id, hostname):
        assert direction in self.DIRECTIONS

        self.direction = direction
        self.ts = ts
        self.hostname = hostname
        self.pushclient_row_id = pushclient_row_id

    def __str__(self):
        header = self.CONTINUATION_TOKEN.pack(self.direction, self.ts, self.pushclient_row_id)
        return header + self.hostname.encode('utf-8')


class QueryBuilderBase(object):
    QUERY_TPL = ''
    CT_QUERY_TPL = ''
    TIMESTAMP_FLD = ''

    IN_TABLE_INTERVALS = [
        datetime.timedelta(hours=1),
        datetime.timedelta(hours=3),
        datetime.timedelta(hours=20),
    ]

    def __init__(self,
                 timestamp_type='Int64'):
        self._table_path = None
        self._table = None
        self._order = None
        self._rev_order = None
        self._limit = None
        self._declarations = []
        self._assigns = []
        self._clauses = []

        self._param_types = {}
        self._params = {}

        self._template = self.QUERY_TPL
        self._ct1 = None
        self._ct2 = None
        self._ct3 = None
        self._ct_operator = None
        self._ct_strict_operator = None

        self._timestamp_type = timestamp_type

    def copy(self):
        new_builder = type(self)()
        new_builder._table_path = self._table_path
        new_builder._table = self._table
        new_builder._order = self._order
        new_builder._rev_order = self._rev_order
        new_builder._limit = self._limit
        new_builder._declarations.extend(self._declarations)
        new_builder._assigns.extend(self._assigns)
        new_builder._clauses.extend(self._clauses)
        new_builder._param_types.update(self._param_types.iteritems())
        new_builder._params.update(self._params.iteritems())
        new_builder._template = self._template
        new_builder._ct1 = self._ct1
        new_builder._ct2 = self._ct2
        new_builder._ct3 = self._ct3
        new_builder._ct_operator = self._ct_operator
        new_builder._ct_strict_operator = self._ct_strict_operator
        new_builder._timestamp_type = self._timestamp_type
        return new_builder

    def declare(self, name, var_type):
        self._declarations.append(DECLARE_TPL.format(name, var_type))
        if var_type.startswith('List'):
            simple_type = var_type[5:-1]
            self._param_types[name] = ydb.ListType(getattr(ydb.PrimitiveType, simple_type))
        else:
            self._param_types[name] = getattr(ydb.PrimitiveType, var_type)

    def assign(self, name, value):
        self._assigns.append(ASSIGN_TPL.format(name, value))

    def add_clause(self, placeholder, var_type, value, clause):
        self.declare(placeholder, var_type)
        self._params[placeholder] = value
        self._clauses.append(clause.format(placeholder))

    def add_clause_with_assign(self, placeholder, var_type, assign_expression, clause):
        self.assign(placeholder, assign_expression)
        self._clauses.append(clause.format(placeholder))

    def with_table_path(self, path):
        if TABLE_PATH_RE.match(path) is None:
            raise ValueError('table path "{}" contains '
                             'disallowed symbols'.format(path))

        self._table_path = path
        return self

    @staticmethod
    def _simple_str_clause(name):
        def clause(self, value):
            if value:
                placeholder = '$' + name
                clause = '`{name}` == {placeholder}'.format(name=name, placeholder=placeholder)
                self.add_clause('$' + name, 'Utf8', value, clause)
            return self

        return clause

    def with_value_list(self, name, value_type, value_list):
        if not value_list:
            return self

        values, include = value_list[:2]
        if values:
            placeholder = '$' + name
            if len(values) == 1:  # simple optimization
                clause = '`{name}` {exclude}= {placeholder}'.format(
                    name=name,
                    exclude='!' if not include else '=',
                    placeholder=placeholder,
                )
                var_type = value_type
                self.add_clause(placeholder, var_type, values[0], clause)
            else:
                clause = '`{name}` {exclude} IN COMPACT {placeholder}'.format(
                    name=name,
                    exclude='NOT' if not include else '',
                    placeholder=placeholder,
                )
                var_type = 'List<{}>'.format(value_type)
                self.add_clause(placeholder, var_type, values, clause)
        return self

    @staticmethod
    def _simple_list_clause(name, value_type):
        def clause(self, value_list):
            return self.with_value_list(name, value_type, value_list)

        return clause

    def _with_substring_search(self, value_list, field, placeholder_prefix):
        if not value_list:
            return self

        strings, include = value_list[:2]
        if strings:
            for i, string in enumerate(strings):
                string = RE_QUOTE.sub(r'\1\\E', string)
                placeholder = '$%s%d' % (placeholder_prefix, i)
                self.declare(placeholder, 'Utf8')
                self._params[placeholder] = '(?i)\\Q{}\\E'.format(string)
                self.assign('$%s%d_re' % (placeholder_prefix, i), 'Hyperscan::Grep($%s%d)' % (placeholder_prefix, i))

            glue = ' OR ' if include else ' AND '
            clause = '(%s)' % glue.join(
                "%s$%s%d_re(`%s`)" % ('' if include else 'NOT ', placeholder_prefix, i, field)
                for i in xrange(len(strings)))
            self._clauses.append(clause)

        return self

    def _with_grep(self, value_list, field, placeholder_prefix):
        if not value_list:
            return self

        strings, include, op = value_list
        if op is None or op == dproxy_pb2.EQ:
            return self.with_value_list(field, 'Utf8', value_list)
        assert op == dproxy_pb2.GREP
        if strings:
            clauses = []
            for i, string in enumerate(strings):
                placeholder = '$%s%d' % (placeholder_prefix, i)
                placeholder_re = placeholder + '_re'
                self.declare(placeholder, 'Utf8')
                self._params[placeholder] = string
                self.assign(placeholder_re, 'Pire::Grep(%s)' % placeholder)
                clauses.append("%s%s(`%s`)" % ('' if include else 'NOT ', placeholder_re, field))

            glue = ' OR ' if include else ' AND '
            self._clauses.append('(%s)' % glue.join(clauses))

        return self

    def with_messages(self, value_list):
        return self._with_substring_search(value_list, 'message', 'message')

    def with_stack_traces(self, value_list):
        return self._with_substring_search(value_list, 'stack_trace', 'stack_trace')

    def _with_user_fields(self, column, fields):
        for i, (path_parts, values, include) in enumerate(fields or []):
            placeholder = '${}{}_path'.format(column, i)
            path = "lax $." + '.'.join(repr(part) for part in path_parts)
            # self.declare(placeholder, 'Utf8')
            # self._params[placeholder] = path

            placeholder = '${}{}_values'.format(column, i)
            self.declare(placeholder, 'List<Utf8>')
            self._params[placeholder] = values

            clause = 'JSON_VALUE(CAST(`{column}` as Json), "{path}" DEFAULT "" ON EMPTY DEFAULT "" ON ERROR) {op} IN COMPACT ${column}{i}_values'.format(
                column=column,
                path=path,
                i=i,
                op='' if include else 'NOT',
            )

            self._clauses.append(clause)

        return self

    def order(self, order):
        self._order = ORDER_PB_TO_STR.get(order, DESC_STR)
        self._rev_order = REV_ORDER_PB_TO_STR.get(order, ASC_STR)
        return self

    def with_timestamp(self, timerange, order):
        # NOTE ydb uses negative timestamps, so we have to negate
        # passed limits and make reverted comparison operators in clauses
        begin, end = timerange
        if begin is not None:
            begin_msecs = -dt_to_microseconds(begin)
            if '$continuation_timestamp' in self._params and self._ct_operator == '<=':
                if begin_msecs < self._params['$continuation_timestamp']:
                    self._params['$continuation_timestamp'] = begin_msecs
            elif '$continuation_timestamp' in self._params and self._ct_operator == '<':
                if begin_msecs < self._params['$continuation_timestamp']:
                    self._params['$continuation_timestamp'] = begin_msecs
                    self._ct_operator = '<='
            else:
                self.add_clause('$timestamp_begin', self._timestamp_type, begin_msecs,
                                '`{TIMESTAMP_FLD}` <= {}'.replace('{TIMESTAMP_FLD}', self.TIMESTAMP_FLD))
        if end is not None:
            end_msecs = -dt_to_microseconds(end)
            if '$continuation_timestamp' in self._params and self._ct_operator == '>=':
                if end_msecs > self._params['$continuation_timestamp']:
                    self._params['$continuation_timestamp'] = end_msecs
            elif '$continuation_timestamp' in self._params and self._ct_operator == '>':
                if end_msecs > self._params['$continuation_timestamp']:
                    self._params['$continuation_timestamp'] = end_msecs
                    self._ct_operator = '>='
            else:
                self.add_clause('$timestamp_end', self._timestamp_type, end_msecs,
                                '`{TIMESTAMP_FLD}` >= {}'.replace('{TIMESTAMP_FLD}', self.TIMESTAMP_FLD))

        return self

    def with_table(self, table):
        self._table = table
        return self

    def limit(self, limit):
        self._limit = limit
        return self

    def get_format_args(self):
        declarations = '\n'.join(self._declarations)
        assigns = '\n'.join(self._assigns)
        clauses = ' AND '.join(self._clauses)
        where = ('WHERE ' + clauses) if clauses else ''
        ct_where = ('AND ' + clauses) if clauses else ''

        return dict(
            path=self._table_path,
            declare=declarations,
            assigns=assigns,
            where=where,
            order=self._order,
            rev_order=self._rev_order,
            limit=self._limit,
            table=self._table,
            ct_where=ct_where,
            ct_operator=self._ct_operator,
            ct_strict_operator=self._ct_strict_operator,
        )

    def build(self):
        assert self._table_path is not None, "table_path is not set"
        assert self._limit is not None, "limit is not set"
        assert self._order is not None, "order is not set"

        query = self._template.format(**self.get_format_args())
        return query, self._params, self._param_types


class QueryBuilder(QueryBuilderBase):
    TIMESTAMP_FLD = 'timestamp'

    def with_continuation_token(self, token):
        if not token:
            return self

        token = ContinuationToken.from_str(token)

        self._template = self.CT_QUERY_TPL
        self.declare('$continuation_timestamp', 'Int64')
        self.declare('$continuation_seq', 'Uint64')
        self.declare('$continuation_container', 'Utf8')
        self._params['$continuation_timestamp'] = -token.ts
        self._params['$continuation_seq'] = token.seq
        self._params['$continuation_container'] = token.container_id
        self._ct1 = '`timestamp` = $continuation_timestamp AND `container_id` = $continuation_container AND `seq` {ct_operator} $continuation_seq'
        self._ct2 = '`timestamp` = $continuation_timestamp AND `container_id` {ct_strict_operator} $continuation_container'
        self._ct3 = '`timestamp` {ct_strict_operator} $continuation_timestamp'
        self._ct_operator = token.operator
        self._ct_strict_operator = token.strict_operator

        return self

    def with_user_fields(self, fields):
        return self._with_user_fields('context', fields)

    with_hosts = QueryBuilderBase._simple_list_clause('host', 'Utf8')
    with_pods = QueryBuilderBase._simple_list_clause('pod', 'Utf8')
    with_boxes = QueryBuilderBase._simple_list_clause('box', 'Utf8')
    with_workloads = QueryBuilderBase._simple_list_clause('workload', 'Utf8')
    with_containers = QueryBuilderBase._simple_list_clause('container_id', 'Utf8')
    with_logger_names = QueryBuilderBase._simple_list_clause('logger_name', 'Utf8')
    with_log_levels = QueryBuilderBase._simple_list_clause('log_level', 'Utf8')
    with_pod_transient_fqdns = QueryBuilderBase._simple_list_clause('pod_transient_fqdn', 'Utf8')
    with_pod_persistent_fqdns = QueryBuilderBase._simple_list_clause('pod_persistent_fqdn', 'Utf8')
    with_node_fqdns = QueryBuilderBase._simple_list_clause('node_fqdn', 'Utf8')
    with_thread_names = QueryBuilderBase._simple_list_clause('thread_name', 'Utf8')
    with_request_ids = QueryBuilderBase._simple_list_clause('request_id', 'Utf8')
    with_log_levels_int = QueryBuilderBase._simple_list_clause('log_level_int', 'Int64')


class OlapQueryBuilder(QueryBuilderBase):
    TIMESTAMP_TYPE = 'Timestamp'
    TIMESTAMP_FLD = 'timestamp'

    def __init__(self):
        super(OlapQueryBuilder, self).__init__(timestamp_type=OlapQueryBuilder.TIMESTAMP_TYPE)
        self.continuation_timestamp_ms = None

    def assign_continuation_timestamp(self):
        if self.continuation_timestamp_ms is None:
            return

        timestamp_dt = microseconds_to_dt(self.continuation_timestamp_ms)
        timestamp_str = dt_to_str(timestamp_dt)
        timestamp_cast = 'CAST("' + timestamp_str + '" AS Timestamp)'

        self.assign('$continuation_timestamp', timestamp_cast)

    def with_continuation_token(self, token):
        if not token:
            return self

        token = OlapContinuationToken.from_str(token)

        self._template = self.CT_QUERY_TPL
        self.continuation_timestamp_ms = token.ts
        self.declare('$continuation_seq', 'Uint64')
        self.declare('$continuation_host', 'Utf8')
        self._params['$continuation_seq'] = token.seq
        self._params['$continuation_host'] = token.host
        self._ct_operator = token.operator
        self._ct_strict_operator = token.strict_operator

        return self

    def with_user_fields(self, fields):
        return self._with_user_fields('context', fields)

    def copy(self):
        new_builder = super(OlapQueryBuilder, self).copy()
        new_builder.continuation_timestamp_ms = self.continuation_timestamp_ms
        return new_builder

    with_hosts = QueryBuilderBase._simple_list_clause('host', 'Utf8')
    with_pods = QueryBuilderBase._simple_list_clause('pod', 'Utf8')
    with_boxes = QueryBuilderBase._simple_list_clause('box', 'Utf8')
    with_workloads = QueryBuilderBase._simple_list_clause('workload', 'Utf8')
    with_containers = QueryBuilderBase._simple_list_clause('container_id', 'Utf8')
    with_logger_names = QueryBuilderBase._simple_list_clause('logger_name', 'Utf8')
    with_log_levels = QueryBuilderBase._simple_list_clause('log_level', 'Utf8')
    with_pod_transient_fqdns = QueryBuilderBase._simple_list_clause('pod_transient_fqdn', 'Utf8')
    with_pod_persistent_fqdns = QueryBuilderBase._simple_list_clause('pod_persistent_fqdn', 'Utf8')
    with_node_fqdns = QueryBuilderBase._simple_list_clause('node_fqdn', 'Utf8')
    with_thread_names = QueryBuilderBase._simple_list_clause('thread_name', 'Utf8')
    with_request_ids = QueryBuilderBase._simple_list_clause('request_id', 'Utf8')
    with_log_levels_int = QueryBuilderBase._simple_list_clause('log_level_int', 'Int64')

    def with_timestamp(self, timerange, order):
        begin_timestamp_dt, end_timestamp_dt = timerange

        if begin_timestamp_dt is not None:
            begin_timestamp_str = dt_to_str(begin_timestamp_dt)
            begin_timestamp_cast = 'CAST("' + begin_timestamp_str + '" AS Timestamp)'

            begin_timestamp_ms = dt_to_microseconds(begin_timestamp_dt)
            if self.continuation_timestamp_ms is not None and self._ct_operator in ['>=', '>']:
                if self.continuation_timestamp_ms < begin_timestamp_ms:
                    self.continuation_timestamp_ms = begin_timestamp_ms
                    self._ct_operator = '>='
            else:
                self.add_clause_with_assign(
                    '$timestamp_begin',
                    OlapQueryBuilder.TIMESTAMP_TYPE,
                    begin_timestamp_cast,
                    '{} <= `{TIMESTAMP_FLD}`'.replace('{TIMESTAMP_FLD}', self.TIMESTAMP_FLD)
                )

        if end_timestamp_dt is not None:
            end_timestamp_str = dt_to_str(end_timestamp_dt)
            end_timestamp_cast = 'CAST("' + end_timestamp_str + '" AS Timestamp)'

            end_timestamp_ms = dt_to_microseconds(end_timestamp_dt)
            if self.continuation_timestamp_ms is not None and self._ct_operator in ['<=', '<']:
                if self.continuation_timestamp_ms > end_timestamp_ms:
                    self.continuation_timestamp_ms = end_timestamp_ms
                    self._ct_operator = '<='
            else:
                self.add_clause_with_assign(
                    '$timestamp_end',
                    OlapQueryBuilder.TIMESTAMP_TYPE,
                    end_timestamp_cast,
                    '`{TIMESTAMP_FLD}` <= {}'.replace('{TIMESTAMP_FLD}', self.TIMESTAMP_FLD)
                )

        self.assign_continuation_timestamp()

        return self


class AwacsQueryBuilder(QueryBuilderBase):
    TIMESTAMP_FLD = 'timestamp_us'

    def with_continuation_token(self, token):
        if not token:
            return self

        token = AwacsContinuationToken.from_str(token)

        self._template = self.CT_QUERY_TPL
        self.declare('$continuation_timestamp', 'Int64')
        self.declare('$continuation_pushclient_row_id', 'Int64')
        self.declare('$continuation_hostname', 'Utf8')
        self._params['$continuation_timestamp'] = -token.ts
        self._params['$continuation_pushclient_row_id'] = token.pushclient_row_id
        self._params['$continuation_hostname'] = token.hostname
        self._ct1 = '`timestamp_us` = $continuation_timestamp AND `hostname` = $continuation_hostname AND `seq` {ct_operator} $continuation_pushclient_row_id'
        self._ct2 = '`timestamp_us` = $continuation_timestamp AND `hostname` {ct_strict_operator} $continuation_hostname'
        self._ct3 = '`timestamp_us` {ct_strict_operator} $continuation_timestamp'
        self._ct_operator = token.operator
        self._ct_strict_operator = token.strict_operator

        return self

    def with_header_fields(self, fields):
        return self._with_user_fields('headers', fields)

    def with_cookie_fields(self, fields):
        return self._with_user_fields('cookies', fields)

    def with_requests(self, value_list):
        return self._with_grep(value_list, 'request', 'request')

    with_env_types = QueryBuilderBase._simple_list_clause('env_type', 'Utf8')
    with_domains = QueryBuilderBase._simple_list_clause('domain', 'Utf8')
    with_upstreams = QueryBuilderBase._simple_list_clause('upstream', 'Utf8')
    with_client_ips = QueryBuilderBase._simple_list_clause('client_ip', 'Utf8')
    with_client_ports = QueryBuilderBase._simple_list_clause('client_port', 'Uint32')
    with_hostnames = QueryBuilderBase._simple_list_clause('hostname', 'Utf8')
    with_yandexuids = QueryBuilderBase._simple_list_clause('yandexuid', 'Utf8')
    with_methods = QueryBuilderBase._simple_list_clause('method', 'Utf8')
    with_reasons = QueryBuilderBase._simple_list_clause('reason', 'Utf8')
    with_request_ids = QueryBuilderBase._simple_list_clause('request_id', 'Utf8')
    with_statuses = QueryBuilderBase._simple_list_clause('status', 'Int32')
    with_process_time_list = QueryBuilderBase._simple_list_clause('process_time', 'Double')


class SearchLogsQueryBuilder(QueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `timestamp`, `container_id`, `host`, `pod`, `box`, `workload`,
               `logger_name`, `user_id`, `request_id`, `message`,
               `log_level`, `seq`, `context`, `pod_transient_fqdn`, `pod_persistent_fqdn`,
               `node_fqdn`, `stack_trace`, `thread_name`, `log_level_int`
        FROM `{table}`
        {where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
        SELECT `timestamp`, `container_id`, `host`, `pod`, `box`, `workload`,
               `logger_name`, `user_id`, `request_id`, `message`,
               `log_level`, `seq`, `context`, `pod_transient_fqdn`, `pod_persistent_fqdn`,
               `node_fqdn`, `stack_trace`, `thread_name`, `log_level_int`
        FROM `{table}`
        WHERE
        `timestamp` = $continuation_timestamp
        AND `container_id` = $continuation_container
        AND `seq` {ct_operator} $continuation_seq
        {ct_where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit}
        );
        $q2 = (
        SELECT `timestamp`, `container_id`, `host`, `pod`, `box`, `workload`,
               `logger_name`, `user_id`, `request_id`, `message`,
               `log_level`, `seq`, `context`, `pod_transient_fqdn`, `pod_persistent_fqdn`,
               `node_fqdn`, `stack_trace`, `thread_name`, `log_level_int`
        FROM `{table}`
        WHERE
        `timestamp` = $continuation_timestamp
        AND `container_id` {ct_strict_operator} $continuation_container
        {ct_where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit}
        );
        $q3 = (
        SELECT `timestamp`, `container_id`, `host`, `pod`, `box`, `workload`,
               `logger_name`, `user_id`, `request_id`, `message`,
               `log_level`, `seq`, `context`, `pod_transient_fqdn`, `pod_persistent_fqdn`,
               `node_fqdn`, `stack_trace`, `thread_name`, `log_level_int`
        FROM `{table}`
        WHERE
        `timestamp` {ct_strict_operator} $continuation_timestamp
        {ct_where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit}
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT * from $q4
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit}
        """)


class OlapSearchLogsQueryBuilder(OlapQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `timestamp`, `container_id`, `host`, `pod`, `box`, `workload`,
               `logger_name`, `user_id`, `request_id`, `message`,
               `log_level`, `seq`, `context`, `pod_transient_fqdn`, `pod_persistent_fqdn`,
               `node_fqdn`, `stack_trace`, `thread_name`, `log_level_int`
        FROM `{table}`
        {where}
        ORDER BY `timestamp` {order}, `host` {order}, `seq` {order}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `timestamp`, `container_id`, `host`, `pod`, `box`, `workload`,
               `logger_name`, `user_id`, `request_id`, `message`,
               `log_level`, `seq`, `context`, `pod_transient_fqdn`, `pod_persistent_fqdn`,
               `node_fqdn`, `stack_trace`, `thread_name`, `log_level_int`
        FROM `{table}`
        WHERE
        (`timestamp`, `host`, `seq`) {ct_operator} ($continuation_timestamp, $continuation_host, $continuation_seq)
        {ct_where}
        ORDER BY `timestamp` {order}, `host` {order}, `seq` {order}
        LIMIT {limit}
        """)


class AwacsSearchLogsQueryBuilder(AwacsQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `timestamp_us`, `pushclient_row_id`, `env_type`, `namespace`, `domain`, `upstream`, `client_ip`,
               `client_port`, `hostname`, `cookies`, `headers`,
               `yandexuid`, `method`, `process_time`, `reason`,
               `request`, `request_id`, `status`, `workflow`
        FROM `{table}`
        {where}
        ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
            SELECT `timestamp_us`, `pushclient_row_id`, `env_type`, `namespace`, `domain`, `upstream`, `client_ip`,
                   `client_port`, `hostname`, `cookies`, `headers`,
                   `yandexuid`, `method`, `process_time`, `reason`,
                   `request`, `request_id`, `status`, `workflow`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` = $continuation_hostname
            AND `pushclient_row_id` {ct_operator} $continuation_pushclient_row_id
            {ct_where}
            ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
            LIMIT {limit}
        );
        $q2 = (
            SELECT `timestamp_us`, `pushclient_row_id`, `env_type`, `namespace`, `domain`, `upstream`, `client_ip`,
                   `client_port`, `hostname`, `cookies`, `headers`,
                   `yandexuid`, `method`, `process_time`, `reason`,
                   `request`, `request_id`, `status`, `workflow`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` {ct_strict_operator} $continuation_hostname
            {ct_where}
            ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
            LIMIT {limit}
        );
        $q3 = (
            SELECT `timestamp_us`, `pushclient_row_id`, `env_type`, `namespace`, `domain`, `upstream`, `client_ip`,
                   `client_port`, `hostname`, `cookies`, `headers`,
                   `yandexuid`, `method`, `process_time`, `reason`,
                   `request`, `request_id`, `status`, `workflow`
            FROM `{table}`
            WHERE
            `timestamp_us` {ct_strict_operator} $continuation_timestamp
            {ct_where}
            ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
            LIMIT {limit}
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT * from $q4
        ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
        LIMIT {limit}
        """)


class ContextKeysQueryBuilder(QueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `context`, `timestamp`, `container_id`, `seq`
        FROM `{table}`
        {where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
        SELECT `context`, `timestamp`, `container_id`, `seq`
        FROM `{table}`
        WHERE
        `timestamp` = $continuation_timestamp
        AND `container_id` = $continuation_container
        AND `seq` {ct_operator} $continuation_seq
        {ct_where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit};
        );
        $q2 = (
        SELECT `context`, `timestamp`, `container_id`, `seq`
        FROM `{table}`
        WHERE
        `timestamp` = $continuation_timestamp
        AND `container_id` {ct_strict_operator} $continuation_container
        {ct_where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit};
        );
        $q3 = (
        SELECT `context`, `timestamp`, `container_id`, `seq`
        FROM `{table}`
        WHERE
        `timestamp` {ct_strict_operator} $continuation_timestamp
        {ct_where}
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit};
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT * from $q4
        ORDER BY `timestamp` {rev_order}, `container_id` {rev_order}, `seq` {rev_order}
        LIMIT {limit}
        """)

    def with_key_prefix(self, prefix):
        return self


class OlapContextKeysQueryBuilder(OlapQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `context`, `timestamp`, `host`, `seq`
        FROM `{table}`
        {where}
        ORDER BY `timestamp` {order}, `host` {order}, `seq` {order}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `context`, `timestamp`, `host`, `seq`
        FROM `{table}`
        WHERE
        (`timestamp`, `host`, `seq`) {ct_operator} ($continuation_timestamp, $continuation_host, $continuation_seq)
        {ct_where}
        ORDER BY `timestamp` {order}, `host` {order}, `seq` {order}
        LIMIT {limit};
        """)

    def with_key_prefix(self, prefix):
        return self


class AwacsContextKeysQueryBuilder(AwacsQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT `{field}`, `timestamp_us`, `hostname`, `pushclient_row_id`
        FROM `{table}`
        {where}
        ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
            SELECT `{field}`, `timestamp_us`, `hostname`, `pushclient_row_id`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` = $continuation_hostname
            AND `pushclient_row_id` {ct_operator} $continuation_pushclient_row_id
            {ct_where}
            ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
            LIMIT {limit};
        );
        $q2 = (
            SELECT `{field}`, `timestamp_us`, `hostname`, `pushclient_row_id`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` = $continuation_hostname
            {ct_where}
            ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
            LIMIT {limit};
        );
        $q3 = (
            SELECT `{field}`, `timestamp_us`, `hostname`, `pushclient_row_id`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            {ct_where}
            ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
            LIMIT {limit};
            );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT * from $q4
        ORDER BY `timestamp_us` {rev_order}, `hostname` {rev_order}, `pushclient_row_id` {rev_order}
        LIMIT {limit}
        """)

    def __init__(self):
        super(AwacsContextKeysQueryBuilder, self).__init__()
        self.field = None

    def copy(self):
        new_builder = super(AwacsContextKeysQueryBuilder, self).copy()
        new_builder.field = self.field
        return new_builder

    def with_key_prefix(self, prefix):
        return self

    def with_field(self, field):
        self.field = field
        return self

    def get_format_args(self):
        assert self.field is not None
        kwargs = super(AwacsContextKeysQueryBuilder, self).get_format_args()
        kwargs['field'] = self.field
        return kwargs


class SearchSuggestsQueryBuilder(QueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{key_type}`
        FROM `{table}`
        {where}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
        SELECT DISTINCT `{key_type}`
        FROM `{table}`
        WHERE
        `timestamp` = $continuation_timestamp
        AND `container_id` = $continuation_container
        AND `seq` {ct_operator} $continuation_seq
        {ct_where}
        LIMIT {limit};
        );
        $q2 = (
        SELECT DISTINCT `{key_type}`
        FROM `{table}`
        WHERE
        `timestamp` = $continuation_timestamp
        AND `container_id` {ct_strict_operator} $continuation_container
        {ct_where}
        LIMIT {limit};
        );
        $q3 = (
        SELECT DISTINCT `{key_type}`
        FROM `{table}`
        WHERE
        `timestamp` {ct_strict_operator} $continuation_timestamp
        {ct_where}
        LIMIT {limit};
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT DISTINCT `{key_type}` from $q4
        LIMIT {limit}
        """)

    def __init__(self):
        super(SearchSuggestsQueryBuilder, self).__init__()
        self.key_type = None

    def copy(self):
        new_builder = super(SearchSuggestsQueryBuilder, self).copy()
        new_builder.key_type = self.key_type
        return new_builder

    def with_value_prefix(self, key_type, value_prefix):
        self.key_type = key_type
        if value_prefix:
            self.add_clause('$value_prefix', 'Utf8', value_prefix, 'String::StartsWithIgnoreCase(`{key_type}`, {{}})'.format(key_type=key_type))

        return self

    def get_format_args(self):
        assert self.key_type is not None
        kwargs = super(SearchSuggestsQueryBuilder, self).get_format_args()
        kwargs['key_type'] = self.key_type
        return kwargs


class OlapSearchSuggestsQueryBuilder(OlapQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{key_type}`
        FROM `{table}`
        {where}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{key_type}`
        FROM `{table}`
        WHERE
        (`timestamp`, `host`, `seq`) {ct_operator} ($continuation_timestamp, $continuation_host, $continuation_seq)
        {ct_where}
        LIMIT {limit};
        """)

    def __init__(self):
        super(OlapSearchSuggestsQueryBuilder, self).__init__()
        self.key_type = None

    def copy(self):
        new_builder = super(OlapSearchSuggestsQueryBuilder, self).copy()
        new_builder.key_type = self.key_type
        return new_builder

    def with_value_prefix(self, key_type, value_prefix):
        self.key_type = key_type
        if value_prefix:
            self.add_clause('$value_prefix', 'Utf8', value_prefix, 'String::StartsWithIgnoreCase(`{key_type}`, {{}})'.format(key_type=key_type))

        return self

    def get_format_args(self):
        assert self.key_type is not None
        kwargs = super(OlapSearchSuggestsQueryBuilder, self).get_format_args()
        kwargs['key_type'] = self.key_type
        return kwargs


class AwacsSearchSuggestsQueryBuilder(AwacsQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{field}`
        FROM `{table}`
        {where}
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
            SELECT DISTINCT `{field}`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` = $continuation_hostname
            AND `pushclient_row_id` {ct_operator} $continuation_pushclient_row_id
            {ct_where}
            LIMIT {limit};
        );
        $q2 = (
            SELECT DISTINCT `{field}`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` {ct_strict_operator} $continuation_hostname
            {ct_where}
            LIMIT {limit};
        );
        $q3 = (
            SELECT DISTINCT `{field}`
            FROM `{table}`
            WHERE
            `timestamp_us` {ct_strict_operator} $continuation_timestamp
            {ct_where}
            LIMIT {limit};
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT DISTINCT `{field}` from $q4
        LIMIT {limit}
        """)

    def __init__(self):
        super(AwacsSearchSuggestsQueryBuilder, self).__init__()
        self.field = None

    def copy(self):
        new_builder = super(AwacsSearchSuggestsQueryBuilder, self).copy()
        new_builder.field = self.field
        return new_builder

    def with_value_prefix(self, field, value_prefix):
        self.field = field
        if value_prefix:
            self.add_clause('$value_prefix', 'Utf8', value_prefix, 'String::StartsWithIgnoreCase(`{field}`, {{}})'.format(field=field))

        return self

    def get_format_args(self):
        assert self.field is not None
        kwargs = super(AwacsSearchSuggestsQueryBuilder, self).get_format_args()
        kwargs['field'] = self.field
        return kwargs


class FastSearchSuggestsQueryBuilder(SearchSuggestsQueryBuilder):
    IN_TABLE_INTERVALS = [
        datetime.timedelta(hours=1),
    ]

    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{key_type}`
        FROM (
            SELECT `{key_type}`
            FROM `{table}`
            {where}
            LIMIT 1000
        )
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
            SELECT `{key_type}`
            FROM `{table}`
            WHERE
            `timestamp` = $continuation_timestamp
            AND `container_id` = $continuation_container
            AND `seq` {ct_operator} $continuation_seq
            {ct_where}
            LIMIT 1000
        );
        $q2 = (
            SELECT `{key_type}`
            FROM `{table}`
            WHERE
            `timestamp` = $continuation_timestamp
            AND `container_id` {ct_strict_operator} $continuation_container
            {ct_where}
            LIMIT 1000
        );
        $q3 = (
            SELECT `{key_type}`
            FROM `{table}`
            WHERE
            `timestamp` {ct_strict_operator} $continuation_timestamp
            {ct_where}
            LIMIT 1000
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT DISTINCT `{key_type}`
        FROM $q4
        LIMIT {limit};
        """)

    def with_timestamp(self, timerange, order):
        # Here we set more complex limits in regards to base class,
        # because we need to limit lookup with 1h distance at most.

        begin, end = timerange
        begin = -dt_to_microseconds(begin) if begin is not None else None
        end = -dt_to_microseconds(end) if end is not None else None

        if order == dproxy_pb2.DESC:
            if end is None:
                end = -int(time.time() * 1e6)

            if begin is None:
                begin = end + int(60 * 60 * 1e6)
            else:
                begin = min(begin, end + int(60 * 60 * 1e6))

        if begin is not None:
            self.add_clause('$timestamp_begin', 'Int64', begin, '`timestamp` <= {}')
        if end is not None:
            self.add_clause('$timestamp_end', 'Int64', end, '`timestamp` >= {}')

        return self


class OlapFastSearchSuggestsQueryBuilder(OlapSearchSuggestsQueryBuilder):
    QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{key_type}`
        FROM (
            SELECT `{key_type}`
            FROM `{table}`
            {where}
            LIMIT 1000
        )
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA Kikimr.EnableLlvm="false";
        PRAGMA Kikimr.KqpPushOlapProcess="true";
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{key_type}`
        FROM(
            SELECT `{key_type}`
            FROM `{table}`
            WHERE
            (`timestamp`, `host`, `seq`) {ct_operator} ($continuation_timestamp, $continuation_host, $continuation_seq)
            {ct_where}
            LIMIT 1000
        )
        LIMIT {limit};
        """)

    def with_timestamp(self, timerange, order):
        # Here we set more complex limits in regards to base class,
        # because we need to limit lookup with 1h distance at most.

        begin_timestamp_dt, end_timestamp_dt = timerange
        begin_timestamp_ms = dt_to_microseconds(begin_timestamp_dt) if begin_timestamp_dt is not None else None
        end_timestamp_ms = dt_to_microseconds(end_timestamp_dt) if end_timestamp_dt is not None else None

        if order == dproxy_pb2.DESC:
            if end_timestamp_ms is None:
                end_timestamp_ms = int(time.time() * 1e6)

            begin_timestamp_ms_limit = end_timestamp_ms - int(60 * 60 * 1e6)
            if begin_timestamp_ms is None:
                begin_timestamp_ms = 0

            begin_timestamp_ms = max(begin_timestamp_ms, begin_timestamp_ms_limit)

        if begin_timestamp_ms is not None:
            begin_timestamp_str = dt_to_str(begin_timestamp_dt)
            begin_timestamp_cast = 'CAST("' + begin_timestamp_str + '" AS Timestamp)'

            self.add_clause_with_assign(
                '$timestamp_begin',
                OlapQueryBuilder.TIMESTAMP_TYPE,
                begin_timestamp_cast,
                '{} <= `timestamp`'
            )

        if end_timestamp_ms is not None:
            end_timestamp_str = dt_to_str(end_timestamp_dt)
            end_timestamp_cast = 'CAST("' + end_timestamp_str + '" AS Timestamp)'

            self.add_clause_with_assign(
                '$timestamp_end',
                OlapQueryBuilder.TIMESTAMP_TYPE,
                end_timestamp_cast,
                '`timestamp` <= {}'
            )

        return self


class AwacsFastSearchSuggestsQueryBuilder(AwacsSearchSuggestsQueryBuilder):
    IN_TABLE_INTERVALS = [
        datetime.timedelta(hours=1),
    ]

    QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        SELECT DISTINCT `{field}`
        FROM (
            SELECT `{field}`
            FROM `{table}`
            {where}
            LIMIT 1000
        )
        LIMIT {limit};
        """)

    CT_QUERY_TPL = textwrap.dedent("""\
        PRAGMA TablePathPrefix("{path}");
        PRAGMA AnsiInForEmptyOrNullableItemsCollections;
        {declare}
        {assigns}
        $q1 = (
            SELECT `{field}`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` = $continuation_hostname
            AND `pushclient_row_id` {ct_operator} $continuation_pushclient_row_id
            {ct_where}
            LIMIT 1000
        );
        $q2 = (
            SELECT `{field}`
            FROM `{table}`
            WHERE
            `timestamp_us` = $continuation_timestamp
            AND `hostname` {ct_strict_operator} $continuation_hostname
            {ct_where}
            LIMIT 1000
        );
        $q3 = (
            SELECT `{field}`
            FROM `{table}`
            WHERE
            `timestamp_us` {ct_strict_operator} $continuation_timestamp
            {ct_where}
            LIMIT 1000
        );
        $q4 = (SELECT * FROM $q1 UNION ALL SELECT * FROM $q2 UNION ALL SELECT * FROM $q3);
        SELECT DISTINCT `{field}`
        FROM $q4
        LIMIT {limit};
        """)

    def with_timestamp(self, timerange, order):
        # Here we set more complex limits in regards to base class,
        # because we need to limit lookup with 1h distance at most.

        begin, end = timerange
        begin = -dt_to_microseconds(begin) if begin is not None else None
        end = -dt_to_microseconds(end) if end is not None else None

        if order == dproxy_pb2.DESC:
            if end is None:
                end = -int(time.time() * 1e6)

            if begin is None:
                begin = end + int(60 * 60 * 1e6)
            else:
                begin = min(begin, end + int(60 * 60 * 1e6))

        if begin is not None:
            self.add_clause('$timestamp_begin', 'Int64', begin, '`timestamp_us` <= {}')
        if end is not None:
            self.add_clause('$timestamp_end', 'Int64', end, '`timestamp_us` >= {}')

        return self
