#include <infra/ebpf-agent/lib/config.h>
#include <infra/ebpf-agent/lib/utils.h>

#include <library/cpp/json/json_reader.h>
#include <library/cpp/protobuf/json/json2proto.h>

#include <util/stream/file.h>

#include <libbpf.h>

namespace NEbpfAgent {

    void TConfig::InitDefaults() {
        MutableCommon()->SetDebug(false);
        MutableCommon()->SetUnistatPusher(false);
        MutableCommon()->SetMlockLimitMb(400);
        MutableCommon()->SetMountCgroup2(false);

        MutablePrograms()->AddEnabled("tcp_rto");
        MutablePrograms()->SetCheckInterval(30);
        MutablePrograms()->SetDetachOnExit(true);

        MutableTcp()->SetSynRetries(4);
        MutableTcp()->SetSynAckRetries(4);
        MutableTcp()->SetOrphanRetries(3);
        MutableTcp()->SetRetries1(3);
        MutableTcp()->SetRetries2(3);

        MutableTcpRto()->SetCrossDcRto(true);
        MutableTcpRto()->SetSockMinRto(false);
        MutableTcpRto()->SetYttlEnabled(false);
        if (IsExtendedTcpCountersSupported()) {
            MutableTcpRto()->SetTcpBytesAcked(true);
            MutableTcpRto()->SetTcpBytesSent(true);
            MutableTcpRto()->SetTcpBytesRetrans(true);
            MutableTcpRto()->SetTcpBytesExtRetrans(true);
        } else {
            MutableTcpRto()->SetTcpBytesAcked(false);
            MutableTcpRto()->SetTcpBytesSent(false);
            MutableTcpRto()->SetTcpBytesRetrans(false);
            MutableTcpRto()->SetTcpBytesExtRetrans(false);
        }
        MutableTcpRto()->SetCongControl(false);

        MutablePahom()->SetHost("pahom.yandex-team.ru");
        MutablePahom()->SetPort(80);
        MutablePahom()->SetConnectTimeoutSecs(5);
        MutablePahom()->SetSocketTimeoutSecs(10);
        MutablePahom()->SetYttlBlacklistUpdateInterval(0);
    }

    TConfig::TConfig() noexcept
        : NumCpus(libbpf_num_possible_cpus())
    {
        Y_VERIFY(NumCpus > 0);
        InitDefaults();
    }

    void TConfig::LoadJson(const TString& file) {
        TFileInput input(file);
        NJson::TJsonValue json = NJson::ReadJsonFastTree(input.ReadAll());

        NProtobufJson::TJson2ProtoConfig config;
        config.SetReplaceRepeatedFields(true);
        NProtobufJson::MergeJson2Proto(json, *this, config);
    }

    void TConfig::FixMisconfig() {
        if (GetPrograms().GetCheckInterval() < 5) {
            MutablePrograms()->SetCheckInterval(5);
        }
    }

    TConfig& Config() noexcept {
        static TConfig config;
        return config;
    }

} // namespace NEbpfAgent
