#pragma once

#include <infra/ebpf-agent/lib/error.h>
#include <infra/ebpf-agent/lib/utils.h>

#include <util/generic/noncopyable.h>

#include <unistd.h>
#include <fcntl.h>

#include <bpf.h>

namespace NEbpfAgent {

    class TFd: public TNonCopyable {
    public:
        inline TFd() noexcept
            : Fd(-1)
        {
        }

        inline TFd(int fd) noexcept
            : Fd(fd)
        {
        }

        inline ~TFd()
        {
            Close();
        }

        inline TFd(TFd&& other) noexcept
            : Fd(other.Fd)
        {
            other.Fd = -1;
        }

        inline TFd& operator=(TFd&& other) noexcept {
            Close();
            Fd = other.Fd;
            other.Fd = -1;
            return *this;
        }

        inline operator int() const noexcept {
            return Fd;
        }

        void Close() const noexcept {
            if (Fd >= 0) {
                close(Fd);
            }
        }

    protected:
        int Fd;
    };

    class TBpfFd: public TFd {
    public:
        using TFd::TFd;

        ui32 GetId() const noexcept {
            return Id;
        }

        void Pin(const TString& path) {
            if (bpf_obj_pin(Fd, path.c_str()) < 0) {
                ythrow TBpfError() << "Failed to pin bpf object to " << path;
            }
        }

        static TBpfFd Get(const TString& path) {
            int fd = bpf_obj_get(path.c_str());
            if (fd < 0) {
                ythrow TBpfError() << "Failed to get bpf object from " << path;
            }
            return TBpfFd(fd);
        }

    private:
        friend TBpfFd GetProgFdById(ui32 id);
        friend TBpfFd GetMapFdById(ui32 id);
        TBpfFd(int fd, ui32 id)
            : TFd(fd)
            , Id(id)
        {
        }

        ui32 Id = 0;
    };

    inline TBpfFd GetProgFdById(ui32 id) {
        int fd = bpf_prog_get_fd_by_id(id);
        if (fd < 0) {
            ythrow TBpfError() << "Failed to get program fd by id " << id;
        }
        return TBpfFd(fd, id);
    }

    inline TBpfFd GetMapFdById(ui32 id) {
        int fd = bpf_map_get_fd_by_id(id);
        if (fd < 0) {
            ythrow TBpfError() << "Failed to get map fd by id " << id;
        }
        return TBpfFd(fd, id);
    }

    class TCgroupFd: public TFd {
    public:
        using TFd::TFd;

        const TString& GetPath() const noexcept {
            return Path;
        }

    private:
        friend TCgroupFd OpenCgroup(const TString& path);
        TCgroupFd(int fd, const TString& path)
            : TFd(fd)
            , Path(path)
        {
        }

        TString Path;
    };

    inline TCgroupFd OpenCgroup(const TString& path) {
        int fd = open(path.c_str(), O_RDONLY);
        if (fd < 0) {
            ythrow TCgroupError() << "Failed to open cgroup at " << path;
        }
        return TCgroupFd(fd, path);
    }

    inline TCgroupFd OpenCgroupRoot() {
        return OpenCgroup(FindCgroupRoot());
    }

} // namespace NEbpfAgent

template <>
inline void Out<NEbpfAgent::TBpfFd>(IOutputStream& stream, const NEbpfAgent::TBpfFd& fd) {
    stream << fd.GetId();
}

template <>
inline void Out<NEbpfAgent::TCgroupFd>(IOutputStream& stream, const NEbpfAgent::TCgroupFd& fd) {
    stream << fd.GetPath();
}
