#include <infra/ebpf-agent/lib/metrics.h>
#include <infra/ebpf-agent/lib/config.h>
#include <infra/ebpf-agent/lib/error.h>

#include <util/generic/array_size.h>

namespace {

    const NUnistat::TIntervals ErrorIntervals = { 0, 1, 2, 5, 10, 25 };

} // namespace

namespace NEbpfAgent {

    TSignal Signals[] = {
        { ESignal::OldKernel, "old_kernel", "tmmv" },
        { ESignal::LeakedCgroups, "leaked_cgroups", "tmmv" },
        { ESignal::Running, "running", "tmmv" },
        { ESignal::Disabled, "disabled", "tmmv" },

        { ESignal::BpfErrors, "bpf_errors", "ahhh", ErrorIntervals },
        { ESignal::CgroupErrors, "cgroup_errors", "ahhh", ErrorIntervals },

        //{ ESignal::NetStatRxBytesBbSas, "net_stat_rx_bytes_bb_sas", "tmmv", true },
        //{ ESignal::NetStatRxBytesBbVla, "net_stat_rx_bytes_bb_vla", "tmmv", true },
        //{ ESignal::NetStatRxBytesBbMan, "net_stat_rx_bytes_bb_man", "tmmv", true },
        //{ ESignal::NetStatRxBytesBbIva, "net_stat_rx_bytes_bb_iva", "tmmv", true },
        //{ ESignal::NetStatRxBytesBbMyt, "net_stat_rx_bytes_bb_myt", "tmmv", true },

        { ESignal::NetStatRxBytesFbSas, "net_stat_rx_bytes_fb_sas", "tmmv", true },
        { ESignal::NetStatRxBytesFbVla, "net_stat_rx_bytes_fb_vla", "tmmv", true },
        { ESignal::NetStatRxBytesFbMan, "net_stat_rx_bytes_fb_man", "tmmv", true },
        { ESignal::NetStatRxBytesFbIva, "net_stat_rx_bytes_fb_iva", "tmmv", true },
        { ESignal::NetStatRxBytesFbMyt, "net_stat_rx_bytes_fb_myt", "tmmv", true },

        //{ ESignal::NetStatTxBytesBbSas, "net_stat_tx_bytes_bb_sas", "tmmv", true },
        //{ ESignal::NetStatTxBytesBbVla, "net_stat_tx_bytes_bb_vla", "tmmv", true },
        //{ ESignal::NetStatTxBytesBbMan, "net_stat_tx_bytes_bb_man", "tmmv", true },
        //{ ESignal::NetStatTxBytesBbIva, "net_stat_tx_bytes_bb_iva", "tmmv", true },
        //{ ESignal::NetStatTxBytesBbMyt, "net_stat_tx_bytes_bb_myt", "tmmv", true },

        { ESignal::NetStatTxBytesFbSas, "net_stat_tx_bytes_fb_sas", "tmmv", true },
        { ESignal::NetStatTxBytesFbVla, "net_stat_tx_bytes_fb_vla", "tmmv", true },
        { ESignal::NetStatTxBytesFbMan, "net_stat_tx_bytes_fb_man", "tmmv", true },
        { ESignal::NetStatTxBytesFbIva, "net_stat_tx_bytes_fb_iva", "tmmv", true },
        { ESignal::NetStatTxBytesFbMyt, "net_stat_tx_bytes_fb_myt", "tmmv", true },
        
        { ESignal::TcpBytesAcked, "tcp_bytes_acked", "tmmv", true },
        { ESignal::TcpBytesSent, "tcp_bytes_sent", "tmmv", true },
        { ESignal::TcpBytesRetrans, "tcp_bytes_retrans", "tmmv", true },
        { ESignal::TcpBytesExtRetrans, "tcp_bytes_ext_retrans", "tmmv", true },
    };
    static_assert(Y_ARRAY_SIZE(Signals) == static_cast<int>(ESignal::SignalsCount));

    void StartUnistatPusher() {
        auto initializer = [](TUnistat& t) {
            auto priority = NUnistat::TPriority(10);

            for (const auto& signal: Signals) {
                if (!signal.Intervals.empty()) {
                    t.DrillHistogramHole(signal.Name, signal.Suffix, priority, signal.Intervals);
                } else {
                    t.DrillFloatHole(signal.Name, signal.Suffix, priority);
                }
            }
        };

        TUnistatPusher::TOptions options;
        options.ParseTags("itype=rtcsys;prj=ebpf-agent");

        TUnistatPusher::Instance().Start(initializer, options);
    }

    void StopUnistatPusher() {
        TUnistatPusher::Instance().Stop();
    }

    void UpdateErrorMetrics() {
        PushSignal(ESignal::BpfErrors, TBpfError::ResetCounter());
        PushSignal(ESignal::CgroupErrors, TCgroupError::ResetCounter());
    }

} // namespace NEbpfAgent
