#pragma once

#include <infra/ebpf-agent/lib/log.h>
#include <infra/ebpf-agent/lib/fd.h>

#include <util/generic/hash_set.h>

#include <dirent.h>
#include <errno.h>
#include <sched.h>

namespace NEbpfAgent {

    class TNetNamespace {
    public:
        template <typename F>
        static void ForEach(F&& f) {
            THolder<DIR, TDirDestroyer> dir(opendir("/proc"));
            if (!dir) {
                ythrow TSystemError() << "Failed to open /proc";
            }
            struct stat st;
            if (fstatat(dirfd(dir.Get()), "thread-self/ns/net", &st, 0) < 0) {
                ythrow TSystemError() << "Failed to fstatat(thread-self/ns/net)";
            }
            ino_t startNsIno = st.st_ino, curNsIno = startNsIno;
            TFd startNsFd = openat(dirfd(dir.Get()), "thread-self/ns/net", 0);
            if (startNsFd < 0) {
                ythrow TSystemError() << "Failed to open(thread-self/ns/net)";
            }
            TReverter reverter(curNsIno, startNsIno, std::move(startNsFd));
            THashSet<ino_t> processedInos(128);
            do {
                // To distinguish end of stream and from an
                // error, set errno to zero before calling readdir() and then check the
                // value of errno if NULL is returned.
                errno = 0;
                struct dirent* d = readdir(dir.Get());
                if (!d) {
                    if (errno) {
                        ythrow TSystemError() << "Failed to read /proc entry";
                    }
                    return;
                }

                if (!isdigit(*d->d_name)) {
                    continue;
                }

                char nsPath[64];
                snprintf(nsPath, sizeof(nsPath), "%s/ns/net", d->d_name);
                if (fstatat(dirfd(dir.Get()), nsPath, &st, 0) < 0) {
                    if (errno != ENOENT) {
                        auto error = LastSystemErrorText();
                        ERROR_LOG << "Failed to fstatat(" << nsPath << "): " << error << Endl;
                    }
                    continue;
                }

                ino_t nsIno = st.st_ino;
                if (!processedInos.contains(nsIno)) {
                    if (curNsIno != nsIno) {
                        if (!Enter(dirfd(dir.Get()), nsPath, nsIno)) {
                            continue;
                        }
                        curNsIno = nsIno;
                    }

                    try {
                        f(nsIno);
                    } catch (...) {
                        ERROR_LOG << CurrentExceptionMessage() << Endl;
                    }

                    processedInos.emplace(nsIno);
                }
            } while(true);
        }

    private:
        struct TDirDestroyer {
            static void Destroy(DIR* dir) {
                if (dir) {
                    closedir(dir);
                }
            }
        };

        struct TReverter {
            TReverter(ino_t& curNsIno, ino_t startNsIno, TFd&& startNsFd)
                : CurNsIno(curNsIno)
                , StartNsIno(startNsIno)
                , StartNsFd(std::move(startNsFd))
            {
            }

            ~TReverter()
            {
                if (CurNsIno != StartNsIno) {
                    Enter(StartNsFd, StartNsIno);
                }
            }

            ino_t& CurNsIno;
            ino_t StartNsIno;
            TFd StartNsFd;
        };

        static bool Enter(const TFd& nsFd, ino_t nsIno) {
            if (setns(nsFd, CLONE_NEWNET) < 0) {
                auto error = LastSystemErrorText();
                ERROR_LOG << "Failed to enter net namespace " << nsIno << ": " << error << Endl;
                return false;
            }
            INFO_LOG << "Successfully entered net namespace " << nsIno << Endl;
            return true;
        }

        static bool Enter(int dirFd, const char* nsPath, ino_t nsIno) {
            TFd nsFd = openat(dirFd, nsPath, O_RDONLY);
            if (nsFd < 0) {
                auto error = LastSystemErrorText();
                ERROR_LOG << "Failed to open(" << nsPath << "): " << error << Endl;
                return false;
            }
            return Enter(nsFd, nsIno);
        }
    };

} // namespace NEbpfAgent
