#pragma once

#include <stdbool.h>
#include <string.h>

#include <infra/ebpf-agent/lib/program.h>

#include <infra/ebpf-agent/progs/include/utils.h>
#include <infra/ebpf-agent/progs/include/defs.h>

#include <util/generic/vector.h>

namespace NEbpfAgent {

    struct TYaNetwork {
        __u32 Network;
        __u8 PrefixLen;

        EDatacenter Datacenter;
        ENetwork Type;

        __u8 LpmKeyBuf[sizeof(struct bpf_lpm_trie_key) + sizeof(__u32)];
        struct bpf_lpm_trie_key* LpmKey;
        __u32 LpmValue;

        inline TYaNetwork(__u32 network, __u8 prefixLen, EDatacenter datacenter = DC_UNKNOWN, ENetwork type = NET_UNKNOWN) noexcept
            : Network(network)
            , PrefixLen(prefixLen)
            , Datacenter(datacenter)
            , Type(type)
            , LpmKey(reinterpret_cast<struct bpf_lpm_trie_key*>(LpmKeyBuf))
        {
            LpmKey->prefixlen = PrefixLen - 32;
            for (int i = 0; i < 4; ++i) {
                LpmKey->data[i] = (Network >> (8 * i)) & 0xff;
            }

            LpmValue = ((__u32)Type << 16) | (__u32)Datacenter;
        }

        inline TYaNetwork(const TYaNetwork& other) noexcept
            : Network(other.Network)
            , PrefixLen(other.PrefixLen)
            , Datacenter(other.Datacenter)
            , Type(other.Type)
            , LpmKey(reinterpret_cast<struct bpf_lpm_trie_key*>(LpmKeyBuf))
            , LpmValue(other.LpmValue)
        {
            memcpy(LpmKeyBuf, other.LpmKeyBuf, sizeof(LpmKeyBuf));
        }

        bool operator==(const TYaNetwork& other) const {
            return Network == other.Network &&
                   PrefixLen == other.PrefixLen &&
                   Datacenter == other.Datacenter &&
                   Type == other.Type;
        }
    };

    IOutputStream& operator<<(IOutputStream& out, const TYaNetwork& net) noexcept;

    struct TYaNetworkHashFunctor {
        inline size_t operator()(const TYaNetwork& net) const noexcept {
            return MultiHash(net.Network, net.PrefixLen);
        }
    };

    using TYaNetworkSet = THashSet<TYaNetwork, TYaNetworkHashFunctor>;

    inline TYaNetworkSet GetDiff(const TYaNetworkSet& lhs, const TYaNetworkSet& rhs) {
        TYaNetworkSet result;
        CopyIf(lhs.begin(), lhs.end(), std::inserter(result, result.begin()),
            [&rhs] (const TYaNetwork& needle) {
                return rhs.find(needle) == rhs.end();
            }
        );

        return result;
    }

    extern const TVector<TYaNetwork> YaNetworks;

    extern const TVector<std::pair<__u32, ETrafficClass>> ProjectIds;

    inline bool MatchAddress(__u32 a, __u32 mask, __u32 b)
    {
        return (a & mask) == b;
    }

    inline enum EDatacenter GetDatacenter(const __u32 addr[4])
    {
        if (addr[0] != YA_NETS)
            return DC_UNKNOWN;

        for (const auto& net: YaNetworks) {
            if (MatchAddress(addr[1], MASK(net.PrefixLen - 32), net.Network)) {
                return net.Datacenter;
            }
        }

        return DC_UNKNOWN;
    }

    inline enum EDatacenter GetDatacenter(const char* addr)
    {
        unsigned char buf[sizeof(struct in6_addr)];
        if (inet_pton(AF_INET6, addr, buf) <= 0) {
            return DC_UNKNOWN;
        }
        const struct in6_addr* a = reinterpret_cast<const struct in6_addr*>(buf);
        return GetDatacenter(a->s6_addr32);
    }

} // namespace NEbpfAgent
