#include <infra/ebpf-agent/lib/pahom_client.h>
#include <infra/ebpf-agent/lib/config.h>

#include <library/cpp/logger/global/global.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/ipv6_address/ipv6_address.h>

namespace NEbpfAgent {
    TPahomClient::TPahomClient(const TString& host, ui16 port, TDuration socketTimeout, TDuration connectTimeout)
        : HttpClient(host, port, socketTimeout, connectTimeout)
    {
    }

    TYaNetworkSet TPahomClient::GetYttlBlacklistNets() {
        EDatacenter myDatacenter = GetMyDatacenter();
        TString path = "/api/v1/yttl_network_blacklist";

        if (myDatacenter != DC_UNKNOWN) {
            path.append("?dc=" + DatacenterToString(myDatacenter));
        }

        TStringStream stream;
        HttpClient.DoGet(path, &stream);

        return ParseYttlBlacklistNets(stream);
    }

    // modified code from library/cpp/ipmath/ipmath.cpp
    TMaybe<TYaNetwork> TPahomClient::StringToYaNet(const TString& str) {
        auto idx = str.rfind('/');
        if (idx == TString::npos) {
            return Nothing();
        }

        TStringBuf sb{str};
        TStringBuf address, prefix;
        sb.SplitAt(idx, address, prefix);
        prefix.Skip(1);

        __u8 prefixLen = 0;
        if (!::TryFromString(prefix, prefixLen)) {
            return Nothing();
        }

        bool ok;

        const auto netIPv6 = TIpv6Address::FromString(address, ok);
        if (!ok) {
            return Nothing();
        }

        __u32 network = HTONL(__u32(ui128(netIPv6) >> 64));

        return TYaNetwork{network, prefixLen};
    }

    TYaNetworkSet TPahomClient::ParseYttlBlacklistNets(TStringStream& stream) {
        TYaNetworkSet blacklistNets;

        const auto json(NJson::ReadJsonTree(&stream, true));

        for (const auto& deviceNetworksJson: json["device_networks"].GetArray()) {
            for (const auto& yttlBlacklistNetJson : deviceNetworksJson["networks"].GetArray()) {
                TString netString = yttlBlacklistNetJson["network"].GetString();

                if (auto net = StringToYaNet(netString)) {
                    blacklistNets.emplace(*net);
                } else {
                    ERROR_LOG << "Failed to parse yttl blacklist network=\"" << netString << "\"" << Endl;
                    continue;
                }
            }
        }

        return blacklistNets;
    }
} // namespace NEbpfAgent
