#include <infra/ebpf-agent/lib/program.h>
#include <infra/ebpf-agent/lib/metrics.h>
#include <infra/ebpf-agent/lib/config.h>
#include <infra/ebpf-agent/lib/log.h>
#include <infra/ebpf-agent/lib/nets.h>

// progs
#include <infra/ebpf-agent/progs/generated/tcp_rto.h>
#include <infra/ebpf-agent/progs/generated/tcp_rto_0b2472d.h>
#include <infra/ebpf-agent/progs/generated/tcp_rto_3a7eda2.h>
#include <infra/ebpf-agent/progs/generated/tcp_tos.h>
#include <infra/ebpf-agent/progs/generated/net_stat_rx.h>
#include <infra/ebpf-agent/progs/generated/net_stat_tx.h>
#include <infra/ebpf-agent/progs/generated/net_stat_dc_rx.h>
#include <infra/ebpf-agent/progs/generated/net_stat_dc_tx.h>
#include <infra/ebpf-agent/progs/generated/tclass_lock.h>

#include <library/cpp/json/writer/json.h>

#include <util/system/yassert.h>
#include <util/generic/array_size.h>
#include <util/string/builder.h>

#include <errno.h>

#include <bpf.h>

struct net_stat operator+(const struct net_stat& a, const struct net_stat& b)
{
    return { a.packets + b.packets, a.bytes + b.bytes };
}

struct cgroup_net_stat operator+(const struct cgroup_net_stat& a, const struct cgroup_net_stat& b)
{
    return { a.localhost + b.localhost, a.other + b.other };
}

struct cgroup_net_stat_dc operator+(const struct cgroup_net_stat_dc& a, const struct cgroup_net_stat_dc& b)
{
    struct cgroup_net_stat_dc res;
    res.localhost = a.localhost + b.localhost;
    res.other = a.other + b.other;
    for (int i = 0; i < DC__COUNT - 1; ++i) {
        for (int j = 0; j < NET__COUNT - 1; ++j) {
            res.dc[i][j] = a.dc[i][j] + b.dc[i][j];
        }
    }
    return res;
}

void WriteNetStat(NJsonWriter::TBuf& buf, struct net_stat& stat)
{
    buf
        .BeginObject()
        .WriteKey("packets")
        .WriteULongLong(stat.packets)
        .WriteKey("bytes")
        .WriteULongLong(stat.bytes)
        .EndObject();
}

TString ToString(struct cgroup_net_stat& stat)
{
    NJsonWriter::TBuf buf;
    buf.BeginObject();

    buf.WriteKey("localhost");
    WriteNetStat(buf, stat.localhost);

    buf.WriteKey("other");
    WriteNetStat(buf, stat.other);

    buf.EndObject();
    return buf.Str();
}

TString ToString(struct cgroup_net_stat_dc& stat)
{
    NJsonWriter::TBuf buf;
    buf.BeginObject();

    buf.WriteKey("localhost");
    WriteNetStat(buf, stat.localhost);

    buf.WriteKey("other");
    WriteNetStat(buf, stat.other);

    for (int i = 0; i < DC__COUNT - 1; ++i) {
        buf
            .WriteKey(::ToString(static_cast<EDatacenter>(i + 1)))
            .BeginObject();
        for (int j = 0; j < NET__COUNT - 1; ++j) {
            buf.WriteKey(::ToString(static_cast<ENetwork>(j + 1)));
            WriteNetStat(buf, stat.dc[i][j]);
        }
        buf.EndObject();
    }

    buf.EndObject();
    return buf.Str();
}

namespace NEbpfAgent {

    TBpfFd TBpfProgram::ConfigMapFd;

    const TBpfFd& TBpfProgram::InitConfigMap() noexcept {
        if (ConfigMapFd == -1) {
            ConfigMapFd = bpf_create_map(BPF_MAP_TYPE_ARRAY, sizeof(__u32), sizeof(__u32), CONF__COUNT, 0);
            Y_VERIFY(ConfigMapFd >= 0);

            const TVector<std::pair<__u32, __u32>> configs = {
                { CONF_JIFFIE_MS, GetJiffieMs() },
                { CONF_MY_DC, GetMyDatacenter() },
                { CONF_CROSS_DC_RTO, Config().GetTcpRto().GetCrossDcRto() },
                { CONF_SOCK_MIN_RTO, Config().GetTcpRto().GetSockMinRto() },
                { CONF_YTTL, Config().GetTcpRto().GetYttlEnabled() },
                { CONF_TCP_BYTES_ACKED, Config().GetTcpRto().GetTcpBytesAcked() },
                { CONF_TCP_BYTES_SENT, Config().GetTcpRto().GetTcpBytesSent() },
                { CONF_TCP_BYTES_RETRANS, Config().GetTcpRto().GetTcpBytesRetrans() },
                { CONF_TCP_BYTES_EXT_RETRANS, Config().GetTcpRto().GetTcpBytesExtRetrans() },
                { CONF_CONG_CONTROL, Config().GetTcpRto().GetCongControl() },
            };

            for (const auto &config: configs) {
                __u32 key, value;

                key = config.first;
                value = config.second;
                Y_VERIFY(!bpf_map_update_elem(ConfigMapFd, &key, &value, 0));
            }
        }
        return ConfigMapFd;
    }

    void TBpfProgram::EnableYttl() noexcept {
        static bool enabled = false;
        if (enabled) {
            return;
        }

        Y_VERIFY(ConfigMapFd >= 0);

        __u32 key = CONF_YTTL;
        __u32 value = 1;
        Y_VERIFY(!bpf_map_update_elem(ConfigMapFd, &key, &value, 0));

        enabled = true;
    }

    TBpfFd TBpfProgram::RtoMapFd;

    const TBpfFd& TBpfProgram::InitRtoMap() noexcept {
        if (RtoMapFd == -1) {
            RtoMapFd = bpf_create_map(BPF_MAP_TYPE_ARRAY, sizeof(__u32), sizeof(__u32), DC__COUNT - 1, 0);
            Y_VERIFY(RtoMapFd >= 0);

            const unsigned int jiffieMs = GetJiffieMs();

            // https://st.yandex-team.ru/RTCNETWORK-806
            const unsigned int dcRttMs[DC__COUNT - 1][DC__COUNT - 1] = {
                // SAS1   SAS2    VLA    VLX    MAN    IVA    MYT
     /* SAS1 */ {     8,     5,     8,     8,    29,    12,    12, },
     /* SAS2 */ {     5,     4,     5,     5,    24,    11,    11, },
     /*  VLA */ {     8,     5,     4,     4,    23,     6,     8, },
     /*  VLX */ {     8,     5,     4,     4,    23,     6,     8, },
     /*  MAN */ {    29,    24,    23,    23,     4,    18,    18, },
     /*  IVA */ {    12,    11,     6,     6,    18,     8,    11, },
     /*  MYT */ {    12,    11,     8,     8,    18,    11,     8, },
            };
            for (int i = 0; i < DC__COUNT - 1; ++i) {
                for (int j = 0; j < DC__COUNT - 1; ++j) {
                    Y_VERIFY(dcRttMs[i][j] > 0);
                    Y_VERIFY(dcRttMs[i][j] == dcRttMs[j][i]);
                }
            }

            const auto& myRttMs = dcRttMs[(int)GetMyDatacenter() - 1];
            for (int i = 0; i < DC__COUNT - 1; ++i) {
                __u32 key = i;
                __u32 value = (myRttMs[i] + jiffieMs - 1) / jiffieMs;
                Y_VERIFY(!bpf_map_update_elem(RtoMapFd, &key, &value, 0));
            }
        }
        return RtoMapFd;
    }

    TBpfFd TBpfProgram::YaNetsMapFd;

    const TBpfFd& TBpfProgram::InitYaNetsMap() noexcept {
        if (YaNetsMapFd == -1) {
            YaNetsMapFd = bpf_create_map(BPF_MAP_TYPE_LPM_TRIE, 8, sizeof(__u32), 10000, BPF_F_NO_PREALLOC);
            Y_VERIFY(YaNetsMapFd >= 0);

            for (const auto& net: YaNetworks) {
                Y_VERIFY(!bpf_map_update_elem(YaNetsMapFd, net.LpmKey, &net.LpmValue, 0));
            }
        }
        return YaNetsMapFd;
    }

    TBpfFd TBpfProgram::ProjectIdMapFd;

    const TBpfFd& TBpfProgram::InitProjectIdMap() noexcept {
        if (ProjectIdMapFd == -1) {
            ProjectIdMapFd = bpf_create_map(BPF_MAP_TYPE_HASH, sizeof(__u32), sizeof(__u8), 1000, 0);
            Y_VERIFY(ProjectIdMapFd >= 0);

            for (const auto& id: ProjectIds) {
                __u32 key = id.first;
                __u8 value = static_cast<__u8>(id.second);

                for (int i = 0; i < 4; ++i) {
                    key = (key >> 8) | ((key & 0xFF) << 24);
                }

                Y_VERIFY(!bpf_map_update_elem(ProjectIdMapFd, &key, &value, 0));
            }
        }
        return ProjectIdMapFd;
    }

    TBpfFd TBpfProgram::YttlBlacklistNetsMapFd;

    const TBpfFd& TBpfProgram::InitYttlBlacklistNetsMap() noexcept {
        if (YttlBlacklistNetsMapFd == -1) {
            YttlBlacklistNetsMapFd = bpf_create_map(BPF_MAP_TYPE_LPM_TRIE, 8, sizeof(__u32), 10000, BPF_F_NO_PREALLOC);
            Y_VERIFY(YttlBlacklistNetsMapFd >= 0);
        }
        return YttlBlacklistNetsMapFd;
    }

    TBpfFd TBpfProgram::TcpBytesCountersMapFd;

    const TBpfFd& TBpfProgram::InitTcpBytesCountersMap() noexcept {
        if (TcpBytesCountersMapFd == -1) {
            TcpBytesCountersMapFd = bpf_create_map(BPF_MAP_TYPE_PERCPU_ARRAY, sizeof(__u32), sizeof(__u64), TCP_BYTES__COUNT, 0);
            Y_VERIFY(TcpBytesCountersMapFd >= 0);
        }

        return TcpBytesCountersMapFd;
    }

    void TBpfProgram::InitCongMap(TBpfFd &mapFd, const TNetCongArray &dcNetCong) noexcept {
        mapFd = bpf_create_map(BPF_MAP_TYPE_ARRAY,
                               sizeof(__u32),
                               sizeof(char) * MAX_CONG_LEN,
                               (DC__COUNT - 1),
                               0);

        Y_VERIFY(mapFd >= 0);

        int myDc = (int)GetMyDatacenter();
        Y_VERIFY(myDc);
        Y_VERIFY((int)dcNetCong[myDc - 1].size() == DC__COUNT - 1);

        for (int i = 0; i < DC__COUNT - 1; ++i) {
            __u32 key = i;

            Y_VERIFY(dcNetCong[myDc - 1][i].size() < MAX_CONG_LEN);
            Y_VERIFY(!bpf_map_update_elem(mapFd, &key, dcNetCong[myDc - 1][i].c_str(), 0));
        }
    }

    TBpfFd TBpfProgram::BbCongMapFd;

    const TBpfFd& TBpfProgram::InitBbCongMap() noexcept {
        if (BbCongMapFd == -1) {
            const TNetCongArray dcBbCong = {
                //   SAS1     SAS2      VLA      VLX      MAN      IVA      MYT
     /* SAS1 */ {      "",      "",      "",      "",      "",      "",      "", },
     /* SAS2 */ {      "",      "",      "",      "",      "",      "",      "", },
     /*  VLA */ {      "",      "",      "",      "",      "",      "",      "", },
     /*  VLX */ {      "",      "",      "",      "",      "",      "",      "", },
     /*  MAN */ {      "",      "",      "",      "",      "",      "",      "", },
     /*  IVA */ {      "",      "",      "",      "",      "",      "",      "", },
     /*  MYT */ {      "",      "",      "",      "",      "",      "",      "", },
            };

            InitCongMap(BbCongMapFd, dcBbCong);
        }
        return BbCongMapFd;
    }

    TBpfFd TBpfProgram::FbCongMapFd;

    const TBpfFd& TBpfProgram::InitFbCongMap() noexcept {
        if (FbCongMapFd == -1) {
            const TNetCongArray dcFbCong = {
                //   SAS1     SAS2      VLA      VLX      MAN      IVA      MYT
     /* SAS1 */ {      "",   "bbr",   "bbr",   "bbr",   "bbr",   "bbr",   "bbr", },
     /* SAS2 */ {   "bbr",      "",   "bbr",   "bbr",   "bbr",   "bbr",   "bbr", },
     /*  VLA */ {   "bbr",   "bbr",      "",      "",   "bbr",   "bbr",   "bbr", },
     /*  VLX */ {   "bbr",   "bbr",      "",      "",   "bbr",   "bbr",   "bbr", },
     /*  MAN */ {   "bbr",   "bbr",   "bbr",   "bbr",      "",   "bbr",   "bbr", },
     /*  IVA */ {   "bbr",   "bbr",   "bbr",   "bbr",   "bbr",      "",   "bbr", },
     /*  MYT */ {   "bbr",   "bbr",   "bbr",   "bbr",   "bbr",   "bbr",      "", },
            };

            InitCongMap(FbCongMapFd, dcFbCong);
        }
        return FbCongMapFd;
    }

    void TBpfProgram::TryReuseMap(const TStringBuf& name, TSharedMapInitFunc initFunc) const noexcept {
        auto map = bpf_object__find_map_by_name(Obj, name.data());
        if (map) {
            const auto& fd = initFunc();
            Y_VERIFY(!bpf_map__reuse_fd(map, fd));
        }
    }

    TBpfProgram::TBpfProgram(const char* name, const unsigned char* bytes, unsigned int size,
                             const char* minKernelVersion, bool enabled) noexcept
        : Name(name)
        , MinKernelVersion(minKernelVersion)
        , Enabled(CompareKernelVersion(MinKernelVersion) >= 0 && (enabled || IsEnabledByConfig(Name)))
        , Obj(bpf_object__open_mem(bytes, size, nullptr))
        , Prog(bpf_program__next(nullptr, Obj))
        , Title(bpf_program__title(Prog, false))
    {
        Y_VERIFY(Obj);
        Y_VERIFY(Prog);
        Y_VERIFY(!libbpf_prog_type_by_name(Title.data(), &Type, &ExpectedAttachType));
        Y_VERIFY(!libbpf_attach_type_by_name(Title.data(), &AttachType));
    }

    TBpfProgram::~TBpfProgram()
    {
        bpf_object__close(Obj); // calls bpf_object__unload()
        Fd = -1; // fd already closed by bpf_object__close()
    }

    void TBpfProgram::Load() const noexcept {
        if (Enabled && Fd < 0) {
            TryReuseMap("config_map", InitConfigMap);
            TryReuseMap("rto_map", InitRtoMap);
            TryReuseMap("ya_nets_map", InitYaNetsMap);
            TryReuseMap("project_id_map", InitProjectIdMap);
            TryReuseMap("yttl_blacklist_nets_map", InitYttlBlacklistNetsMap);
            TryReuseMap("tcp_bytes_counters_map", InitTcpBytesCountersMap);
            TryReuseMap("bb_cong_map", InitBbCongMap);
            TryReuseMap("fb_cong_map", InitFbCongMap);

            Y_VERIFY(!bpf_object__load(Obj));
            Fd = bpf_program__fd(Prog);
            Y_VERIFY(Fd >= 0);
            ui32 len = sizeof(Info);
            Y_VERIFY(!bpf_obj_get_info_by_fd(Fd, &Info, &len));
        }
    }

    TBpfFd TBpfProgram::GetMap(const TStringBuf& name) const noexcept {
        return bpf_object__find_map_fd_by_name(Obj, name.data());
    }

    bool TBpfProgram::IsEnabledByConfig(const TStringBuf& name) {
        for (const auto& enabled: Config().GetPrograms().GetEnabled()) {
            if (name == enabled) {
                return true;
            }
        }
        return false;
    }

    struct bpf_prog_info TBpfProgram::GetProgInfo(const TBpfFd& fd, TVector<ui32>* mapIds) {
        struct bpf_prog_info info = {};
        ui32 len = sizeof(info);
        if (mapIds) {
            info.nr_map_ids = mapIds->size();
            info.map_ids = reinterpret_cast<ui64>(mapIds->data());
        }
        if (bpf_obj_get_info_by_fd(fd, &info, &len)) {
            ythrow TBpfError() << "Failed to get info about program " << fd;
        }
        return info;
    }

    struct bpf_map_info TBpfProgram::GetMapInfo(const TBpfFd& fd) {
        struct bpf_map_info info = {};
        ui32 len = sizeof(info);
        if (bpf_obj_get_info_by_fd(fd, &info, &len)) {
            ythrow TBpfError() << "Failed to get info about map " << fd;
        }
        return info;
    }

    std::size_t TBpfProgram::ListLoadedProgs(TVector<struct bpf_prog_info>* progs) {
        std::size_t count = 0;

        ui32 id = 0;
        while (true) {
            if (bpf_prog_get_next_id(id, &id)) {
                if (errno == ENOENT) {
                    break;
                }
                ythrow TBpfError() << "Failed to get next program";
            }

            ++count;

            if (progs) {
                progs->emplace_back(GetProgInfo(GetProgFdById(id)));
            }
        }

        return count;
    }

    void TCgroupBpfProgram::TAttachedProgram::Detach() const {
        if (Id) {
            try {
                TCgroupBpfProgram::Detach(GetProgFdById(Id), AttachPath, Parent.AttachType);
                Id = 0;
                AttachPath.clear();
            } catch (...) {
                ERROR_LOG << CurrentExceptionMessage() << Endl;
            }
        }
    }

    TCgroupBpfProgram::~TCgroupBpfProgram() {
        if (Config().GetPrograms().GetDetachOnExit()) {
            Detach();
        }
    }

    void TCgroupBpfProgram::Attach(const TBpfFd& progFd, const TCgroupFd& cgFd, enum bpf_attach_type type, unsigned int flags) {
        if (bpf_prog_attach(progFd, cgFd, type, flags)) {
            ythrow TBpfError() << "Failed to attach program " << progFd << " to cgroup " << cgFd;
        }
    }

    void TCgroupBpfProgram::Attach(const TCgroupFd& cgFd) const {
        Attach(Fd, cgFd, AttachType, AttachFlags);
        Children.emplace(*this, Info.id, cgFd.GetPath());
    }

    void TCgroupBpfProgram::Detach(const TBpfFd& progFd, const TCgroupFd& cgFd, enum bpf_attach_type type) {
        if (bpf_prog_detach2(progFd, cgFd, type)) {
            ythrow TBpfError() << "Failed to detach program " << progFd << " from cgroup " << cgFd;
        }
    }

    TVector<struct bpf_prog_info> TCgroupBpfProgram::ListAttachedProgs(const TCgroupFd& cgFd, enum bpf_attach_type type) {
        TVector<struct bpf_prog_info> progs;

        constexpr int BPF_CGROUP_MAX_PROGS = 64; // from kernel/bpf/cgroup.c
        ui32 ids[BPF_CGROUP_MAX_PROGS];
        ui32 flags, count = Y_ARRAY_SIZE(ids);
        if (bpf_prog_query(cgFd, type, 0, &flags, ids, &count)) {
            if (errno == EINVAL) {
                return progs;
            }
            ythrow TBpfError() << "Failed to query programs attached to cgroup " << cgFd;
        }

        for (ui32 i = 0; i < count; ++i) {
            progs.emplace_back(GetProgInfo(GetProgFdById(ids[i])));
        }

        return progs;
    }

    THashMap<enum bpf_attach_type, TVector<struct bpf_prog_info>> TCgroupBpfProgram::ListAttachedProgs(const TCgroupFd& cgFd) {
        THashMap<enum bpf_attach_type, TVector<struct bpf_prog_info>> result;
        for (int i = 0; i < __MAX_BPF_ATTACH_TYPE; ++i) {
            enum bpf_attach_type type = static_cast<enum bpf_attach_type>(i);
            auto progs = ListAttachedProgs(cgFd, type);
            if (progs) {
                result[type] = std::move(progs);
            }
        }
        return result;
    }

    void TCgroupBpfProgram::Check(const TVector<const TCgroupBpfProgram*>& knownProgs, const TCgroupFd& cgFd, bool repair) {
        auto attachedProgs = TCgroupBpfProgram::ListAttachedProgs(cgFd);
        for (const auto& e: attachedProgs) {
            const auto& progs = e.second;
            INFO_LOG << "Found " << progs.size() << " programs attached to cgroup " << cgFd << " with type " << ::ToString(static_cast<EBpfAttachType>(e.first)) << Endl;
            for (const auto& prog: progs) {
                DEBUG_LOG << "Prog:\n" << prog << Endl;
            }
        }

        for (const auto* knownProg: knownProgs) {
            try {
                auto attachType = knownProg->AttachType;
                bool found = false;
                for (const auto& prog: attachedProgs[attachType]) {
                    if (*knownProg == prog) {
                        found = true;
                        knownProg->AddChild(prog, cgFd.GetPath());
                        if (repair) {
                            if (!knownProg->Enabled) {
                                INFO_LOG << "Detach disabled " << knownProg->Name << " program from cgroup " << cgFd << Endl;
                                knownProg->Detach();
                            } else if (knownProg->IsOutdated(prog)) {
                                INFO_LOG << "Found outdated " << knownProg->Name << " program at cgroup " << cgFd << ", reattaching" << Endl;
                                knownProg->Detach();
                                knownProg->Attach(cgFd);
                            }
                        }
                    }
                }
                if (!found && knownProg->Enabled && repair) {
                    INFO_LOG << "Missed " << knownProg->Name << " program at cgroup " << cgFd << ", attaching" << Endl;
                    knownProg->Attach(cgFd);
                }
            } catch (...) {
                ERROR_LOG << CurrentExceptionMessage() << Endl;
            }
        }
    }

    const TCgroupBpfProgram& TCgroupBpfProgram::TcpTos() {
        static const TCgroupBpfProgram prog(BPF_TCP_TOS_NAME, bpf_tcp_tos_bytes, bpf_tcp_tos_size, BPF_F_ALLOW_MULTI);
        return prog;
    }

    const TTcpRtoBpfProgram& TTcpRtoBpfProgram::TcpRto() {
        struct TVersion {
            TString minKernel;
            const unsigned char* bytes;
            unsigned int size;
        };

        // More recent kernel versions should be added in the beginning
        static const TVector<struct TVersion> tcpRtoVersions = {
            {"5.4.187-35.2", bpf_tcp_rto_3a7eda2_bytes, sizeof(bpf_tcp_rto_3a7eda2_bytes)},
            {"5.4.180-31", bpf_tcp_rto_0b2472d_bytes, sizeof(bpf_tcp_rto_0b2472d_bytes)},
            {"", bpf_tcp_rto_bytes, sizeof(bpf_tcp_rto_bytes)},
        };

        static const TTcpRtoBpfProgram prog = [&]
            {
                for (const auto &version: tcpRtoVersions)
                    if (CompareKernelVersion(version.minKernel) >= 0) {
                        INFO_LOG << "Using " << (version.minKernel.empty() ? "default" : version.minKernel) << " tcp_rto" << Endl;
                        return TTcpRtoBpfProgram(BPF_TCP_RTO_NAME, version.bytes, version.size, BPF_F_ALLOW_MULTI);
                    }
                Y_VERIFY(0, "Could not pick correct tcp_rto version");
            } ();

        return prog;
    }

    bool TTcpRtoBpfProgram::IsAnyTcpCounterEnabled() noexcept {
        return Config().GetTcpRto().GetTcpBytesAcked() ||
               Config().GetTcpRto().GetTcpBytesSent() ||
               Config().GetTcpRto().GetTcpBytesRetrans() ||
               Config().GetTcpRto().GetTcpBytesExtRetrans();
    }

    void TTcpRtoBpfProgram::UpdateMetrics() {
        if (TcpRto().Enabled) {
            for (__u32 i = 0; i < TCP_BYTES__COUNT; ++i) {
                static TVector<__u64> perCpuStats(Config().NumCpus);
                __u32 key = i;
                Y_VERIFY(!bpf_map_lookup_elem(TBpfProgram::TcpBytesCountersMapFd, &key, perCpuStats.data()));

                __u64 totalStat = 0;
                for (const auto& perCpuStat: perCpuStats) {
                    totalStat += perCpuStat;
                }

                if (Config().GetTcpRto().GetTcpBytesAcked() && key == TCP_BYTES_ACKED) {
                    if (Config().GetCommon().GetDebug()) {
                        DEBUG_LOG << "Bytes Acked: " << totalStat << Endl;
                    }

                    PushSignal(ESignal::TcpBytesAcked, totalStat);
                } else if (Config().GetTcpRto().GetTcpBytesSent() && key == TCP_BYTES_SENT) {
                    if (Config().GetCommon().GetDebug()) {
                        DEBUG_LOG << "Bytes Sent: " << totalStat << Endl;
                    }

                    PushSignal(ESignal::TcpBytesSent, totalStat);
                } else if (Config().GetTcpRto().GetTcpBytesRetrans() && key == TCP_BYTES_RETRANS) {
                    if (Config().GetCommon().GetDebug()) {
                        DEBUG_LOG << "Bytes Retransmitted: " << totalStat << Endl;
                    }

                    PushSignal(ESignal::TcpBytesRetrans, totalStat);
                } else if (Config().GetTcpRto().GetTcpBytesExtRetrans() && key == TCP_BYTES_EXT_RETRANS) {
                    if (Config().GetCommon().GetDebug()) {
                        DEBUG_LOG << "Bytes Retransmitted (external): " << totalStat << Endl;
                    }

                    PushSignal(ESignal::TcpBytesExtRetrans, totalStat);
                }
            }
        }
    }

    const TNetStatBpfProgram& TNetStatBpfProgram::Rx(bool forceEnable, bool allowMulti) {
        static const TNetStatBpfProgram prog(BPF_NET_STAT_RX_NAME, bpf_net_stat_rx_bytes, bpf_net_stat_rx_size,
                                             forceEnable, allowMulti ? BPF_F_ALLOW_MULTI : 0);
        return prog;
    }

    const TNetStatBpfProgram& TNetStatBpfProgram::Tx(bool forceEnable, bool allowMulti) {
        static const TNetStatBpfProgram prog(BPF_NET_STAT_TX_NAME, bpf_net_stat_tx_bytes, bpf_net_stat_tx_size,
                                             forceEnable, allowMulti ? BPF_F_ALLOW_MULTI : 0);
        return prog;
    }

    const TNetStatDcBpfProgram& TNetStatDcBpfProgram::Rx(bool forceEnable, bool allowMulti) {
        static const TNetStatDcBpfProgram prog(BPF_NET_STAT_DC_RX_NAME, bpf_net_stat_dc_rx_bytes, bpf_net_stat_dc_rx_size,
                                               forceEnable, allowMulti ? BPF_F_ALLOW_MULTI : 0);
        return prog;
    }

    const TNetStatDcBpfProgram& TNetStatDcBpfProgram::Tx(bool forceEnable, bool allowMulti) {
        static const TNetStatDcBpfProgram prog(BPF_NET_STAT_DC_TX_NAME, bpf_net_stat_dc_tx_bytes, bpf_net_stat_dc_tx_size,
                                               forceEnable, allowMulti ? BPF_F_ALLOW_MULTI : 0, false);
        return prog;
    }

    struct cgroup_net_stat_dc TNetStatDcBpfProgram::GetStats() const {
        static TVector<struct cgroup_net_stat_dc> perCpuStats(Config().NumCpus);

        struct cgroup_net_stat_dc stat = {};

        auto cgroupId = GetCgroupId(FindCgroupRoot());
        struct bpf_cgroup_storage_key key = { cgroupId, AttachType };
        if (bpf_map_lookup_elem(StatMapFd, &key, perCpuStats.data()) < 0) {
            return stat;
        }

        for (const auto& perCpuStat: perCpuStats) {
            stat = stat + perCpuStat;
        }

        return stat;
    }

    void TNetStatDcBpfProgram::UpdateMetrics() {
        if (Rx().Enabled) {
            auto stats = Rx().GetStats();
            if (Config().GetCommon().GetDebug()) {
                DEBUG_LOG << "Rx stats: " << ToString(stats) << Endl;
            }

            //PushSignal(ESignal::NetStatRxBytesBbSas, stats.dc[DC_SAS1 - 1][NET_BACKBONE - 1].bytes + stats.dc[DC_SAS2 - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatRxBytesBbVla, stats.dc[DC_VLA - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatRxBytesBbMan, stats.dc[DC_MAN - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatRxBytesBbIva, stats.dc[DC_IVA - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatRxBytesBbMyt, stats.dc[DC_MYT - 1][NET_BACKBONE - 1].bytes);

            PushSignal(ESignal::NetStatRxBytesFbSas, stats.dc[DC_SAS1 - 1][NET_FASTBONE - 1].bytes + stats.dc[DC_SAS2 - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatRxBytesFbVla, stats.dc[DC_VLA - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatRxBytesFbMan, stats.dc[DC_MAN - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatRxBytesFbIva, stats.dc[DC_IVA - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatRxBytesFbMyt, stats.dc[DC_MYT - 1][NET_FASTBONE - 1].bytes);
        }
        if (Tx().Enabled) {
            auto stats = Tx().GetStats();
            if (Config().GetCommon().GetDebug()) {
                DEBUG_LOG << "Tx stats: " << ToString(stats) << Endl;
            }

            //PushSignal(ESignal::NetStatTxBytesBbSas, stats.dc[DC_SAS1 - 1][NET_BACKBONE - 1].bytes + stats.dc[DC_SAS2 - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatTxBytesBbVla, stats.dc[DC_VLA - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatTxBytesBbMan, stats.dc[DC_MAN - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatTxBytesBbIva, stats.dc[DC_IVA - 1][NET_BACKBONE - 1].bytes);
            //PushSignal(ESignal::NetStatTxBytesBbMyt, stats.dc[DC_MYT - 1][NET_BACKBONE - 1].bytes);

            PushSignal(ESignal::NetStatTxBytesFbSas, stats.dc[DC_SAS1 - 1][NET_FASTBONE - 1].bytes + stats.dc[DC_SAS2 - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatTxBytesFbVla, stats.dc[DC_VLA - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatTxBytesFbMan, stats.dc[DC_MAN - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatTxBytesFbIva, stats.dc[DC_IVA - 1][NET_FASTBONE - 1].bytes);
            PushSignal(ESignal::NetStatTxBytesFbMyt, stats.dc[DC_MYT - 1][NET_FASTBONE - 1].bytes);
        }
    }

    TString TCgroupRootBpfProgram::JugglerCheck() {
        try {
            auto check = CheckPrerequisites(false);
            auto error = std::get<0>(check);
            if (error) {
                return "";
            }

            auto enabledNames(Config().GetPrograms().GetEnabled());

            for (const auto& prog: TCgroupRootBpfProgram::KnownProgs()) {
                if (CompareKernelVersion(prog->GetMinKernelVersion()) < 0) {
                    Erase(enabledNames, prog->GetName());
                }
            }

            auto attachedProgs = TCgroupBpfProgram::ListAttachedProgs(OpenCgroupRoot());
            for (const auto& e: attachedProgs) {
                for (const auto& prog: e.second) {
                    if (TBpfProgram::IsEnabledByConfig(prog.name)) {
                        Erase(enabledNames, prog.name);
                    } else {
                        return TStringBuilder() << "Disabled program " << prog.name << " is online at root cgroup";
                    }
                }
            }

            if (!enabledNames.empty()) {
                return TStringBuilder() << "Program " << enabledNames[0] << " is missed at root cgroup";
            }

            return "";
        } catch (...) {
            return CurrentExceptionMessage();
        }
    }

    const TCgroupBpfProgram& TCgroupBpfProgram::TclassLock() {
        static const TCgroupBpfProgram prog(BPF_TCLASS_LOCK_NAME, bpf_tclass_lock_bytes, bpf_tclass_lock_size, "5.4", false, BPF_F_ALLOW_MULTI);
        return prog;
    }

    const TVector<const TCgroupBpfProgram*>& TCgroupRootBpfProgram::KnownProgs() noexcept {
        static const TVector<const TCgroupBpfProgram*> progs({
                &TcpTos(),  // TcpTos first so that TC is set before YaTTL
                &TTcpRtoBpfProgram::TcpRto(),
                &TNetStatDcBpfProgram::Rx(),
                &TNetStatDcBpfProgram::Tx(),
                &TclassLock(),
        });
        return progs;
    }

    const TVector<const TBpfProgram*>& TBpfProgram::KnownProgs() noexcept {
        static TVector<const TBpfProgram*> progs(TCgroupRootBpfProgram::KnownProgs().cbegin(), TCgroupRootBpfProgram::KnownProgs().cend());
        return progs;
    }

} // namespace NEbpfAgent
