#pragma once

#include <linux/bpf.h>

#include "defs.h"

struct net_stat {
    __u64 packets;
    __u64 bytes;
};

struct cgroup_net_stat {
    struct net_stat localhost;
    struct net_stat other;
};

struct cgroup_net_stat_dc {
    struct net_stat localhost;
    struct net_stat other;
    struct net_stat dc[DC__COUNT_POW2][NET__COUNT_POW2];
};

#ifndef __cplusplus

static __always_inline int net_stat(struct __sk_buff *skb, struct bpf_map_def *map)
{
    struct cgroup_net_stat *cg;
    struct net_stat *stat;

    cg = bpf_get_local_storage(map, 0);
    stat = &cg->other;
    if (skb->ifindex == 1)
        stat = &cg->localhost;

    ++stat->packets;
    stat->bytes += skb->len;

    return 1;
}

#ifdef NET_STAT_DC
#include <sys/socket.h>

#include <bpf_utils.h>

static __always_inline int net_stat_dc(struct __sk_buff *skb, struct bpf_map_def *map)
{
    struct cgroup_net_stat_dc *cg;
    struct net_stat *stat;

    cg = bpf_get_local_storage(map, 0);
    stat = &cg->other;
    if (skb->ifindex == 1)
        stat = &cg->localhost;
    else if (skb->family == AF_INET6) {
        __u32 dc_net = get_dc_net(skb->remote_ip6);

        if (dc_net) {
            enum EDatacenter dc = (__u16)dc_net;
            enum ENetwork net = (__u16)(dc_net >> 16);
            stat = &cg->dc[(dc - 1) % DC__COUNT_POW2][(net - 1) % NET__COUNT_POW2];
        }
    }

    ++stat->packets;
    stat->bytes += skb->len;

    return 1;
}
#endif // NET_STAT_DC

#endif // !__cplusplus
