#pragma once

#include <bpf_endian.h>

// from linux/compiler.h
#define likely(x)   __builtin_expect(!!(x), 1)
#define unlikely(x) __builtin_expect(!!(x), 0)

#define __optnone __attribute__((optnone))

#define ARRAY_SIZE(a) (sizeof(a) / sizeof(*a))
#define HTONL         bpf_htonl
#define MASK(n)       HTONL(~(((__u32)1 << (32 - (n))) - 1))
_Static_assert(MASK(1) == HTONL(0x80000000), "MASK macro is broken");
_Static_assert(MASK(32) == HTONL(0xffffffff), "MASK macro is broken");

#define YA_NETS HTONL(0x2a0206b8)

static __always_inline bool match(__u32 a, __u32 mask, __u32 b)
{
    return (a & mask) == b;
}

static __always_inline bool is_nat64(const __u32 _addr[4])
{
    const volatile __u32 *addr = (const volatile __u32 *)_addr;
    __u32 a = addr[0];

    // NAT64 64:ff9b::/64
    if (unlikely(a == HTONL(0x0064ff9b))) {
        a = addr[1];
        return match(a, MASK(64 - 32), HTONL(0x00000000));
    // https://racktables.yandex-team.ru/index.php?page=ipv6net&tab=default&id=40755
    } else if (unlikely(a == HTONL(0x2a0206bc))) // NAT64 2a02:6bc::/32
        return true;

    return false;
}

static __always_inline bool is_extra_l3_addr(const __u32 _addr[4])
{
    const volatile __u32 *addr = (const volatile __u32 *)_addr;
    __u32 a = addr[0];

    if (unlikely(a == HTONL(0x2620010f))) {
        a = addr[1];
        return match(a, MASK(48 - 32), HTONL(0xd0010000)); // L3 2620:10f:d001::/48
    }

    return false;
}

static __always_inline bool is_l3_addr(__u32 a)
{
    return match(a, MASK(64 - 32), HTONL(0x00003400)) || // L3 2a02:6b8:0:3400::/64
           match(a, MASK(62 - 32), HTONL(0x00000000)) || // L3 2a02:6b8::/62
           match(a, MASK(46 - 32), HTONL(0x00040000)) || // L3 2a02:6b8:4::/46
           match(a, MASK(45 - 32), HTONL(0x00080000)) || // L3 2a02:6b8:8::/45
           match(a, MASK(43 - 32), HTONL(0x00200000)) || // L3 2a02:6b8:20::/43
           match(a, MASK(48 - 32), HTONL(0xb01e0000));   // L3 2a02:6b8:b01e::/48
}

static __always_inline bool is_tun64(__u32 a)
{
    // https://racktables.yandex-team.ru/index.php?page=ipv6net&id=7302
    return match(a, MASK(64 - 32), HTONL(0xb010a0ff)); // TUN64 2a02:6b8:b010:a0ff::/64
}

static __always_inline bool is_stateful_addr(const __u32 _addr[4])
{
    const volatile __u32 *addr = (const volatile __u32 *)_addr;
    __u32 a = addr[0];

    if (is_nat64(_addr) || is_extra_l3_addr(_addr))
        return true;

    if (unlikely(a != YA_NETS))
        return false;

    a = addr[1];
    return (is_l3_addr(a) || is_tun64(a));
}
