#include <infra/ebpf-agent/src/daemon.h>
#include <infra/ebpf-agent/lib/program.h>
#include <infra/ebpf-agent/lib/metrics.h>
#include <infra/ebpf-agent/lib/config.h>
#include <infra/ebpf-agent/lib/utils.h>
#include <infra/ebpf-agent/lib/timer.h>
#include <infra/ebpf-agent/lib/log.h>

#include <infra/porto/api/libporto.hpp>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/logger/rotating_file.h>
#include <library/cpp/logger/stream.h>
#include <library/cpp/logger/filter.h>
#include <library/cpp/logger/null.h>

#include <util/system/interrupt_signals.h>
#include <util/system/fs.h>
#include <util/random/random.h>
#include <util/stream/file.h>

#include <sys/mount.h>

namespace {

    static const TDuration YASM_INTERVAL = TDuration::Seconds(5);

    void SignalHandler(int signum) {
        INFO_LOG << "Stopped by signal " << signum << Endl;
        NEbpfAgent::Daemon().Stop();
    }

    bool IsDisabled() {
        static const TString path("/var/run/ebpf-agent.disabled");
        return NFs::Exists(path);
    }

    void Cleanup() {
        using namespace NEbpfAgent;
        MeasureExecTime("Cgroup root programs check", TCgroupRootBpfProgram::Check, false);
        MeasureExecTime("Detach known programs from root cgroup", TCgroupRootBpfProgram::DetachKnownProgs);
        MeasureExecTime("Cleanup net namespaces", CleanupNetNamespaces);
    }

    TString GetRunningConfig() {
        Porto::TPortoApi portoApi;
        TString command;
        if (!portoApi.SetTimeout(10) &&
            !portoApi.Connect() &&
            !portoApi.GetProperty("ebpf-agent", "command", command))
        {
            TVector<TString> args = StringSplitter(command).Split(' ');
            for (auto it = std::cbegin(args); it != std::cend(args); ++it) {
                if (*it == "-c" || *it == "--config") {
                    ++it;
                    if (it != std::cend(args)) {
                        return *it;
                    }
                }
            }
        }
        return "";
    }

} // namespace

namespace NEbpfAgent {

    TDaemon::TDaemon() noexcept
        : BpfWorker(TThread::TParams(BpfWorkerLoopWrapper, this).SetName("ebpf-worker"))
        , PahomUpdater(TThread::TParams(PahomUpdaterLoopWrapper, this).SetName("pahom-updater"))
        , PahomClient(Config().GetPahom().GetHost(),
                      Config().GetPahom().GetPort(),
                      TDuration::Seconds(Config().GetPahom().GetSocketTimeoutSecs()),
                      TDuration::Seconds(Config().GetPahom().GetConnectTimeoutSecs()))
    {
    }

    int TDaemon::RunCommand() {
        switch (Command) {
        case JugglerCheck:
            if (auto error = TCgroupRootBpfProgram::JugglerCheck()) {
                Cout << "PASSIVE-CHECK:ebpf_agent;CRIT;" << error;
            } else {
                Cout << "PASSIVE-CHECK:ebpf_agent;OK;";
            }
            break;
        case Cleanup:
            ::Cleanup();
            break;
        case GetDatacenter:
            Cout << ::ToString(NEbpfAgent::GetDatacenter(CommandArgs[0].c_str())).substr(3) << Endl;
            break;
        default:
            Y_VERIFY(false);
        }
        return 0;
    }

    TDaemon& TDaemon::ParseCmdline(int argc, char* argv[]) {
        using namespace NLastGetopt;

        TOpts opts(TOpts::Default());
        opts.SetFreeArgTitle(0, "command",
                "Available commands:"
                "\n\t\tjuggler-check    Checks that all necessary eBPF programs are online"
                "\n\t\tcleanup          Force detach all known programs and cleanup net namespaces"
                "\n\t\tget-dc           Resolve IPv6 address to Yandex datacenter"
                );
        opts.AddHelpOption('h');
        opts.AddVersionOption('v');

        TOpt& configOpt = opts.AddLongOption('c', "config", "Path to JSON config file").DefaultValue("/usr/share/ebpf-agent/config.json");
        TOpt& logOpt = opts.AddLongOption('l', "log", "Path to log file");
        TOpt& forceOpt = opts.AddLongOption('f', "force", "Force run (do not check all prerequisites)"); forceOpt.Optional().HasArg(NO_ARGUMENT);
        TOpt& crossDcRtoOpt = opts.AddLongOption("cross-dc-rto", "Minimize cross dc RTO too"); crossDcRtoOpt.Optional().HasArg(NO_ARGUMENT);
        TOpt& sockMinRtoOpt = opts.AddLongOption("sock-min-rto", "Minimize RTO on socket"); sockMinRtoOpt.Optional().HasArg(NO_ARGUMENT);
        TOpt& debugOpt = opts.AddLongOption("debug", "Enable debug output"); debugOpt.Optional().HasArg(NO_ARGUMENT);

        TOptsParseResult parsedOpts(&opts, argc, argv);

        TString config = parsedOpts.Get(&configOpt);
        ForceRun = parsedOpts.Has(&forceOpt);

        const auto& freeArgs = parsedOpts.GetFreeArgs();
        if (!freeArgs.empty()) {
            auto command = freeArgs[0];
            if (freeArgs.size() > 1) {
                CommandArgs.insert(CommandArgs.end(), std::next(freeArgs.cbegin(), 1), freeArgs.cend());
            }

            if (command == "juggler-check") {
                Command = JugglerCheck;
                config = GetRunningConfig();
                if (config.empty()) {
                    Cout << "PASSIVE-CHECK:ebpf_agent;OK;";
                    std::exit(0);
                }
                DoInitGlobalLog(MakeHolder<TNullLogBackend>());
            } else if (command == "cleanup") {
                Command = Cleanup;
            } else if (command == "get-dc") {
                Command = GetDatacenter;
                if (CommandArgs.empty()) {
                    ythrow TBadArgumentException() << "get-dc command requires argument";
                }
            } else {
                ythrow TBadArgumentException() << "Unknown command '" << command << "'";
            }
        }

        if (Command == GetDatacenter) {
            return *this;
        }

        try {
            Config().LoadJson(config);
            Config().FixMisconfig();
        } catch (...) {
            ERROR_LOG << "Failed to load JSON config: " << CurrentExceptionMessage() << Endl;
        }

        // overwrite some config options from cmdline
        if (parsedOpts.Has(&crossDcRtoOpt)) {
            Config().MutableTcpRto()->SetCrossDcRto(true);
        }
        if (parsedOpts.Has(&sockMinRtoOpt)) {
            Config().MutableTcpRto()->SetSockMinRto(true);
        }
        if (parsedOpts.Has(&debugOpt)) {
            Config().MutableCommon()->SetDebug(true);
        }

        ELogPriority logPriority = TLOG_INFO;
        if (Config().GetCommon().GetDebug()) {
            logPriority = TLOG_DEBUG;
        }

        if (Command != JugglerCheck) {
            THolder<TLogBackend> logBackend;

            if (parsedOpts.Has(&logOpt)) {
                constexpr ui64 MAX_LOG_FILE_SIZE = 64 * 1024 * 1024;
                constexpr ui32 MAX_LOG_FILES = 1;
                logBackend.Reset(new TFilteredLogBackend(MakeHolder<TRotatingFileLogBackend>(parsedOpts.Get(&logOpt), MAX_LOG_FILE_SIZE, MAX_LOG_FILES), logPriority));
            } else {
                logBackend.Reset(new TFilteredLogBackend(MakeHolder<TStreamLogBackend>(&Cout), logPriority));
            }

            DoInitGlobalLog(std::move(logBackend));
        }

        INFO_LOG << "Used config:\n" << Config().DebugString() << Endl;

        return *this;
    }

    void* TDaemon::BpfWorkerLoopWrapper(void* param) {
        try {
            static_cast<NEbpfAgent::TDaemon*>(param)->BpfWorkerLoop();
        } catch (...) {
            FATAL_LOG << CurrentExceptionMessage() << Endl;
        }
        return nullptr;
    }

    void* TDaemon::PahomUpdaterLoopWrapper(void* param) {
        try {
            static_cast<NEbpfAgent::TDaemon*>(param)->PahomUpdaterLoop();
        } catch (...) {
            FATAL_LOG << CurrentExceptionMessage() << Endl;
        }
        return nullptr;
    }

    void TDaemon::BpfWorkerLoop() {
        auto check = CheckPrerequisites(ForceRun);
        auto error = std::get<0>(check);
        if (error) {
            INFO_LOG << "Prerequisites are not satisfied, do nothing: " << error << Endl;
            auto signal = std::get<1>(check);
            TInstant deadline;
            while (!Stopped.WaitD(deadline)) {
                PushSignal(signal, 1);

                deadline = YASM_INTERVAL.ToDeadLine();
            }
            return;
        }

        // main work
        MeasureExecTime("Cleanup net namespaces on start", CleanupNetNamespaces);
        TInstant deadline, nextCheck;
        while (!Stopped.WaitD(deadline)) {
            PushSignal(ESignal::Running, 1);

            auto now = TInstant::Now();
            if (now >= nextCheck) {
                nextCheck = now + TDuration::Seconds(Config().GetPrograms().GetCheckInterval());
                if (!IsDisabled()) {
                    MeasureExecTime("Setup net namespaces", SetupNetNamespaces);
                    MeasureExecTime("Cgroup root programs check and repair", TCgroupRootBpfProgram::Check, true);
                } else {
                    PushSignal(ESignal::Disabled, 1);
                    INFO_LOG << "Agent is manually disabled" << Endl;
                    ::Cleanup();
                }
            }

            if (TNetStatDcBpfProgram::IsAnyEnabled()) {
                MeasureExecTime("Update net stat metrics", TNetStatDcBpfProgram::UpdateMetrics);
            }

            if (TTcpRtoBpfProgram::TcpRto().IsEnabled() && TTcpRtoBpfProgram::IsAnyTcpCounterEnabled()) {
                MeasureExecTime("Update tcp metrics", TTcpRtoBpfProgram::UpdateMetrics);
            }

            UpdateErrorMetrics();

            deadline = YASM_INTERVAL.ToDeadLine();
        }
        MeasureExecTime("Cleanup net namespaces on exit", CleanupNetNamespaces);
    }

    void TDaemon::TryReadYttlBlacklistNetsFromFile(const TString& path) {
        try {
            TFileInput input(path);
            TStringStream stream(input.ReadAll());
            TYaNetworkSet localBlacklistNets = PahomClient.ParseYttlBlacklistNets(stream);

            for (const auto& net: localBlacklistNets) {
                Y_VERIFY(!bpf_map_update_elem(TBpfProgram::YttlBlacklistNetsMapFd, net.LpmKey, &net.LpmValue, 0));
                DEBUG_LOG << "Added yttl blacklist network from file " << net << Endl;
            }
        } catch (...) {
        }
    }

    void TDaemon::PahomUpdaterLoop() {
        TDuration sleepTime = TDuration::Seconds(RandomNumber<unsigned int>(60));

        INFO_LOG << "Pahom updater thread will be started in " << sleepTime << Endl;

        if (Stopped.WaitT(sleepTime)) {
            return;
        }

        INFO_LOG << "Pahom updater thread is started" << Endl;

        TYaNetworkSet prevYttlBlacklistNets;
        TInstant deadline;
        while (!Stopped.WaitD(deadline)) {
            try {
                auto yttlBlacklistNets = PahomClient.GetYttlBlacklistNets();
                TBpfProgram::EnableYttl();

                auto addedNets = GetDiff(yttlBlacklistNets, prevYttlBlacklistNets);
                auto removedNets = GetDiff(prevYttlBlacklistNets, yttlBlacklistNets);

                for (const auto& net: addedNets) {
                    if (bpf_map_update_elem(TBpfProgram::YttlBlacklistNetsMapFd, net.LpmKey, &net.LpmValue, 0)) {
                        auto error = LastSystemErrorText();
                        ERROR_LOG << "Failed to update element in yttl blacklist map: " << error << Endl;
                        continue;
                    }
                    INFO_LOG << "Added yttl blacklist network " << net << Endl;
                }

                for (const auto& net: removedNets) {
                    if (bpf_map_delete_elem(TBpfProgram::YttlBlacklistNetsMapFd, net.LpmKey)) {
                        auto error = LastSystemErrorText();
                        ERROR_LOG << "Failed to delete element from yttl blacklist map: " << error << Endl;
                        continue;
                    }
                    INFO_LOG << "Removed yttl blacklist network " << net << Endl;
                }

                prevYttlBlacklistNets = std::move(yttlBlacklistNets);
            } catch (...) {
                ERROR_LOG << "Pahom client error: " << CurrentExceptionMessage() << Endl;
            }

            deadline = TDuration::Seconds(Config().GetPahom().GetYttlBlacklistUpdateInterval()).ToDeadLine();
        }
    }

    void TDaemon::Stop() {
        Stopped.Signal();
    }

    int TDaemon::Run() {
        if (Command) {
            return RunCommand();
        }

        SetInterruptSignalsHandler(SignalHandler);

        if (Config().GetCommon().GetUnistatPusher()) {
            StartUnistatPusher();
        }

        INFO_LOG << "Agent is starting, kernel " << KernelVersion() << Endl;
        bool kernelOk = (CompareKernelVersion("4.19") >= 0);

        if (kernelOk) {
            // mount cgroup2 hierarchy if needed
            if (Config().GetCommon().GetMountCgroup2() && FindCgroupRoot(false).Empty()) {
                const constexpr auto CGRPOUP2_PATH = "/sys/fs/cgroup/unified";
                if (!NFs::Exists(CGRPOUP2_PATH)) {
                    if (Mkdir(CGRPOUP2_PATH, 0755)) {
                        ERROR_LOG << "Failed to mkdir " << CGRPOUP2_PATH << ": " << LastSystemErrorText() << Endl;
                        return -1;
                    }
                }
                if (mount("cgroup2", CGRPOUP2_PATH, "cgroup2", 0, "")) {
                    ERROR_LOG << "Failed to mount cgroup2: " << LastSystemErrorText() << Endl;
                    return -1;
                }
            }

            // preload dc
            INFO_LOG << "Datacenter is " << GetMyDatacenter() << Endl;

            // adjust mlock
            // https://st.yandex-team.ru/RTCNETWORK-561
            auto mlockLimitMb = Config().GetCommon().GetMlockLimitMb();
            if (mlockLimitMb) {
                if (!SetMlockLimit(mlockLimitMb * 1024 * 1024)) {
                    auto error = LastSystemErrorText();
                    ERROR_LOG << "Failed to set mlock limit: " << error << Endl;
                    return -1;
                }
            }

            // preload known progs
            for (const auto& prog: TCgroupRootBpfProgram::KnownProgs()) {
                prog->Load();
            }

            if (TTcpRtoBpfProgram::TcpRto().IsEnabled())
                TryReadYttlBlacklistNetsFromFile("/usr/share/ebpf-agent/yttl_blacklist_nets.txt");

            if (Config().GetPahom().GetYttlBlacklistUpdateInterval()) {
                PahomUpdater.Start();
            }
        }

        // always start bpf worker to push status
        BpfWorker.Start();
        BpfWorker.Join();

        if (kernelOk && Config().GetPahom().GetYttlBlacklistUpdateInterval()) {
            PahomUpdater.Join();
        }

        if (Config().GetCommon().GetUnistatPusher()) {
            StopUnistatPusher();
        }

        return 0;
    }

    TDaemon& Daemon() noexcept {
        static TDaemon daemon;
        return daemon;
    }

} // namespace NEbpfAgent
