#!/usr/bin/env python3

import subprocess
import contextlib
import tempfile
import argparse
import difflib
import hashlib
import shutil
import sys
import os
import re
import json
import datetime
import socket
import retrying
import porto


DEFAULT_FS_SIZE = 4*1024**3

PORTO_COMMAND_TIMEOUT_SEC = 6000

DIGEST_FILE = 'SHA1SUMS'
ENVIRONMENT_FILE = 'etc/yandex_environment.json'

log_file = sys.stdout


def print_log(*args):
    print(*args, file=log_file, flush=True)


def print_err(*args):
    print(*args, file=sys.stderr, flush=True)


def run(args, check=True, stdout=None, stderr=None, **kwargs):
    print_log("+ '" + "' '".join(args) + "'")
    if stdout is None:
        stdout = log_file
    if stderr is None:
        stderr = log_file
    return subprocess.run(args, check=check, stdout=stdout, stderr=stderr, **kwargs)


def sudo_run(args, **kwargs):
    return run(['sudo', '-E'] + args, **kwargs)


@contextlib.contextmanager
def tmpdir(keep_tmp, **kwargs):
    tmp = tempfile.mkdtemp(**kwargs)
    try:
        yield tmp
    finally:
        if not keep_tmp:
            # TODO REMOVE THIS
            sudo_run(['rm', '-rf', tmp])
        else:
            print_log('leave tmp: {}'.format(tmp))


def ensure_dir(p):
    if not os.path.exists(p):
        sudo_run(['mkdir', '-p', p])


@contextlib.contextmanager
def chroot_mount(root, arcadia_root=None, arcadia_chroot_path=None):
    m_list = [('proc', 'none', 'proc'),
              ('devpts', 'devpts', 'dev/pts'),
              ('sysfs', 'sysfs', 'sys')]
    mounts = []
    try:
        for t, d, p in m_list:
            m_path = os.path.join(root, p)
            ensure_dir(m_path)
            sudo_run(['mount',  '-t', t, d, m_path])
            mounts.append(m_path)
        if arcadia_root and arcadia_chroot_path:
            m_path = os.path.join(root, arcadia_chroot_path)
            ensure_dir(m_path)
            sudo_run(['mount', '-o', 'bind', arcadia_root, m_path])
            mounts.append(m_path)
        yield mounts
    finally:
        for m in mounts:
            sudo_run(['umount', m], check=False)


def phase_scrips(script_pack, phase):
    phase_path = os.path.join(script_pack, phase)
    scripts = []
    if not os.path.exists(phase_path):
        return scripts
    names = os.listdir(phase_path)
    names.sort()
    for name in names:
        script_path = phase_path + '/' + name
        if name.endswith('.sh') and os.path.isfile(script_path):
            scripts.append(os.path.join(phase, name))
    return scripts


def get_configs(script_pack):
    config_dir = os.path.join(script_pack, 'config')
    configs = []
    if not os.path.exists(config_dir):
        return configs
    names = os.listdir(config_dir)
    names.sort()
    for name in names:
        config_path = config_dir + '/' + name
        if name.startswith('config-') and os.path.isfile(config_path):
            configs.append('config/' + name)
    return configs


def get_digest(file_path):
    h = hashlib.sha1()

    with open(file_path, 'rb') as file:
        while True:
            # Reading is buffered, so we can read smaller chunks.
            chunk = file.read(h.block_size)
            if not chunk:
                break
            h.update(chunk)
    return h.hexdigest()


def read_pack_digest(script_dir):
    digest_file = os.path.join(script_dir, DIGEST_FILE)
    if not os.path.exists(digest_file):
        return []
    with open(digest_file) as f:
        return f.readlines()


def save_pack_digest(script_dir, digest):
    with open(os.path.join(script_dir, DIGEST_FILE), 'w') as f:
        f.writelines(digest)


def get_pack_digest(script_dir):
    digest = []

    for c in get_configs(script_dir):
        h = get_digest(os.path.join(script_dir, c))
        digest.append('{}  {}\n'.format(h, c))

    for p in ['bootstrap', 'pre-os', '.', 'post-os']:
        for s in phase_scrips(script_dir, p):
            h = get_digest(os.path.join(script_dir, s))
            digest.append('{}  {}\n'.format(h, s))

    h = hashlib.sha1()
    h.update(''.join(digest).encode('utf-8'))
    script_digest = h.hexdigest()
    digest.append('# {}  {}\n'.format(script_digest, DIGEST_FILE))
    return digest, script_digest


def compress_option(archive, compress):
    if compress and compress != 'auto':
        if '.' in compress:
            comp = compress.rsplit('.', 2)
            return ['--use-compress-program', comp[0] + ' -' + comp[1]]
        return ['--use-compress-program', compress]
    elif archive.endswith('.zst'):
        # tar < 1.30 has no support for zstd
        return ['--use-compress-program', 'zstdmt']
    else:
        return ['--auto-compress']


def tar_env():
    if '/usr/bin' not in os.getenv('PATH').split(os.pathsep):
        env = os.environ.copy()
        env['PATH'] = env['PATH'] + ':/bin:/usr/bin'
    else:
        env = None
    return env


def unpack_layer(layer, archive, compress=None, keep_overlay=False):

    if compress == 'squash' or archive.endswith(('.squash', '.squashfs')):
        sudo_run(['unsquashfs',
                  '-force',
                  '-no-progress',
                  '-dest', layer,
                  archive],
                  env=tar_env())
        return

    tar_args = ['tar',
                '--numeric-owner',
                '--preserve-permissions',
                '--xattrs',
                '--xattrs-include=security.capability',
                '--xattrs-include=user.*']

    if keep_overlay:
        tar_args += ['--xattrs-include=trusted.overlay.*']

    tar_args += compress_option(archive, compress)

    tar_args += ['--extract', '--file', archive]

    sudo_run(tar_args, cwd=layer, env=tar_env())


def pack_layer(layer, archive, compress=None):
    if compress and compress.startswith('squash') or archive.endswith(('.squash', '.squashfs')):
        comp_args = ['-comp', 'lzo']    # no zstd in linux 4.4

        if compress and '.' in compress:
            comp = compress.split('.')
            comp_args[1] = comp[1]
            if len(comp) > 2:
                comp_args += ['-Xcompression-level', comp[2]]

        sudo_run(['mksquashfs',
                  layer,
                  archive,
                  '-no-progress',
                  '-noappend'] + comp_args,
                  env=tar_env())
        return

    tar_args = ['tar',
                '--create',
                '--one-file-system',
                '--numeric-owner',
                '--preserve-permissions',
                '--xattrs',
                '--sort=name',
                '--transform', 's:^./::']

    tar_args += compress_option(archive, compress)

    tar_args += ['--file', archive, '.']

    sudo_run(tar_args, cwd=layer, env=tar_env())


def read_layer(archive, name, compress=None):
    if compress and compress.startswith('squash') or archive.endswith(('.squash', '.squashfs')):
        with tempfile.TemporaryDirectory() as tmp:
            subprocess.run(['unsquashfs',
                            '-dest', os.path.join(tmp, 'rootfs'),
                            '-no-progress',
                            archive, name],
                            stdout=subprocess.DEVNULL)
            return open(os.path.join(tmp, 'rootfs', name), 'rb').read()

    tar_args = ['tar',
                '--extract',
                '--to-stdout']

    tar_args += compress_option(archive, compress)

    tar_args += ['--file', archive, name]

    return subprocess.check_output(tar_args, env=tar_env())


def handle_list_pack(args):
    digest, _ = get_pack_digest(args.script_pack)
    if args.check:
        saved = read_pack_digest(args.script_pack)
        diff = list(difflib.unified_diff(saved, digest, fromfile='saved', tofile='current'))
        if diff:
            sys.stdout.writelines(diff)
            sys.exit(1)
    elif args.save:
        save_pack_digest(args.script_pack, digest)
    else:
        sys.stdout.writelines(digest)


def build_layer(args, name=None, script_digest=None):
    script_dir = args.script_pack
    if not name:
        name = os.path.basename(script_dir)

    env = os.environ
    build_env = {}
    for arg in args.env:
        k, v = arg.split('=', 1)
        env[k] = v
        build_env[k] = v

    env["virt_mode"] = args.virt_mode

    build_time = datetime.datetime.now().isoformat(sep=' ', timespec='seconds')
    env["build_time"] = build_time

    if args.arcadia_path:
        env["arcadia_path"] = args.arcadia_path

    if script_digest:
        env["script_digest"] = script_digest

    if args.chroot_arcadia:
        arcadia_root = os.getenv('ARCADIA_ROOT')
    else:
        arcadia_root = None

    print_log("Environment variables:")
    for k, v in env.items():
        print_log(k, '=', v)
    print_log()

    with tmpdir(args.keep_temps, prefix='build-layer-' + name) as build_dir:
        rootfs_dir = os.path.join(build_dir, 'rootfs')
        os.mkdir(rootfs_dir)

        print_log("Building", script_dir, "in", rootfs_dir)

        script_pack = os.path.join(rootfs_dir, 'script')
        shutil.copytree(script_dir, script_pack)

        if args.base_layer:
            print_log("Unpacking", args.base_layer)
            unpack_layer(rootfs_dir, os.path.abspath(args.base_layer))

        sp_dir = os.path.join(rootfs_dir, 'script')
        for s in phase_scrips(sp_dir, 'bootstrap'):
            s_path = os.path.join('script', s)
            print_log("\nStarting bootstrap", s_path, '\n')
            sudo_run(['bash', '-ex', s_path], cwd=rootfs_dir, env=env)

        with chroot_mount(rootfs_dir, arcadia_root, args.chroot_arcadia_path):
            for s in phase_scrips(sp_dir, 'pre-os'):
                s_path = os.path.join('script', s)
                print_log("\nStarting pre-os {}".format(s_path), '\n')
                sudo_run(['chroot', '.', 'bash', '-ex', s_path], cwd=rootfs_dir, env=env)

            for s in phase_scrips(sp_dir, '.'):
                s_path = os.path.join('script', s)
                print_log("\nStarting script {}".format(s_path), '\n')
                sudo_run(['chroot', '.', 'bash', '-ex', s_path], cwd=rootfs_dir, env=env)

            for s in phase_scrips(sp_dir, 'post-os'):
                s_path = os.path.join('script', s)
                print_log("\nStarting post-os {}".format(s_path), '\n')
                sudo_run(['chroot', '.', 'bash', '-ex', s_path], cwd=rootfs_dir, env=env)

        sudo_run(['rm', '-rf', script_pack])

        if True:
            info = {
                "arcadia_path": args.arcadia_path,
                "script_digest": script_digest,
                "virt_mode": args.virt_mode,
                "build_time": build_time,
                "build_env": build_env,
            }
            if args.flavor:
                info['flavor'] = args.flavor

            info_str = json.dumps(info, sort_keys=True, indent=4, separators=(',', ': '))

            if not os.path.exists(rootfs_dir + '/etc'):
                sudo_run(['mkdir', 'etc'], cwd=rootfs_dir)

            sudo_run(['tee', ENVIRONMENT_FILE], input=info_str.encode('utf-8'), cwd=rootfs_dir)

        print_log("\nPacking", args.out)
        pack_layer(rootfs_dir, os.path.abspath(args.out), compress=args.compress)
        print_log("Done")


class MissingLayersError(Exception):
    pass


class BuildCommandError(Exception):
    pass


def _run_container_cmd(container, cmd, env=None):
    print_log(f'Running command "{cmd}" in "{container}", env: {env}')
    if env:
        env = ';'.join(f'{k}={v}' for k, v in env.items())
        container.Set(env=env)
    else:
        container.Set(env='')
    container.Set(command=cmd)
    container.Start()
    container.Wait(timeout_s=PORTO_COMMAND_TIMEOUT_SEC)

    reply = container.Get(['exit_code', 'stdout', 'stderr', 'state'])
    print_log(f'current cmd stdout:\n{reply.get("stdout")}\nstderr:\n{reply.get("stderr")}')

    if 'exit_code' in reply and int(reply['exit_code']) == 0 and reply['state'] != 'running':
        print_log(f'\nCommand "{cmd}" exited with: {reply.get("exit_code")}\n'
                  f'stdout:\n{reply.get("stdout")}\nstderr:\n{reply.get("stderr")}')
        container.Stop()
    else:
        print_log(f'failed to run "{cmd}":\nstdout:\n{reply.get("stdout")}\n'
                  f'stdout:\n{reply.get("stdout")}\nstderr:\n{reply.get("stderr")}')
        container.Kill(9)


@contextlib.contextmanager
def _porto_build(name, layer_tarballs, arcadia_root, chroot_arcadia_path, script_dir, cleanup_disasbled):
    conn = porto.Connection()
    layers = []
    # layer ordering magic
    # layers in args in natural order bottommost...topmost
    # porto requires reversed order topmost...bottommost
    for i in range(len(layer_tarballs)-1, -1, -1):
        tarball = layer_tarballs[i]
        l_name = f'root-{i}'
        print_log(f'\nImporting "{tarball}" as "{l_name}"')
        layer = conn.ImportLayer(l_name, tarball)
        layers.append(layer)
    if not all(layers):
        raise MissingLayersError(
            f'Some layers are missing args.base_layer: f{layer_tarballs}, imported layers:f{layers}')
    rootfs = conn.CreateVolume(layers=layers, backend='overlay')
    print_log(f'\nCreated rootfs volume: {rootfs.path}')
    build_container = conn.Create(name=f'build-container-{name}', weak=False)
    build_container.Set(root=rootfs)
    build_container.Set(isolate=True)
    binds = []
    if arcadia_root and chroot_arcadia_path:
        binds.append(f'{arcadia_root} {chroot_arcadia_path} rw')
    binds.append(f'{script_dir} /script ro')
    build_container.Set(bind=';'.join(binds))
    try:
        yield build_container, rootfs
    finally:
        if not cleanup_disasbled:
            print_log(f"\nDestroying container {build_container.name}")
            build_container.Destroy()
            print_log(f"\nDestroying volume: {rootfs.path}")
            rootfs.Destroy()
            print_log(f"\nCleaning up imported layers: {layers}")
            for layer in layers:
                layer.Remove()


def build_porto_layer(args, name=None, script_digest=None):
    script_dir = args.script_pack
    if not name:
        name = os.path.basename(script_dir)

    env = os.environ
    for arg in args.env:
        k, v = arg.split('=', 1)
        env[k] = v

    env["virt_mode"] = args.virt_mode

    build_time = datetime.datetime.now().isoformat(sep=' ', timespec='seconds')
    env["build_time"] = build_time

    if args.arcadia_path:
        env["arcadia_path"] = args.arcadia_path

    if args.chroot_arcadia:
        arcadia_root = os.getenv('ARCADIA_ROOT')
    else:
        arcadia_root = None

    print_log("Environment variables:")
    for k, v in env.items():
        print_log(k, '=', v)
    print_log()

    with _porto_build(name, args.base_layer, arcadia_root, args.chroot_arcadia_path, script_dir, args.keep_temps) as ctx:
        build_container, rootfs = ctx
        for s in phase_scrips(script_dir, 'pre-os'):
            s_path = os.path.join('script', s)
            print_log("\nStarting pre-os {}".format(s_path), '\n')
            cmd = f'/bin/bash -ex /{s_path}'
            _run_container_cmd(build_container, cmd, env)

        for s in phase_scrips(script_dir, '.'):
            s_path = os.path.join('script', s)
            print_log("\nStarting script {}".format(s_path), '\n')
            cmd = f'/bin/bash -ex /{s_path}'
            _run_container_cmd(build_container, cmd, env)

        for s in phase_scrips(script_dir, 'post-os'):
            s_path = os.path.join('script', s)
            print_log("\nStarting post-os {}".format(s_path), '\n')
            cmd = f'/bin/bash -ex /{s_path}'
            _run_container_cmd(build_container, cmd, env)

        print_log("\nExporting", args.out)
        rootfs.Export(tarball=args.out)

    print_log("Done")


def handle_build_image(args):
    # Partless image can not exceed 2Tb: https://st.yandex-team.ru/QEMUKVM-290
    assert(args.size < 2*1024**4)

    name = os.path.basename(args.src)
    layer_digest = get_digest(os.path.join(args.src))

    env = os.environ
    env["image_build_time"] = datetime.datetime.now().isoformat(sep=' ', timespec='seconds')
    env["layer_path"] = args.src
    env["layer_digest"] = layer_digest

    print_log("Environment variables:")
    for k, v in env.items():
        print_log(k, '=', v)
    print_log()

    with tmpdir(args.keep_temps, prefix='build-image-' + name) as build_dir:
        # copy source ?
        src = os.path.join(build_dir, os.path.basename(args.src))
        shutil.copyfile(args.src, src)

        # make raw image with gpt partition table
        gptbin = ""
        gptbin_search = ['/usr/lib/syslinux/gptmbr.bin', '/usr/lib/EXTLINUX/gptmbr.bin', '/usr/lib/SYSLINUX/gptmbr.bin']
        for f in gptbin_search:
            if os.path.exists(f):
                gptbin = f
        if not gptbin:
            raise Exception('Can not fild bootloader stage1 binnary: gptmbr.bin')

        img = 'rootfs.raw'
        run(['truncate', '-s', str(args.size), img], cwd=build_dir)
        run(['sgdisk', '--clear', '-g', '--new', '1::-0', '--typecode=1:8300', img], cwd=build_dir)
        run(['sgdisk', '--attributes=1:set:2', img], cwd=build_dir)
        run(['dd', 'if=' + gptbin, 'of=' + img, 'conv=notrunc', 'bs=440', 'count=1'], cwd=build_dir)
        p = sudo_run(['losetup', '--find', '--show', img], cwd=build_dir, stdout=subprocess.PIPE, encoding='utf-8')
        loopdev = p.stdout.strip()
        sudo_run(['partx', '-a', loopdev], cwd=build_dir)
        # Double check that partprobe have found partition
        loop_part = loopdev + 'p1'
        assert os.path.exists(loop_part)
        try:
            sudo_run(['mkfs.ext4', '-O', '^metadata_csum', loop_part])
        except subprocess.CalledProcessError:
            # Previous command may fail if distro does not know nothing about metadata_csum option
            run(['mkfs.ext4', loop_part])

        rootfs_dir = os.path.join(build_dir, 'rootfs')
        os.mkdir(rootfs_dir)

        # mount raw image
        try:
            sudo_run(['mount',  loop_part, 'rootfs', '-oloop,noload,nobarrier'], cwd=build_dir)

            unpack_layer(rootfs_dir, src)

            # inclide syslinux boot loader
            sudo_run(['extlinux', '--install', 'rootfs'], cwd=build_dir)
        finally:
            sudo_run(['umount', 'rootfs'], cwd=build_dir, check=False)

        # set label=rootfs
        sudo_run(['tune2fs', '-L', 'rootfs', loop_part])

        # disable journal
        if args.ephemeral:
            run(['tune2fs', '-O', '^has_journal', '-E', 'mount_opts=nobarrier', '-e', 'panic', loop_part])

        sudo_run(['fsck.ext4', '-fyD', loop_part], check=False)
        sudo_run(['fsck.ext4', '-fy', '-E', 'discard', loop_part])
        sudo_run(['losetup', '-d', loopdev])

        # convert image
        cmd = [args.qemu_img_tool, 'convert', img, '-O', args.out_fmt]
        # enable compresion for qcow2
        if args.out_fmt == 'qcow2':
            cmd.append('-c')
        run(cmd + [os.path.abspath(args.out)], cwd=build_dir)


def size_suffix_str(value, pattern=re.compile('^([0-9]+)([KkMmGgTt]{0,1})$')):
    if not value:
        return value
    res = pattern.match(value)
    if not res:
        raise argparse.ArgumentTypeError('Invalid size value \"{}\", '
                                         'valid suffixes are Kk, Mm, Gg and Tt or none'.format(value))
    num_str, suffix = res.groups()
    numeric = int(num_str)
    if suffix.lower() == 'k':
        numeric *= 1024
    elif suffix.lower() == "m":
        numeric *= 1048576
    elif suffix.lower() == "g":
        numeric *= 1048576 * 1024
    elif suffix.lower() == "t":
        numeric *= 1048576 * 1048576
    return numeric


def _handle_build_layer(args, builder):
    digest, script_digest = get_pack_digest(args.script_pack)
    if args.check_digest or args.save_digest:
        saved = read_pack_digest(args.script_pack)
        if saved != digest:
            diff = difflib.unified_diff(saved, digest, fromfile='saved', tofile='current')
            if args.check_digest:
                print_log("Check of script pack digest failed")
                sys.stdout.writelines(diff)
                raise Exception("Script pack digest not uptodate")
            else:
                print_log("Save script pack digest changes")
                sys.stdout.writelines(diff)
                save_pack_digest(args.script_pack, digest)

    global log_file
    if args.log:
        log_file = open(args.log, 'a+')

    try:
        builder(args, script_digest=script_digest)
    except:
        if args.log:
            try:
                log_file.seek(0)
                sys.stderr.write(log_file.read())
            except:
                raise
        raise


def handle_build_layer(args):
    _handle_build_layer(args, build_layer)


def handle_build_porto_layer(args):
    try:
        _handle_build_layer(args, build_porto_layer)
    except Exception as e:
        import traceback
        et, ev, tb = sys.exc_info()
        print_log(f'\nGot Exception building layer: {e}\n{"".join(traceback.format_exception(et, ev, tb))}')


def handle_inspect(args):
    if args.current:
        info_text = open('/' + ENVIRONMENT_FILE, 'r').read()
    elif args.layer:
        info_text = read_layer(args.layer, ENVIRONMENT_FILE).decode('utf-8')
    else:
        print_err("No environment to inspect")
        sys.exit(1)

    print_log(info_text)
    info = json.loads(info_text)

    if args.check_arcadia_path and args.check_arcadia_path != info['arcadia_path']:
        print_err("Arcadia path not match:", info['arcadia_path'], 'expected', args.check_arcadia_path)
        sys.exit(1)

    if args.check_virt_mode and args.check_virt_mode != info['virt_mode']:
        print_err("Virt mode not match:", info['virt_mode'], 'expected', args.check_virt_mode)
        sys.exit(1)

    if args.check_flavor is not None and args.check_flavor != info.get('flavor', 'None'):
        print_err("Flavor not match:", info.get('flavor'), 'expected', args.check_flavor)
        sys.exit(1)

    if args.check_script_digest and args.check_script_digest != info['script_digest']:
        print_err("Script digest not match:", info['script_digest'], 'expected', args.check_script_digest)
        sys.exit(1)

    if args.check_script:
        _, script_digest = get_pack_digest(args.check_script)
        if script_digest != info['script_digest']:
            print_err("Script digest not match:", info['script_digest'], 'expected', script_digest)
            sys.exit(1)

    if args.check_network:
        @retrying.retry(stop_max_delay=30000, wait_fixed=200)
        def _sock_connect(addr):
            socket.create_connection(addr.rsplit(':', 1), timeout=3)
        _sock_connect(args.check_network)


def layer_args(parser, handle):
    parser.set_defaults(handle=handle)
    parser.add_argument('script_pack', help='Script pack directory')
    parser.add_argument('--env', nargs='*', default=[], metavar='NAME=VALUE', help='set environment variables')
    parser.add_argument('--out', default='layer.tar.gz',
                        help='result tarball, suffix defines compression, default: %(default)s')
    parser.add_argument('--compress', help='compression method: gzip|xz|zstd|squash[.level], default: auto')
    parser.add_argument('--log', help='save build log into file')
    parser.add_argument('--virt-mode', default='app', choices=['app', 'os', 'vm'],
                        help='layer type, default: %(default)s')
    parser.add_argument('--flavor', nargs='?', help='layer flavor, default: None')
    parser.add_argument('--arcadia-path', metavar='PATH',
                        help='path in arcadia for this environment to save in ' + ENVIRONMENT_FILE)
    parser.add_argument('--keep-temps', dest='keep_temps', default=False, action='store_true')
    parser.add_argument('--check-digest', action='store_true', help='check digest file ' + DIGEST_FILE)
    parser.add_argument('--save-digest', action='store_true', help='update digest file ' + DIGEST_FILE)
    parser.add_argument('--chroot-arcadia', dest='chroot_arcadia', action='store_true', help='mount arcadia to chroot',
                        default=False)
    parser.add_argument('--chroot-arcadia-path', dest='chroot_arcadia_path', default='arcadia',
                        help='arcadia chroot path')
    return parser


def main():
    parser = argparse.ArgumentParser()
    subparsers = parser.add_subparsers(title="Possible actions", dest='action')
    subparsers.required = True

    p_layer = layer_args(subparsers.add_parser(name="build-porto-layer"), handle_build_porto_layer)
    p_layer.add_argument('--base-layer', nargs='*', default=[], metavar='TARBALL', help='tarball with base layer')

    layer = layer_args(subparsers.add_parser(name="build-layer"), handle_build_layer)
    layer.add_argument('--base-layer', metavar='TARBALL', help='tarball with base layer')

    inspect = subparsers.add_parser(name="inspect")
    inspect.set_defaults(handle=handle_inspect)
    inspect.add_argument('--current', action='store_true', help='inspect current environment')
    inspect.add_argument('--layer', help='inspect environment in tarball')
    inspect.add_argument('--check-script', default=None, metavar='SCRIPT_PACK', help='compare script digest')
    inspect.add_argument('--check-script-digest', default=None, metavar='DIGEST', help='compare exact digest')
    inspect.add_argument('--check-virt-mode', default=None, metavar='VIRT_MODE')
    inspect.add_argument('--check-flavor', default=None, const='None', metavar='FLAVOR', nargs='?')
    inspect.add_argument('--check-arcadia-path', default=None, metavar='PATH')
    inspect.add_argument('--check-network', default=None, metavar='HOST:PORT', help='check network connectivity')

    list_pack = subparsers.add_parser(name="list-pack")
    list_pack.set_defaults(handle=handle_list_pack)
    list_pack.add_argument('script_pack', help='Script pack directory')
    list_pack.add_argument('-c', '--check', default=False, action='store_true', help='check digest')
    list_pack.add_argument('-s', '--save', default=False, action='store_true', help='save digest')

    image = subparsers.add_parser(name="build-image")
    image.set_defaults(handle=handle_build_image)
    image.add_argument('src', help='Source layer tarball')
    image.add_argument('--out', default='rootfs.img', help='result tarball, default: %(default)s')
    image.add_argument('--size', type=size_suffix_str, default=DEFAULT_FS_SIZE, help='Image virtual size, default: %(default)s')
    image.add_argument('--out-fmt', default='qcow2', help='Result image format, default: %(default)s')
    image.add_argument('--qemu-img-tool', default='qemu-img', help='qemu-img tool to use, default: %(default)s')
    image.add_argument('--keep-temps', dest='keep_temps', default=False, action='store_true', help='Keep temporary directory')
    image.add_argument('--ephemeral', dest='ephemeral', default=False, action='store_true', help='Make image for non-persistent storage, non-journalled filesystem')

    args = parser.parse_args()
    args.handle(args)


if __name__ == '__main__':
    main()
