package server

import (
	"context"
	"fmt"
	"os"
	"os/signal"
	"sync"
	"syscall"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
)

const (
	Addr = "[::]:80"
)

type Config struct {
	TmpDir   string
	RepoPath string
	Addr     string
	Hostname string
	Debug    bool
}

type Fwmanager struct {
	Server   *Server
	l        log.Logger
	hostname string
}

func CreateFwmanager(config *Config) (*Fwmanager, error) {
	cfg := zap.ConsoleConfig(log.InfoLevel)
	l, _ := zap.New(cfg)
	serv := CreateServer(l, config.Addr, config.Debug)
	return &Fwmanager{
		l:        l,
		Server:   serv,
		hostname: config.Hostname,
	}, nil
}

func (f *Fwmanager) Start() error {
	ctx, cancel := context.WithCancel(context.Background())
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		s := <-sigs
		f.l.Infof("Received %s, closing", s)
		cancel()
	}()
	errCh := make(chan error)
	wg := sync.WaitGroup{}
	wg.Add(3)
	go func() {
		err := f.Server.Start(ctx)
		if err != nil {
			errCh <- fmt.Errorf("server: %w", err)
		}
		wg.Done()
	}()
	done := make(chan bool)
	go func() {
		wg.Wait()
		done <- true
	}()
	select {
	case err := <-errCh:
		f.l.Fatal(err.Error())
		return err
	case <-done:
		return nil
	}
}
