package server

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"os"
	"os/signal"
	"syscall"

	"a.yandex-team.ru/library/go/core/log"

	"a.yandex-team.ru/infra/fwmanager/internal/api"
	"a.yandex-team.ru/infra/fwmanager/internal/fetcher"
	"a.yandex-team.ru/infra/fwmanager/internal/httpserver"

	"github.com/go-chi/chi/v5/middleware"
)

type Server struct {
	serv *httpserver.Server
	l    log.Logger
}

func CreateServer(l log.Logger, addr string, debug bool) *Server {
	hs := httpserver.NewHTTP(addr)
	api.RouteRegister(hs.Mux())
	if debug {
		hs.Mux().Mount("/debug", middleware.Profiler())
	}
	return &Server{
		serv: hs,
		l:    l,
	}
}

func (s *Server) Start(ctx context.Context) error {
	ctx, cancel := context.WithCancel(ctx)
	errCh := make(chan error)
	sigs := make(chan os.Signal, 1)
	signal.Notify(sigs, syscall.SIGINT, syscall.SIGTERM)
	go func() {
		<-sigs
		cancel()
	}()
	go func() {
		if err := s.serv.Run(ctx, s.l); err != http.ErrServerClosed {
			errCh <- fmt.Errorf("http server: %w", err)
		}
		fmt.Println("http server closed")
	}()
	go func() {
		if err := fetcher.SetFetchCron(); err != errors.New("fetcher cron wasnt started") {
			errCh <- fmt.Errorf("fetcher cron: %w", err)
		}
		fmt.Println("fetcher failed to start")
	}()

	return nil
}
