import flask
import json
import re
import json_validator

from flask_wtf import Form
from wtforms import Field, StringField, IntegerField, SubmitField, SelectField, TextAreaField, BooleanField, FloatField, HiddenField
from wtforms.validators import InputRequired, Optional, ValidationError
from wtforms.widgets import TextInput, TextArea

import docs_help


class CommaSeparatedListFieldBase(Field):
    def _value(self):
        if self.data:
            return u', '.join(self.data)
        else:
            return u''

    def process_formdata(self, valuelist):
        if valuelist and valuelist[0].strip():
            self.data = [x.strip() for x in valuelist[0].split(',')]
        else:
            self.data = []

    def readonly(self):
        return self.render_kw and self.render_kw.get('readonly', False)


class CommaSeparatedListStringField(CommaSeparatedListFieldBase):
    widget = TextInput()


class CommaSeparatedListTextAreaField(CommaSeparatedListFieldBase):
    widget = TextArea()


def owners_default():
    if flask.g.yandex_login:
        return [flask.g.yandex_login]
    else:
        return []


def populate_form(form, data_dict):
    for k, v in data_dict.iteritems():
        getattr(form, k).data = v


def get_changed_fields(form, default_data_dict):
    changed_form_fields = {}
    for key, value in get_modifiable_fields_dict(form).iteritems():
        key = key.split('-')[-1]
        if value != default_data_dict.get(key, ''):
            changed_form_fields[key] = value

    return changed_form_fields


def get_modifiable_fields_dict(form):
    relevant_fields = {}
    for field in form:
        if not isinstance(field, (SubmitField, HiddenField)):
            relevant_fields[field.name] = field.data

    return relevant_fields


def validate_volumes(volumes):
    scheme = {
        'type': list,
        'items': {
            'type': dict,
            'scheme': {
                'quota': {'type': unicode, 'regex': re.compile(r'^[0-9.]+\sGb$').match},
                'symlinks': {
                    'type': list,
                    'items': {
                        'type': unicode,
                    },
                },
                'shared': {'type': bool},
                'generate_deprecated_uuid': {'type': bool},
                'uuid_generator_version': {'type': int},
                'mount_point_workdir': {'type': bool},
                'node_mount_point': {'type': unicode},
                'pod_mount_point': {'type': unicode}
            }
        }
    }
    data = json.loads(volumes)
    json_validator.validate(data, scheme)

    if data:
        print('DATA: {}'.format(data))

        requred = {'': 1, '/': 1}
        counts = {}
        for volume in data:
            if volume['pod_mount_point'] in requred:
                requred[volume['pod_mount_point']] -= 1
            else:
                if volume['pod_mount_point'] not in counts:
                    counts[volume['pod_mount_point']] = 0
                counts[volume['pod_mount_point']] += 1

        if any(requred.itervalues()):
            raise ValueError('Count root,workdir volume must be 1')

        if [x for x in counts.itervalues() if x > 1]:
            raise ValueError('Mount point must be uniq')


class BaseGroupForm(Form):
    group = StringField('Group name', validators=[InputRequired()])
    description = TextAreaField('Description', validators=[InputRequired()],
                                description=docs_help.GROUP_DESCRIPTION_HELP)
    owners = CommaSeparatedListStringField('Owners', default=owners_default, validators=[InputRequired()],
                                           description=docs_help.GROUP_OWNERS_HELP)
    watchers = CommaSeparatedListStringField('Watchers', description=docs_help.GROUP_WATCHERS_HELP)

    ctype = SelectField('Cluster Type (a_ctype)', validators=[InputRequired()], choices=[],
                        default='test', description=docs_help.CTYPE_HELP)
    itype = SelectField('Instance Type (a_itype)', validators=[InputRequired()], choices=[],
                        default='none', description=docs_help.ITYPE_HELP)
    metaprj = SelectField('Project (a_metaprj)', validators=[InputRequired()], choices=[],
                          default='unknown', description=docs_help.METAPRJ_HELP)
    prj = CommaSeparatedListStringField('Project</br>(a_prj)', default=['none'], validators=[InputRequired()],
                                        description=docs_help.PRJ_HELP)
    itag = CommaSeparatedListStringField('Custom tags', description=docs_help.ITAG_HELP)

    dispenser_project_key = SelectField('Dispenser project', choices=[], default='')

    instance_count = IntegerField('Instance count', default=1, validators=[InputRequired()],
                                  description=docs_help.INSTANCE_COUNT_HELP)
    cores_per_instance = FloatField('Cores guarantee', default=1., validators=[InputRequired()],
                                    description=docs_help.CORE_PER_INSTANCE_HELP)
    memory = FloatField('Memory guarantee', default=1., validators=[InputRequired()], description=docs_help.MEMORY_HELP)
    disk = FloatField('HDD quota', default=1., validators=[InputRequired()], description=docs_help.DISK_HELP)
    ssd = FloatField('SSD quota', default=0., validators=[InputRequired()], description=docs_help.SSD_HELP)
    net = FloatField('Net quota', default=0., validators=[InputRequired()], description=docs_help.NET_HELP)

    submit = SubmitField("Submit")

    def __init__(self, prefix):
        super(BaseGroupForm, self).__init__(prefix=prefix)

        self.ctype.choices = [(x, x) for x in flask.current_app._context.states.get_trunk_ctypes()]
        self.itype.choices = [(x, x) for x in flask.current_app._context.states.get_trunk_itypes()]
        self.metaprj.choices = [(x, x) for x in flask.current_app._context.states.get_trunk_metaprjs()]
        self.dispenser_project_key.choices = [('', '')] + [(x, x) for x in flask.current_app._context.states.get_user_dispenser_projects(flask.g.yandex_login)]


class AllocateGroupInDynamicForm(BaseGroupForm):
    affinity_category = StringField('Affinity category', description=docs_help.AFFINITY_CATEGORY_HELP)

    ipv64_tunnel_v2 = BooleanField('Tunnel 6to4 V2', default=False, description=docs_help.IP6_TUNNEL_V2_HELP)
    mtn_export_to_cauth = BooleanField('Export MTN hostnames to CAUTH', default=False, description=docs_help.CAUTH_HELP)
    use_mtn_in_config = BooleanField('Use MTN addresses in CONFIGS', default=False, description=docs_help.MTN_CONFIG_HELP)
    portovm_mtn_addrs = BooleanField('Use MTN addresses in PORTOVM', default=False, description=docs_help.MTN_ADDR_HELP)
    hbf_slb_name = CommaSeparatedListStringField('SLB name', default=[], description=docs_help.GROUP_SLB_NAME_HELP)

    allocate_in_man = BooleanField('MAN', default=False)
    allocate_in_sas = BooleanField('SAS', default=False)
    allocate_in_msk_iva = BooleanField('MSK_IVA', default=False)
    allocate_in_msk_myt = BooleanField('MSK_MYT', default=False)
    allocate_in_vla = BooleanField('VLA', default=False)

    volumes = TextAreaField('Json', description='Volumes json definition')
    parent_macro = StringField('Parent macros', default='', description=docs_help.GROUP_PARENT_MACRO_HELP)

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    def __init__(self):
        super(AllocateGroupInDynamicForm, self).__init__(prefix='allocate_in_dynamic_form')

        self.group.description = docs_help.GROUP_NAME_HELP
        self._location_errors = []

        self.dispenser_project_key.render_kw = {'disable': True}

    def validate(self):
        valid = super(AllocateGroupInDynamicForm, self).validate()

        if not re.match(r'^[A-Z][A-Z0-9_]+[A-Z0-9]$', self.group.data):
            self.group.errors.append('{} not match ^[A-Z][A-Z0-9_]+[A-Z0-9]$'.format(self.group.data))
            valid = False

        if not re.match(r'^[a-z][a-z0-9-]+$', self.itype.data):
            self.itype.errors.append('{} not match ^[A-Z][A-Z0-9_]+[A-Z0-9]$'.format(self.itype.data))
            valid = False

        if not re.match(r'^[a-z][a-z0-9-]+$', self.ctype.data):
            self.ctype.errors.append('{} not match ^[A-Z][A-Z0-9_]+[A-Z0-9]$'.format(self.ctype.data))
            valid = False

        for prj in self.prj.data:
            if not re.match(r'^[a-z][a-z0-9-]+$', prj):
                self.prj.errors.append('{} not match ^[a-z][a-z0-9-]+$'.format(prj))
                valid = False

        for i, owner in enumerate(self.owners.data):
            if not owner.strip():
                self.owners.errors.append('{}th owner is empty, maybe need remove comma at end'.format(i + 1))
                valid = False
            if owner.startswith("abc:") and owner.count(":") > 1:
                self.owners.errors.append('You cannot use scopes for abc services anymore. Details: https://st.yandex-team.ru/GENCFG-4567')
                valid = False

        for i, watcher in enumerate(self.watchers.data):
            if not watcher.strip():
                self.watchers.errors.append('{}th watcher is empty, maybe need remove comma at end'.format(i + 1))
                valid = False

        if not any(location.data for location in (
            self.allocate_in_man,
            self.allocate_in_sas,
            self.allocate_in_msk_iva,
            self.allocate_in_msk_myt,
            self.allocate_in_vla
        )):
            self._location_errors.append('Choose locations')
            valid = False

        if any([self.group.data.startswith(x) for x in ('MAN', 'SAS', 'VLA', 'MSK_IVA', 'MSK_MYT')]):
            self.group.errors.append('Group name cannot begin with prefix location')
            valid = False

        try:
            validate_volumes(self.volumes.data)
        except Exception as e:
            self.volumes.errors.append(str(e))
            valid = False

        return valid

    def location_errors(self):
        return [('error', error) for error in self._location_errors]


class ModifyGroupCardForm(BaseGroupForm):
    hdd_io_read_limit = FloatField('HDD mb/s read', default=0.0)
    hdd_io_write_limit = FloatField('HDD mb/s write', default=0.0)
    hdd_io_ops_read_limit = IntegerField('HDD iops read', default=0)
    hdd_io_ops_write_limit = IntegerField('HDD iops write', default=0)
    ssd_io_read_limit = FloatField('SSD mb/s read', default=0.0)
    ssd_io_write_limit = FloatField('SSD mb/s write', default=0.0)
    ssd_io_ops_read_limit = IntegerField('SSD iops read', default=0)
    ssd_io_ops_write_limit = IntegerField('SSD iops write', default=0)

    group_cores = FloatField('Cores guarantee', default=1.,)
    group_memory = FloatField('Memory guarantee (GB)', default=1.)
    group_disk = FloatField('HDD quota (GB)', default=1.)
    group_ssd = FloatField('SSD quota (GB)', default=0.)
    group_net = FloatField('Net quota (Mbit)', default=0.)

    port = StringField('Port', default='8041', description=docs_help.GROUP_PORT_HELP)
    mtn_export_to_cauth = BooleanField('Export MTN hostnames to CAUTH', default=True, description=docs_help.CAUTH_HELP)
    mtn_use_mtn_in_config = BooleanField('Use MTN addresses in CONFIGS', default=True,
                                         description=docs_help.MTN_CONFIG_HELP)
    mtn_portovm_mtn_addrs = BooleanField('Use MTN addresses in PORTOVM', default=False, description=docs_help.MTN_ADDR_HELP)
    mtn_hbf_slb_name = CommaSeparatedListStringField('SLB name', default=[], description=docs_help.GROUP_SLB_NAME_HELP)
    mtn_hbf_range = SelectField('MTN HBF Range', default='_GENCFG_SEARCHPRODNETS_ROOT_',
                                choices=[('_GENCFG_SEARCHPRODNETS_ROOT_', '_GENCFG_SEARCHPRODNETS_ROOT_'), ('_SANDBOX_', '_SANDBOX_')])
    mtn_hbf_parent_macros = SelectField('MTN HBF Alias (Parent macro)', choices=[],
                                        description=docs_help.GROUP_PARENT_MACRO_HELP)
    mtn_hbf_alias_parent = StringField('MTN HBF Alias parent', default='')

    tunnel_ipip6_ext_tunnel = BooleanField('Tunnel 6to4 V1 (deprecated)', default=False, description=docs_help.IP6_TUNNEL_HELP)
    tunnel_ipip6_ext_tunnel_v2 = BooleanField('Tunnel 6to4 V2', default=False, description=docs_help.IP6_TUNNEL_V2_HELP)

    volumes = TextAreaField('Json', validators=[InputRequired()], description='Volumes json definition')

    another_groups = CommaSeparatedListStringField('Another groups', default=[], description=docs_help.ANOTHER_GROUPS_HELP)

    moved_to_yp = BooleanField(
        'Moved to YP',
        default=False,
        description='group is moved to YP',
    )
    moved_to_yp_endpoint_set = StringField(
        'endpoint_set id',
        description='group is moved to this endpoint_set',
        default='',
    )
    moved_to_yp_location = SelectField(
        'endpoint_set location',
        default='unknown',
        description='endpoint_set YP cluster',
        choices=[
            ('sas', 'sas'),
            ('vla', 'vla'),
            ('man', 'man'),
            ('msk', 'msk'),
            ('sas-test', 'sas-test'),
            ('man-pre', 'man-pre'),
            ('unknown', 'unknown'),
        ],
    )

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    def __init__(self, group, card, instances_info, readonly, my_hbf_macroses, my_hbf_ranges, alias_parent):
        super(ModifyGroupCardForm, self).__init__(prefix='modify_form')

        instances_count = len(instances_info)
        if instances_count:
            avg_power = int(sum(i['power'] for i in instances_info) / instances_count)
        else:
            avg_power = 0
        avg_power = avg_power / 40.

        if card.get('legacy', {}).get('funcs', {}).get('instancePort', 'default') == 'default':
            port = '8041'
        else:
            port = card.get('legacy', {}).get('funcs', {}).get('instancePort', 'default').replace('new', '')
            port = port.replace('old', '')

        hbf_parent_macros = card.get('properties', {}).get('hbf_parent_macros') or ''
        if hbf_parent_macros not in my_hbf_macroses:
            my_hbf_macroses.append(hbf_parent_macros)

        self.mtn_hbf_parent_macros.choices = [(x, x) for x in my_hbf_macroses]
        self.mtn_hbf_range.choices = [(x, x) for x in my_hbf_ranges]

        self.group.default = group
        self.description.default = card.get('description', '')
        self.port.default = port
        self.owners.default = card.get('owners', [])
        self.watchers.default = card.get('watchers', [])
        self.itag.default = card.get('tags', {}).get('itag', [])

        self.instance_count.default = instances_count
        self.cores_per_instance.default = round(avg_power, 2)
        self.memory.default = round(
            card.get('reqs', {}).get('instances', {}).get('memory_guarantee', 0) / 1024. / 1024. / 1024.,
            2
        )
        self.disk.default = round(card.get('reqs', {}).get('instances', {}).get('disk', 0) / 1024. / 1024. / 1024., 2)
        self.ssd.default = round(card.get('reqs', {}).get('instances', {}).get('ssd', 0) / 1024. / 1024. / 1024., 2)
        self.net.default = round(
            card.get('reqs', {}).get('instances', {}).get('net_guarantee', 0) * 8 / 1024. / 1024.,
            2
        )

        self.hdd_io_read_limit.default = round(card.get('reqs', {}).get('instances', {}).get('hdd_io_read_limit', 0) / 1024. / 1024., 2)
        self.hdd_io_write_limit.default = round(card.get('reqs', {}).get('instances', {}).get('hdd_io_write_limit', 0) / 1024. / 1024., 2)
        self.hdd_io_ops_read_limit.default = card.get('reqs', {}).get('instances', {}).get('hdd_io_ops_read_limit', 0)
        self.hdd_io_ops_write_limit.default = card.get('reqs', {}).get('instances', {}).get('hdd_io_ops_write_limit', 0)
        self.ssd_io_read_limit.default = round(card.get('reqs', {}).get('instances', {}).get('ssd_io_read_limit', 0) / 1024. / 1024., 2)
        self.ssd_io_write_limit.default = round(card.get('reqs', {}).get('instances', {}).get('ssd_io_write_limit', 0) / 1024. / 1024., 2)
        self.ssd_io_ops_read_limit.default = card.get('reqs', {}).get('instances', {}).get('hdd_io_ops_read_limit', 0)
        self.ssd_io_ops_write_limit.default = card.get('reqs', {}).get('instances', {}).get('ssd_io_ops_write_limit', 0)

        self.group_cores.default = round(self.cores_per_instance.default * self.instance_count.default, 2)
        self.group_memory.default = round(self.memory.default * self.instance_count.default, 2)
        self.group_disk.default = round(self.disk.default * self.instance_count.default, 2)
        self.group_ssd.default = round(self.ssd.default * self.instance_count.default, 2)
        self.group_net.default = round(self.net.default * self.instance_count.default, 2)

        self.prj.default = card.get('tags', {}).get('prj', [])
        self.ctype.default = card.get('tags', {}).get('ctype', '')
        self.itype.default = card.get('tags', {}).get('itype', [])
        self.metaprj.default = card.get('tags', {}).get('metaprj', '')
        self.dispenser_project_key.default = card.get('dispenser', {}).get('project_key', '') or ''

        self.mtn_export_to_cauth.default = card.get('properties', {}).get('mtn', {}).get('export_mtn_to_cauth', False)
        self.mtn_use_mtn_in_config.default = card.get('properties', {}).get('mtn', {}).get('use_mtn_in_config', False)
        self.mtn_portovm_mtn_addrs.default = card.get('properties', {}).get('mtn', {}).get('portovm_mtn_addrs', False)
        self.mtn_hbf_slb_name.default = \
            card.get('properties', {}).get('mtn', {}).get('tunnels', {}).get('hbf_slb_name', '')
        self.mtn_hbf_range.default = card.get('properties', {}).get('hbf_range', '_GENCFG_SEARCHPRODNETS_ROOT_')
        self.mtn_hbf_parent_macros.default = hbf_parent_macros
        self.mtn_hbf_alias_parent.default = alias_parent

        self.tunnel_ipip6_ext_tunnel.default = card.get('properties', {}).get('ipip6_ext_tunnel', False)
        self.tunnel_ipip6_ext_tunnel_v2.default = card.get('properties', {}).get('ipip6_ext_tunnel_v2', False)

        self.volumes.default = json.dumps(simplify_volumes(card.get('reqs', {}).get('volumes', [])), indent=4)

        self.moved_to_yp.default = card.get('yp', {}).get('moved', False)
        self.moved_to_yp_endpoint_set.default = card.get('yp', {}).get('where', {}).get('endpoint_set') or ''
        self.moved_to_yp_location.default = card.get('yp', {}).get('where', {}).get('location') or 'unknown'

        self.instance_count.render_kw = {'readonly': True}
        self.group.render_kw = {'readonly': True}
        self.port.render_kw = {'readonly': True}
        self.mtn_hbf_parent_macros.render_kw = {'readonly': True}
        self.mtn_hbf_alias_parent.render_kw = {'readonly': True}
        self.tunnel_ipip6_ext_tunnel.render_kw = {'readonly': True}

        self.hdd_io_read_limit.render_kw = {'readonly': True}
        self.hdd_io_write_limit.render_kw = {'readonly': True}
        self.hdd_io_ops_read_limit.render_kw = {'readonly': True}
        self.hdd_io_ops_write_limit.render_kw = {'readonly': True}
        self.ssd_io_read_limit.render_kw = {'readonly': True}
        self.ssd_io_write_limit.render_kw = {'readonly': True}
        self.ssd_io_ops_read_limit.render_kw = {'readonly': True}
        self.ssd_io_ops_write_limit.render_kw = {'readonly': True}

        self.group_cores.render_kw = {'readonly': True}
        self.group_memory.render_kw = {'readonly': True}
        self.group_disk.render_kw = {'readonly': True}
        self.group_ssd.render_kw = {'readonly': True}
        self.group_net.render_kw = {'readonly': True}

        if card.get('master') in ['ALL_DYNAMIC', 'ALL_PERSONAL', 'ALL_SOX']:
            self.cores_per_instance.render_kw = {'readonly': True}
            self.memory.render_kw = {'readonly': True}
            self.net.render_kw = {'readonly': True}

        self.disk.render_kw = {'readonly': True}
        self.ssd.render_kw = {'readonly': True}

        if not self.tunnel_ipip6_ext_tunnel.default:
            self.tunnel_ipip6_ext_tunnel.render_kw = {'disabled': True}

        if readonly:
            self.ctype.choices = [(self.ctype.default, self.ctype.default)]
            self.itype.choices = [(self.itype.default, self.itype.default)]
            self.metaprj.choices = [(self.metaprj.default, self.metaprj.default)]
            self.dispenser_project_key.choices = [(self.dispenser_project_key.default, self.dispenser_project_key.default)]

            self.mtn_hbf_parent_macros.choices = [(self.mtn_hbf_parent_macros.default, self.mtn_hbf_parent_macros.default)]
            self.mtn_hbf_range.choices = [(self.mtn_hbf_range.default, self.mtn_hbf_range.default)]

            self.description.render_kw = {'readonly': True}
            self.owners.render_kw = {'readonly': True}
            self.watchers.render_kw = {'readonly': True}
            self.ctype.render_kw = {'readonly': True}
            self.itype.render_kw = {'readonly': True}
            self.prj.render_kw = {'readonly': True}
            self.itag.render_kw = {'readonly': True}
            self.metaprj.render_kw = {'readonly': True}
            self.dispenser_project_key.render_kw = {'readonly': True}
            self.instance_count.render_kw = {'readonly': True}
            self.cores_per_instance.render_kw = {'readonly': True}
            self.memory.render_kw = {'readonly': True}
            self.disk.render_kw = {'readonly': True}
            self.ssd.render_kw = {'readonly': True}
            self.net.render_kw = {'readonly': True}
            self.mtn_export_to_cauth.render_kw = {'disabled': True}
            self.mtn_use_mtn_in_config.render_kw = {'disabled': True}
            self.mtn_portovm_mtn_addrs.render_kw = {'disabled': True}
            self.mtn_hbf_slb_name.render_kw = {'readonly': True}
            self.mtn_hbf_range.render_kw = {'readonly': True}
            self.tunnel_ipip6_ext_tunnel_v2.render_kw = {'disabled': True}
            self.volumes.render_kw = {'disabled': True}
            self.moved_to_yp.render_kw = {'disabled': True}
            self.moved_to_yp_endpoint_set.render_kw = {'disabled': True}
            self.moved_to_yp_location.render_kw = {'disabled': True}

    def validate(self):
        valid = super(ModifyGroupCardForm, self).validate()

        if not re.match(r'^[A-Z][A-Z0-9_]+[A-Z0-9]$', self.group.data):
            self.group.errors.append('{} not match ^[A-Z][A-Z0-9_]+[A-Z0-9]$'.format(self.group.data))
            valid = False

        if not re.match(r'^[a-z][a-z0-9-]+$', self.itype.data):
            self.itype.errors.append('{} not match ^[A-Z][A-Z0-9_]+[A-Z0-9]$'.format(self.itype.data))
            valid = False

        if not re.match(r'^[a-z][a-z0-9-]+$', self.ctype.data):
            self.ctype.errors.append('{} not match ^[A-Z][A-Z0-9_]+[A-Z0-9]$'.format(self.ctype.data))
            valid = False

        for prj in self.prj.data:
            if not re.match(r'^[a-z][a-z0-9-]+$', prj):
                self.prj.errors.append('{} not match ^[a-z][a-z0-9-]+$'.format(prj))
                valid = False

        for i, owner in enumerate(self.owners.data):
            if not owner.strip():
                self.owners.errors.append('{}th owner is empty, maybe need remove comma at end'.format(i + 1))
                valid = False
            if owner.startswith("abc:") and owner.count(":") > 1:
                self.owners.errors.append('You cannot use scopes for abc services anymore. Details: https://st.yandex-team.ru/GENCFG-4567')
                valid = False

        for i, watcher in enumerate(self.watchers.data):
            if not watcher.strip():
                self.watchers.errors.append('{}th watcher is empty, maybe need remove comma at end'.format(i + 1))
                valid = False

        try:
            validate_volumes(self.volumes.data)
        except Exception as e:
            self.volumes.errors.append(str(e))
            valid = False

        yp_endpoint_set_is_default = self.moved_to_yp_endpoint_set.data == ''
        yp_location_is_default = self.moved_to_yp_location.data == 'unknown'

        if yp_endpoint_set_is_default != yp_location_is_default:
            self.moved_to_yp_endpoint_set.errors.append(
                'yp location and endpoint_set must be both set or both default'
            )
            valid = False

        return valid


class ReclusterInDynamicForm(Form):
    target_instances_count = IntegerField('Instance count', validators=[InputRequired()],
                                          description=docs_help.INSTANCE_COUNT_HELP)
    target_instance_power = FloatField('Cores guarantee', validators=[InputRequired()],
                                       description=docs_help.CORE_PER_INSTANCE_HELP)
    target_instance_memory = FloatField('Memory guarantee', validators=[Optional()], description=docs_help.MEMORY_HELP)
    target_instance_disk = FloatField('Disk guarantee', validators=[Optional()], description=docs_help.DISK_HELP)
    target_instance_ssd = FloatField('SSD guarantee', validators=[Optional()], description=docs_help.SSD_HELP)
    target_instance_net = FloatField('Net guarantee', validators=[Optional()], description=docs_help.NET_HELP)
    volumes = TextAreaField('Json', description='Volumes json definition')

    another_groups = CommaSeparatedListStringField('Another groups', default=[],
                                                   description=docs_help.ANOTHER_GROUPS_HELP)

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    submit = SubmitField("Submit")

    def __init__(self, readonly):
        super(ReclusterInDynamicForm, self).__init__(prefix='recluster_in_dynamic')
        self.target_instances_count.render_kw = {'disabled': readonly}
        self.target_instance_memory.render_kw = {'disabled': readonly}
        self.target_instance_power.render_kw = {'disabled': readonly}
        self.target_instance_disk.render_kw = {'readonly': True}
        self.target_instance_ssd.render_kw = {'readonly': True}
        self.target_instance_net.render_kw = {'disabled': readonly}
        self.commit_message.render_kw = {'disabled': readonly}
        self.submit.render_kw = {'disabled': readonly}

    def validate(self):
        if not Form.validate(self):
            return False

        if not self.target_instances_count.data and not self.target_instance_memory.data and not self.target_instance_power.data:
            for field in [self.instances, self.memory, self.cpu]:
                field.errors.append('Please change at least one field')
            return False

        return True


class RemoveGroupForm(Form):
    group = StringField(
        'Group name', validators=[InputRequired()], description='Enter group name for validation'
    )

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    submit = SubmitField("Submit")

    def __init__(self, readonly, my_groups, admin, owner, group):
        super(RemoveGroupForm, self).__init__(prefix='remove_group')
        self.admin = admin
        self.owner = owner
        self.request_group = group
        self.my_groups = my_groups
        self.group.render_kw = {'disabled': readonly}
        self.commit_message.render_kw = {'disabled': readonly}
        self.submit.render_kw = {'disabled': readonly}

    def validate(self):
        if not Form.validate(self):
            return False

        if not self.group.data:
            self.group.errors.append('Fill group name')
            return False

        elif self.request_group != self.group.data:
            self.group.errors.append('Wrong group name')
            return False

        elif not self.owner and not self.admin:
            self.group.errors.append('Permission denied.')
            return False

        return True


class CreateGroupForm(Form):
    # Common group params
    group = StringField('Group name', validators=[InputRequired()], description=docs_help.GROUP_NAME_HELP)
    group_description = TextAreaField('Group description', description=docs_help.GROUP_DESCRIPTION_HELP)
    owners = CommaSeparatedListStringField("Group owners", default=owners_default, validators=[InputRequired()], description=docs_help.GROUP_OWNERS_HELP)
    metaprj = StringField('Meta project', default="unknown", validators=[InputRequired()])

    instance_port_func = StringField('Port function', default="auto", validators=[InputRequired()])

    template_group = StringField('Template group')
    parent_group = StringField('Parent group')

    group_type = SelectField(
        'Group type',
        choices=[('group_with_resources', 'Group with resources'),
                 ('fake_group', 'Fake group'),
                 ('full_host_group', 'Full host group')],
        validators=[InputRequired()]
    )

    instance_memory = StringField('Instance Memory', default="1 GB", validators=[InputRequired()])
    instance_power = IntegerField('Instance Power', default=40, validators=[InputRequired()])

    submit = SubmitField("Submit")

    def validate_group(form, field):
        if not field.data.startswith("ALL_"):
            raise ValidationError("Currently group name must start with 'ALL_'")


class GroupResourcesForm(Form):
    # hosts_to_add = CommaSeparatedListTextAreaField('Hosts to add')
    # hosts_to_remove = CommaSeparatedListTextAreaField('Hosts to remove')

    disk = FloatField('Disk')
    ssd = FloatField('SSD')

    memory_guarantee = FloatField('Memory guarantee (in GBs)')
    memory_overcommit = FloatField('Memory overcommit (in GBs)')

    submit = SubmitField("Submit")

    def validate(self):
        if not Form.validate(self):
            return False

        # if not self.hosts_to_add.data and not self.hosts_to_remove.data:
        #     for field in [self.hosts_to_add, self.hosts_to_remove]:
        #         field.errors.append("You must either remove or add some hosts")
        #     return False

        return True

    def validate_reserved_hosts(self, group):
        reserved_hosts = flask.current_app._context.states.get_reserved_hosts(group)
        to_add = self.hosts_to_add.data
        to_add_not_in_reserved = set(to_add) - reserved_hosts

        if to_add_not_in_reserved:
            self.hosts_to_add.errors.append("The following hosts are not in reserved groups: {}".format(', '.join(to_add_not_in_reserved)))
            return False

        return True


class ModifyCardForm(Form):
    owners = CommaSeparatedListStringField("Group owners", validators=[InputRequired()], description=docs_help.GROUP_OWNERS_HELP)
    ctype = StringField('ctype')
    itype = StringField('itype')
    metaprj = StringField('metaprj')

    submit = SubmitField("Submit")


class CreateMacrosForm(Form):
    macros = StringField('Macros name', validators=[InputRequired()], default='')
    parent = StringField('Parent macros', default='')
    description = TextAreaField('Description', validators=[InputRequired()],
                                description='Macros purpose and other human-readable description')
    owners = CommaSeparatedListStringField("Macros owners", validators=[InputRequired()])

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    submit = SubmitField("Submit")

    def validate(self):
        valid = Form.validate(self)

        if not re.match(r'^([0-9A-Z]+[_]?[0-9A-Z]+)+$', self.macros.data):
            self.macros.errors.append('{} not match ^([0-9A-Z]+[_]?[0-9A-Z]+)+$'.format(self.macros.data))
            valid = False

        return valid


class CreateTypeForm(Form):
    type_class = SelectField(
        'Type class',
        validators=[InputRequired()],
        choices=[('itype', 'itype'), ('ctype', 'ctype')]
    )
    type_name = StringField('Type name', validators=[InputRequired()], default='')
    description = StringField('Description', default='')

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')
    submit = SubmitField("Submit")

    def validate(self):
        valid = Form.validate(self)

        check_ptrn = r'^[a-z][a-z0-9-]+$'
        if self.type_class.data == 'itype':
            check_ptrn = '^[a-zA-Z0-9]{1,93}(?:[a-zA-Z0-9]{1,7}|_custom)\Z'

        if not re.match(check_ptrn, self.type_name.data):
            self.type_name.errors.append('{} not match {}'.format(self.type_name.data, check_ptrn))
            valid = False

        return valid


class RemoveMacrosForm(Form):
    macros = StringField(
        'Macros name', validators=[InputRequired()], description='Enter macros name for validation'
    )

    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    submit = SubmitField("Submit")

    def __init__(self, readonly, my_macroses, admin, macros):
        super(RemoveMacrosForm, self).__init__(prefix='remove_macros')
        self.admin = admin
        self.request_macros = macros
        self.my_macroses = my_macroses
        self.macros.render_kw = {'disabled': readonly}
        self.commit_message.render_kw = {'disabled': readonly}
        self.submit.render_kw = {'disabled': readonly}

    def validate(self):
        if not Form.validate(self):
            return False

        if not self.macros.data:
            self.macros.errors.append('Fill macros name')
            return False

        elif self.request_macros != self.macros.data:
            self.macros.errors.append('Wrong macros name')
            return False

        elif self.macros.data not in self.my_macroses and not self.admin:
            self.macros.errors.append('Permission denied.')
            return False

        return True


class ModifyMacrosForm(Form):
    macros = StringField('Macros name')
    parent = StringField('Parent macros')
    # description = TextAreaField('Description', default='')
    owners = CommaSeparatedListStringField("Macros owners", default='')
    commit_message = TextAreaField('Commit message', description='This text will be added to commit message')

    submit = SubmitField("Submit")

    def __init__(self, readonly, my_macroses, admin, macros):
        super(ModifyMacrosForm, self).__init__(prefix='modify_macros')
        self.admin = admin
        self.my_macroses = my_macroses

        self.macros.default = macros['name']
        self.parent.default = macros.get('parent_macros') or ''
        # self.description.default = macros['description']
        self.owners.default = macros['owners']

        self.macros.render_kw = {'readonly': True}
        self.parent.render_kw = {'disabled': readonly}
        # self.description.render_kw = {'disabled': True}
        self.owners.render_kw = {'disabled': readonly}
        self.commit_message.render_kw = {'disabled': readonly}
        self.submit.render_kw = {'disabled': readonly}

    def validate(self):
        if not Form.validate(self):
            return False

        if self.macros.data not in self.my_macroses and not self.admin:
            self.macros.errors.append('Permission denied.')
            return False

        return True


class BuildImproxyForm(Form):
    comment = StringField('Comment', validators=[InputRequired()])
    submit = SubmitField("Submit")


def simplify_volumes(volumes):
    remap = {'guest_mp': 'pod_mount_point', 'host_mp_root': 'node_mount_point', 'quota': 'quota', 'symlinks': 'symlinks'}

    ret = []
    for volume in volumes:
        ret.append({remap[x]: str(round(volume[x] / 1024. / 1024 / 1024, 2)) + ' Gb' if x == 'quota' else volume[x] for x in remap.keys()})
    return sorted(ret)


def desimplify_volumes(volumes):
    remap = {'pod_mount_point': 'guest_mp', 'node_mount_point': 'host_mp_root', 'quota': 'quota', 'symlinks': 'symlinks'}

    ret = []
    for volume in volumes:
        ret.append({remap[x]: volume[x] for x in remap.keys()})
    return sorted(ret)
