# coding: utf8
import os
import sys
import time
import logging

from datetime import datetime


class Logger(object):
    format = '%(levelname)s - %(asctime)s - %(name)s - %(message)s'

    def __init__(self, logger_name, format=None):
        self.logger_name = logger_name
        self.format = format or Logger.format
        self.formatter = logging.Formatter(self.format)
        self.logger = logging.getLogger(logger_name)
        self.logger_file_path = ''

        sh = logging.StreamHandler(sys.stdout)
        sh.setFormatter(self.formatter)
        self.logger.addHandler(sh)

    def time_counting(self):
        def decorator(func):
            logger = Logger('{}.{}'.format(self.logger_name, func.__name__), self.format)

            def wrapper(*args, **kwargs):
                start = time.time()
                logger.info('called')
                result = func(*args, **kwargs)
                logger.info('finished {}s'.format(time.time() - start))
                return result

            wrapper.__name__ = func.__name__
            return wrapper
        return decorator

    def exception_logger(self, ExceptionType=Exception, raise_on_catch=True):
        def decorator(func):
            logger = Logger('{}.{}'.format(self.logger_name, func.__name__), self.format)

            def wrapper(*args, **kwargs):
                try:
                    return func(*args, **kwargs)
                except ExceptionType as e:
                    logger.exception('{}: {}'.format(type(e), e))
                    if raise_on_catch:
                        raise

            wrapper.__name__ = func.__name__
            return wrapper
        return decorator

    def debug(self, text):
        self._prepare_logger()
        self.logger.debug(text)

    def info(self, text):
        self._prepare_logger()
        self.logger.info(text)

    def exception(self, text):
        self._prepare_logger()
        self.logger.exception(text)

    def error(self, text):
        self._prepare_logger()
        self.logger.error(text)

    def _prepare_logger(self):
        self.logger.setLevel(logging.INFO)

        logger_file_path = 'logs'
        if not os.path.exists(logger_file_path):
            os.makedirs(logger_file_path)
        logger_file_path = os.path.join(logger_file_path, '{}.log'.format(datetime.today().strftime('%Y-%m-%d')))

        if logger_file_path != self.logger_file_path:
            for hanler in self.logger.handlers:
                if isinstance(hanler, logging.FileHandler):
                    hanler.close()
                    self.logger.removeHandler(hanler)

            fh = logging.FileHandler(logger_file_path)
            fh.setFormatter(self.formatter)
            self.logger.addHandler(fh)

            self.logger_file_path = logger_file_path
