import os
import gevent.monkey
gevent.monkey.patch_all()  # noqa

import argparse
import logging
import jinja2

import library.python.resource as resource
import gevent.pywsgi as wsgi
import gevent

from flask import Flask
from flask_bootstrap import Bootstrap

from reports import reports_blueprint
from states import States


COOKIE_SECRET_KEY = os.environ.get('COOKIE_SECRET_KEY', 'secret')


def parse_args():
    parser = argparse.ArgumentParser(formatter_class=argparse.ArgumentDefaultsHelpFormatter)
    parser.add_argument('--port', help='Port to listen', type=int, required=True)
    parser.add_argument('--debug', help='Use flask built-in server with nice debugging capabilities', action='store_true')
    parser.add_argument('--sandbox', help='Sandbox url (override for local testing)', default='sandbox.yandex-team.ru')

    return parser.parse_args()


def _load_template_from_resources(name):
    return resource.find('/templates/{}'.format(name))


def loop(func, interval):
    while True:
        # noinspection PyBroadException
        try:
            func()
        except Exception:
            logging.exception('Unhandled exception in loop')
        finally:
            gevent.sleep(interval)


def configure_app(sandbox_url):
    app = Flask(__name__, static_folder='')
    Bootstrap(app)

    app.config['BOOTSTRAP_SERVE_LOCAL'] = True
    app.secret_key = COOKIE_SECRET_KEY
    app.register_blueprint(reports_blueprint)
    app._context = _AppContext(States(), sandbox_url)
    app.jinja_env.loader = jinja2.FunctionLoader(_load_template_from_resources)

    return app


class _AppContext(object):
    def __init__(self, states_, sandbox):
        self.states = states_
        self.sandbox = sandbox
        states_.update()
        gevent.spawn(loop, states_.update, 60)


def main():
    args = parse_args()

    logging.basicConfig(level=logging.DEBUG)

    app = configure_app(args.sandbox)

    if args.debug:
        app.run(host='::', port=args.port, use_reloader=False, debug=True)
    else:
        wsgi.WSGIServer(('::', args.port), app).serve_forever()
