# coding: utf-8 -*-
import os
import re
import json
import mongo
import flask
import socket
import requests
import sandbox_tasks
import change_requests
import pymongo.operations
import library.python.resource as resource

from logger import Logger
from flask import request
from forms import (
    GroupResourcesForm, ReclusterInDynamicForm, populate_form, get_changed_fields,
    ModifyGroupCardForm, RemoveGroupForm, RemoveMacrosForm, ModifyMacrosForm, AllocateGroupInDynamicForm,
    CreateTypeForm, BuildImproxyForm, desimplify_volumes, simplify_volumes
)
from settings import DEFAULT_VOLUMES, DEFAULT_SAMOGON_VOLUMES

import utils
from utils import make_promise, wait_promises

reports_blueprint = flask.Blueprint('viewer', __name__)

logger = Logger('reports')

ADMINS = ['osol', 'sereglond', 'mcden', 'okats', 'kulikov', 'shotinleg', 'desertfury', 'ssmike', 'nikita-uvarov', 'glebx777', 'evgakimutin', 'avatar', 'skorodumov-s']
GENCG_GUI_ID = os.environ.get('OAUTH_APP_ID', '')
GENCG_GUI_PASSWORD = os.environ.get('OAUTH_APP_PASSWORD', '')
PROD_ARCADIA = 'a.yandex-team.ru'
DYNAMIC_MASTERS = ('ALL_DYNAMIC', 'ALL_PERSONAL', 'ALL_SOX')
DYNAMIC_CLOSED_MESSAGE = unicode("""<div><p style="text-align: left; font-size: 14pt;">New groups allocation in ALL_DYNAMIC (ALL_PERSONAL, ALL_SOX) is closed.
<a href="https://clubs.at.yandex-team.ru/yp/240">https://clubs.at.yandex-team.ru/yp/240</a><br/>
If you still need to create a group in GENCFG for some reason, please create a ticket in st/GENCFG with group specification and description why you can't use YP instead.</p></div>
""", 'utf8')
DYNAMIC_CLOSED_MESSAGE_2 = unicode("""<div><p style="text-align: left; font-size: 14pt;">Recluster groups in ALL_DYNAMIC (ALL_PERSONAL, ALL_SOX) is closed.
<a href="https://clubs.at.yandex-team.ru/yp/240">https://clubs.at.yandex-team.ru/yp/240</a><br/>
If you still need to recluster a group in GENCFG for some reason, please create a ticket in st/GENCFG with group specification and description why you can't use YP instead.</p></div>
""", 'utf8')


def time_distribution(start_field, end_field, requests_data):
    timeline_data = {}
    for request_mongo in requests_data:
        if start_field not in request_mongo or end_field not in request_mongo:
            continue

        added = request_mongo[start_field]
        finished = request_mongo[end_field]

        month, day = added.date().month, added.date().day
        hour, minute = added.time().hour, added.time().minute // 15 * 15
        stimestamp = '{}-{} {}:{}'.format(day, month, hour, minute)
        timestamp = month * 10 ** 6 + day * 10 ** 4 + hour * 10 ** 2 + minute

        subtask = request_mongo['sandbox_task_data']['subtask']
        subtask = subtask.replace('-', '_')

        if subtask not in timeline_data:
            timeline_data[subtask] = {}
        if (timestamp, stimestamp) not in timeline_data[subtask]:
            timeline_data[subtask][(timestamp, stimestamp)] = []
        timeline_data[subtask][(timestamp, stimestamp)].append((finished - added).seconds)

        subtask = 'all_request_types'
        if subtask not in timeline_data:
            timeline_data[subtask] = {}
        if (timestamp, stimestamp) not in timeline_data[subtask]:
            timeline_data[subtask][(timestamp, stimestamp)] = []
        timeline_data[subtask][(timestamp, stimestamp)].append((finished - added).seconds)

    timeline = {}
    for subtask in timeline_data:
        timeline[subtask] = []
        subtask_timeline = timeline_data[subtask]
        for key in sorted(subtask_timeline, key=lambda x: x[0]):
            timeline[subtask].append([key[1], sum(subtask_timeline[key]) / len(subtask_timeline[key])])
    return timeline


def time_count_distribution(field, requests_data):
    timeline_data = {}
    for request_mongo in requests_data:
        if field not in request_mongo:
            continue

        field_data = request_mongo[field]

        month, day = field_data.date().month, field_data.date().day
        hour, minute = field_data.time().hour, field_data.time().minute // 15 * 15
        stimestamp = '{}-{} {}:{}'.format(day, month, hour, minute)
        timestamp = month * 10 ** 6 + day * 10 ** 4 + hour * 10 ** 2 + minute

        subtask = request_mongo['sandbox_task_data']['subtask']
        subtask = subtask.replace('-', '_')

        if subtask not in timeline_data:
            timeline_data[subtask] = {}
        if (timestamp, stimestamp) not in timeline_data[subtask]:
            timeline_data[subtask][(timestamp, stimestamp)] = 0
        timeline_data[subtask][(timestamp, stimestamp)] += 1

        subtask = 'all_request_types'
        if subtask not in timeline_data:
            timeline_data[subtask] = {}
        if (timestamp, stimestamp) not in timeline_data[subtask]:
            timeline_data[subtask][(timestamp, stimestamp)] = 0
        timeline_data[subtask][(timestamp, stimestamp)] += 1

    timeline = {}
    for subtask in timeline_data:
        timeline[subtask] = []
        subtask_timeline = timeline_data[subtask]
        for key in sorted(subtask_timeline, key=lambda x: x[0]):
            timeline[subtask].append([key[1], subtask_timeline[key]])
    return timeline


def flask_flash_form_error(form):
    """Flash error"""
    msg = ['There was a problem when submitting the form. Errors:']
    for field_name in sorted(form.errors):
        msg.append('   Field <{}>: msg <{}>'.format(field_name, form.errors[field_name]))
    msg = '\n'.join(msg)

    flask.flash(msg, 'error')


def arcadia_commit_url(commit, arcadia_url=PROD_ARCADIA):
    # https://a.yandex-team.ru/arc/commit/3536639
    return 'http://{}/arc/commit/{}'.format(arcadia_url, commit)


@reports_blueprint.before_request
def get_login():
    # TODO: proper authentication via blackbox
    if 'Authorization' in flask.request.headers:
        flask.session['yandex_token'] = flask.request.headers['Authorization'].split(' ')[-1]

        r = requests.get(
            'https://login.yandex-team.ru/info?format=json',
            headers={'Authorization': 'OAuth {}'.format(flask.session['yandex_token'])}
        )

        flask.session['yandex_login'] = r.json()['login']

    if flask.session.get('yandex_token', None) and flask.session.get('yandex_login', None):
        flask.g.yandex_login = flask.session['yandex_login']
        # flask.g.yandex_login = '<testuser>'
        flask.g.yandex_token = flask.session['yandex_token']
        flask.g.tag = flask.session.get('tag', 'trunk')
        flask.g.my_groups = flask.current_app._context.states.get_my_groups(
            flask.g.yandex_login, flask.g.yandex_token
        )
        flask.g.ctx = {
            'status': flask.current_app._context.states.get_gencfg_status(),
            'hostname': socket.gethostname()
        }
    elif 'get_token' in flask.request.path:
        return
    else:
        flask.session['redirect_path'] = flask.request.path
        for key, value in flask.request.args.iteritems():
            sep = '&' if '?' in flask.session['redirect_path'] else '?'
            flask.session['redirect_path'] = '{}{}{}={}'.format(flask.session['redirect_path'], sep, key, value)

        return flask.redirect(
            'https://oauth.yandex-team.ru/authorize?response_type=code&client_id={}'.format(
                GENCG_GUI_ID
            )
        )

    # flask.g.yandex_login = flask.request.cookies.get('yandex_login') or 'shotinleg'
    # flask.g.tag = flask.session.get('tag', 'trunk')
    # flask.g.ctx = {'status': flask.current_app._context.states.get_gencfg_status()}


@reports_blueprint.route('/auth')
@logger.time_counting()
def get_auth():
    return flask.redirect(
        'https://oauth.yandex-team.ru/authorize?response_type=code&client_id={}'.format(GENCG_GUI_ID)
    )


@reports_blueprint.route('/get_token')
@logger.time_counting()
def get_token():
    code = request.args['code']

    payload = {
        'grant_type': 'authorization_code',
        'code': str(code),
        'client_id': GENCG_GUI_ID,
        'client_secret': GENCG_GUI_PASSWORD
    }
    r = requests.post('https://oauth.yandex-team.ru/token', data=payload)
    access_token = r.json().get('access_token', r.json())

    r = requests.get(
        'https://login.yandex-team.ru/info?format=json',
        headers={'Authorization': 'OAuth {}'.format(access_token)}
    )
    login = r.json()['login']

    flask.session['yandex_login'] = login
    flask.session['yandex_token'] = access_token
    redirect_path = flask.session['redirect_path']
    flask.session.pop('redirect_path')

    return flask.redirect(redirect_path)


@reports_blueprint.route('/search', defaults={'q': ''}, methods=["GET", "POST"])
@logger.time_counting()
def search(q):
    q = flask.request.args.get('q', q).strip()

    groups = []
    if '@' not in q or q.startswith('G@'):
        groups = [
            {'name': x}
            for x in flask.current_app._context.states.get_all_groups()
            if q.replace('G@', '').upper() in x
        ]

    if q.endswith('gencfg-c.yandex.net'):
        group = flask.current_app._context.states.group_of_hostname(q)
        if group:
            groups.append({'name': group})

    macroses = []
    if '@' not in q or q.startswith('M@'):
        macroses = [
            {'name': x['name']}
            for x in flask.current_app._context.states.get_trunk_hbf_macroses()
            if q.replace('M@', '').upper() in x['name']
        ]

    groups_by_hosts = {}
    if '@' not in q or q.startswith('H@'):
        for hostname, host_groups in flask.current_app._context.states.get_hosts_groups(q.replace('H@', '')).items():
            groups_by_hosts[hostname] = [{'name': group_name} for group_name in host_groups]

    if (len(groups) + len(macroses) + len(groups_by_hosts)) == 1:
        if groups:
            return flask.redirect(flask.url_for('.trunk_group', group=groups[0]['name']))
        elif macroses:
            return flask.redirect(flask.url_for('.card_macros', macros=macroses[0]['name']))
    elif (len(groups) + len(macroses) + len(groups_by_hosts)) == 0:
        return flask.render_template(
            'no_data_info.html',
            essence='Search for "{}"'.format(q),
            essence_type='search',
            info='At your request, nothing was found.<br />'
                 'The data may not have been updated yet, please retry the request later, or contact st/RX for assistance.'
        )

    return flask.render_template('serp.html', groups=groups, macroses=macroses, groups_by_hosts=groups_by_hosts)


@reports_blueprint.route('/')
@reports_blueprint.route('/my_groups')
@logger.time_counting()
def my_groups():
    return flask.render_template('my_groups.html', groups=flask.g.my_groups)


@reports_blueprint.route('/group_graphs/<group>')
@logger.time_counting()
def group_graphs(group):
    card = app_states().get_group_card(group)
    itype = card['tags']['itype']
    ctype = card['tags']['ctype']

    return flask.redirect('https://yasm.yandex-team.ru/template/panel/Porto-container/itype={};ctype={};gencfg={}'.format(
        itype, ctype, group
    ), code=302)


@reports_blueprint.route('/group/hosts', methods=["GET", "POST"])
@reports_blueprint.route('/trunk/groups/<group>/hosts')
@reports_blueprint.route('/<tag>/groups/<group>/hosts')
@logger.time_counting()
def group_hosts(group=None, tag=None):
    tag = flask.request.args.get('tag', tag)
    tag = tag if tag != 'trunk' else None
    group = flask.request.args.get('group', group)

    if group not in flask.current_app._context.states.get_all_groups(tag):
        return flask.abort(404)

    card = app_states().get_group_card(group)
    hosts_info = app_states().get_group_hosts_info(group, tag)
    instances = app_states().get_group_instances(group, tag)
    hosts = [x['hostname'] for x in instances]

    readonly = tag is not None
    readonly = readonly if card['master'] in DYNAMIC_MASTERS else True

    full_hosts_info = {}
    for host in hosts:
        if host in hosts_info:
            full_hosts_info[host] = hosts_info[host]
        else:
            full_hosts_info[host] = {}

    tags = {k: '' for k in flask.current_app._context.states.get_gencfg_tags()}
    if tag:
        tags[tag] = 'selected'
    tag = tag or 'trunk'

    return flask.render_template(
        'group_hosts_info.html',
        group=group, tags=tags, hosts=full_hosts_info, tag=tag, readonly=readonly, select_tag=tag
    )


@reports_blueprint.route('/group/replace_hosts', methods=['POST'])
@logger.time_counting()
def replace_group_hosts():
    group = flask.request.form.get('group')
    hosts = flask.request.form.getlist('hosts[]')

    push_task(
        'manipulate_hosts',
        group,
        {
            'action': 'replace',
            'list_hosts': hosts,
            'fields': {
                'group_name': group
            }
        }
    )

    return flask.redirect(flask.url_for('.background_tasks'))


@reports_blueprint.route('/trunk/groups/<group>/hosts_diff')
@logger.time_counting()
def group_hosts_diff(group):
    rename = {
        'diff_count_hosts': 'Count hosts',
        'new_hosts': 'New hosts',
        'del_hosts': 'Removed hosts',
        'power': 'CPU',
        'porto_limits.memory_guarantee': 'Memory guarantee (porto)',
        'porto_limits.memory_limit': 'Memory limit (porto)'
    }

    hosts_diff = flask.current_app._context.states.get_group_diff(group)

    if not hosts_diff:
        return flask.render_template(
            'no_data_info.html', essence=group, essence_type='group', info='Group is missing online.'
        )

    return flask.render_template('group_hosts_diff.html', group=group, diff=hosts_diff, rename=rename)


@reports_blueprint.route('/diff/<group>/<tag_or_online>/<tag_or_trunk>', methods=["GET"])
@reports_blueprint.route('/diff/', methods=["GET"])
@logger.time_counting()
def group_diff(group=None, tag_or_online='online', tag_or_trunk='trunk'):
    group = flask.request.args.get('group', group)
    tag_or_online = flask.request.args.get('tag_or_online', tag_or_online) or 'online'
    tag_or_trunk = flask.request.args.get('tag_or_trunk', tag_or_trunk) or 'trunk'

    tags = app_states().get_gencfg_tags(30)

    left_menu_tags = {k: '' for k in tags}
    left_menu_tags['online'] = ''
    left_menu_tags[tag_or_online or 'online'] = 'selected'

    right_menu_tags = {k: '' for k in tags}
    right_menu_tags['trunk'] = ''
    right_menu_tags[tag_or_trunk or 'trunk'] = 'selected'

    diff_card = None
    if group is None:
        message = 'Enter group name and two states to fields at top.'
    else:
        diff_card = app_states().get_group_diff_tags(group, tag_or_online, tag_or_trunk)
        message = ('Can not calculate diff between {0} and {1} because NO DATA.<br />'
                   'Maybe {0} or {1} too old. Or group not found in one of states.<br />'.format(tag_or_online, tag_or_trunk))

    if diff_card is None:
        return flask.render_template('group_diff.html', group=group or '', left_version_name=tag_or_online,
                                     right_version_name=tag_or_trunk, diff_card=None,
                                     left_menu_tags=left_menu_tags, right_menu_tags=right_menu_tags,
                                     message=message)

    # diff_card = flask.current_app._context.states.get_group_diff_tags(group, tag_or_online, tag_or_trunk)

    self_hanler = lambda v: v
    power_handler = lambda v: ['{} Pu (~{} Cores)'.format(x, x / 40) for x in v]
    gb_handler = lambda v: [get_size_value_str(x, 'MB') for x in v]
    list_handler = lambda v: ['<br>'.join(x) for x in v]

    # 'field_name': ('Human readble name', value_handler_func)
    mapping = {
        'resources.ninstances': ('Instances count', self_hanler),
        'owners': ('Owners', list_handler),
        'reqs.instances.power': ('CPU', self_hanler),
        'reqs.instances.disk': ('HDD', gb_handler),
        'reqs.instances.ssd': ('SSD', gb_handler),
        'legacy.funcs.instancePort': ('Port', self_hanler),
        'tags.ctype': ('Custom type (ctype)', self_hanler),
        'tags.itype': ('Instance type (itype)', self_hanler),
        'tags.metaprj': ('metaprj', self_hanler),
        'tags.itag': ('itag', self_hanler),
        'tags.prj': ('prj', list_handler),
        'properties.hbf_parent_macros': ('Parent HBF macro', self_hanler),
        'properties.ipip6_ext_tunnel': ('IPv6 tunnel (v1)', self_hanler),
        'properties.ipip6_ext_tunnel_v2': ('IPv6 tunnel (v2)', self_hanler),
        'properties.mtn.tunnels.hbf_slb_name': ('SLB names', list_handler),
        'properties.mtn.export_mtn_to_cauth': ('Export MTN hostnames to CAUTH', self_hanler),
        'properties.mtn.portovm_mtn_addrs': ('Use MTN addresses in PORTOVM', self_hanler),
        'properties.mtn.use_mtn_in_config': ('Use MTN addresses in CONFIGS', self_hanler),
        'hosts': ('Hosts', list_handler),
        'volumes': ('Volumes', list_handler),
        'porto_power': ('CPU (porto)', power_handler),
        'porto_memory_limit': ('Memory limit (porto)', gb_handler),
        'porto_memory_guarantee': ('Memory guarantee (porto)', gb_handler),
    }
    get_allias = lambda k: mapping[k][0] if k in mapping else k
    handle_value = lambda k, v: mapping[k][1](v)

    diff_card = {get_allias(k): handle_value(k, v) for k, v in diff_card.iteritems() if v is not None}

    # return flask.Response(json.dumps(diff_card, indent=4), content_type='application/json')
    return flask.render_template('group_diff.html', group=group, left_version_name=tag_or_online,
                                 right_version_name=tag_or_trunk, diff_card=diff_card, left_menu_tags=left_menu_tags,
                                 right_menu_tags=right_menu_tags)


@reports_blueprint.route('/extra_groups/<hostname>', methods=["GET"])
@reports_blueprint.route('/extra_groups/', methods=["GET"])
@logger.time_counting()
def extra_groups(hostname=None):
    hostname = flask.request.args.get('hostname', hostname)

    host_diff = None
    if hostname:
        hostname = hostname if '.' in hostname else '{}.search.yandex.net'.format(hostname)
        host_diff = app_states().get_extra_groups_on_host(hostname)

    return flask.render_template('extra_groups.html', hostname=hostname, host_diff=host_diff)


@reports_blueprint.route('/redirect_to_group_resources_page/<group>')
@logger.time_counting()
def redirect_to_group_resources_page(group):
    card = flask.current_app._context.states.get_group_card(group)

    if card["master"] == "ALL_DYNAMIC":
        return flask.redirect(flask.url_for('.recluster_in_dynamic', group=group))
    else:
        return flask.redirect(flask.url_for('.group_hosts_and_resources', group=group))


@reports_blueprint.route('/group_hosts_and_resources/<group>', methods=["GET", "POST"])
@logger.time_counting()
def group_hosts_and_resources(group):
    card = flask.current_app._context.states.get_group_card(group)
    existing_hosts = flask.current_app._context.states.get_group_info(group)['hosts']
    # TODO: validate group is a master group
    form = GroupResourcesForm()

    card_fields = {}
    for resource_name in ['disk', 'ssd', 'memory_guarantee', 'memory_overcommit']:
        card_fields[resource_name] = gbs(card['reqs']['instances'][resource_name])

    sb_task_params = {'group': group, 'hosts_to_add': [], 'hosts_to_remove': []}
    if form.validate_on_submit():  # and form.validate_reserved_hosts(group):
        changed_form_fields = get_changed_fields(form, card_fields)
        if not changed_form_fields:
            flask.flash('No changes requested. Please change some fields', 'error')
            return flask.redirect(flask.url_for('.group_hosts_and_resources', group=group))
        sb_task_params.update(changed_form_fields)

        push_task(
            'update_resources_in_master_group',
            group,
            sb_task_params,
        )

        # flask.flash('Successfully created background task for updating group hosts {}'.format(group), 'success')
        flask.flash('This function is not supported yet', 'error')
    elif form.is_submitted():
        flask.flash('There was a problem when submitting the form', 'error')
    else:  # on GET
        populate_form(form, card_fields)

    return flask.render_template('group_hosts.html', existing_hosts=existing_hosts, form=form, group=group)


@reports_blueprint.route('/create_group', methods=['GET', 'POST'])
@logger.time_counting()
def create_group():
    if not is_admin():
        return flask.render_template(
            'no_data_info.html', essence='ALLOCATE IN DYNAMIC', essence_type='group',
            info=DYNAMIC_CLOSED_MESSAGE,
        )

    form = AllocateGroupInDynamicForm()

    if form.submit.data and form.validate_on_submit():
        group = form.group.data

        form_data = {}
        for key, value in form.data.iteritems():
            if key == 'volumes':
                form_data[key] = json.dumps(desimplify_volumes(json.loads(form.data[key])))
            elif key == 'affinity_category':
                form_data[key] = form.data[key] if form.data[key] else None
            else:
                form_data[key] = form.data[key]
        push_task('allocate_in_all_dynamic', group, form_data)

        flask.flash('Successfully created background request for allocating group {}'.format(group), 'success')

        return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask_flash_form_error(form)

    else:
        flask.flash(
            flask.Markup(
                '<a href="https://wiki.yandex-team.ru/users/slonnn/Vydeljajjte-resursy-ALLDYNAMIC-v-YP-vmesto-GenCfg">'
                'The same resources can be quickly allocated via YP'
                '</a>',
            ),
            'info'
        )
        form.process()

    if is_admin():
        my_hbf_macroses = {x['name']: '' for x in flask.current_app._context.states.get_trunk_hbf_macroses()}
    else:
        my_hbf_macroses = {
            x['name']: '' for x in flask.current_app._context.states.get_user_hbf_macroses(flask.g.yandex_login)
        }
    my_hbf_macroses[''] = 'selected'

    slb_names = {x: '' for x in app_states().get_slb_names()}

    return flask.render_template(
        'create_group.html', allocate_in_dynamic_form=form, samogon_volumes=DEFAULT_SAMOGON_VOLUMES,
        default_volumes=DEFAULT_VOLUMES, hbf_macroses=my_hbf_macroses, slb_names=slb_names
    )


@reports_blueprint.route('/trunk/groups/<group>', methods=["GET", "POST"])
@reports_blueprint.route('/trunk/groups/', defaults={'group': ''}, methods=["GET", "POST"])
@logger.time_counting()
def trunk_group(group):
    group = flask.request.args.get('group', group)
    if group not in app_states().get_all_groups():
        return flask.render_template('custom_404.html', info='Group {} not found in trunk. May be it was removed.'.format(
            group
        ))

    parent = None
    if group.endswith('_GUEST'):
        parent = group[0:-6]
        flask.flash(
            'Editing of this GUEST groups is not supported. To change this group, you must change {}'.format(parent),
            'info'
        )

    group_name_raw = utils.raw_group_name(group)

    p_requests_in_processing = make_promise(
        app_states().get_requests,
        {
            '$and': [
                {'$or': [{'essence': group}, {'essence': group_name_raw}]},
                {'$or': [{'status': 'enqueued'}, {'status': 'executing'}]}
            ]
        },
        limit=1
    )
    p_card = make_promise(app_states().get_group_card, group)
    p_instances = make_promise(app_states().get_group_instances, group)
    p_tags = make_promise(app_states().get_gencfg_tags)
    p_slb_names = make_promise(app_states().get_slb_names)

    (
        requests_in_processing, card, instances_info, tags, slb_names,
    ) = wait_promises(
        p_requests_in_processing, p_card, p_instances, p_tags, p_slb_names,
    )
    tags = {k: '' for k in tags}
    slb_names = {x: '' for x in slb_names}

    if not card:
        return flask.render_template('custom_404.html', info='Group {} not found in trunk. May be it was removed.'.format(
            group
        ))

    my_hbf_macroses = app_states().get_user_hbf_macroses(flask.g.yandex_login)
    if is_admin():
        my_hbf_macroses = app_states().get_trunk_hbf_macroses()
    my_hbf_macroses = [x['name'] for x in my_hbf_macroses
                       if is_admin() or set(x['resolved_owners']).intersection(set(card['resolved_owners']))]
    my_hbf_ranges = list(app_states().get_user_hbf_ranges(flask.g.yandex_login).keys())

    hbf_macroses = {x: '' for x in my_hbf_macroses}
    hbf_macroses[card['properties']['hbf_parent_macros'] or ''] = 'selected'

    for slb_name in card['properties']['mtn']['tunnels']['hbf_slb_name']:
        slb_names[slb_name] = 'selected'

    another_groups = {x: '' for x in app_states().get_all_groups() if group_name_raw in x}
    another_groups[group] = 'selected'

    instances_count = len(instances_info)
    if instances_count:
        avg_power = int(sum(i['power'] for i in instances_info) / instances_count)
    else:
        avg_power = 0
    avg_power = avg_power / 40.

    default_card = {
        'group': group,
        'description': card['description'],
        'owners': card['owners'],
        'watchers': card['watchers'],
        'itag': card['tags']['itag'],
        'instance_count': instances_count,
        'cores_per_instance': avg_power,
        'memory': gbs(card['reqs']['instances']['memory_guarantee']),
        'disk': gbs(card['reqs']['instances']['disk']),
        'ssd': gbs(card['reqs']['instances']['ssd']),
        'net': mbits(card['reqs']['instances']['net_guarantee']),
        'prj': card['tags']['prj'],
        'ctype': card['tags']['ctype'],
        'itype': card['tags']['itype'],
        'metaprj': card['tags']['metaprj'],
        'mtn_export_to_cauth': card['properties']['mtn']['export_mtn_to_cauth'],
        'mtn_use_mtn_in_config': card['properties']['mtn']['use_mtn_in_config'],
        'mtn_portovm_mtn_addrs': card['properties']['mtn']['portovm_mtn_addrs'],
        'mtn_hbf_slb_name': card['properties']['mtn']['tunnels']['hbf_slb_name'],
        'mtn_hbf_parent_macros': card['properties']['hbf_parent_macros'] or '',
        'mtn_hbf_range': card['properties']['hbf_range'],
        'tunnel_ipip6_ext_tunnel': card['properties']['ipip6_ext_tunnel'],
        'tunnel_ipip6_ext_tunnel_v2': card['properties']['ipip6_ext_tunnel_v2'],
        'volumes': json.dumps(card['reqs']['volumes'], indent=4),
        'moved_to_yp': card['yp']['moved'],
        'moved_to_yp_endpoint_set': card['yp']['where']['endpoint_set'] or '',
        'moved_to_yp_location': card['yp']['where']['location'] or 'unknown',
        'dispenser_project_key': card['dispenser']['project_key'] or '',
    }

    key_path = {
        'group': 'group',
        'description': 'description',
        'owners': 'owners',
        'watchers': 'watchers',
        'itag': 'tags.itag',
        'instance_count': 'reqs.shards.min_replicas',
        'cores_per_instance': 'reqs.shards.min_power',  # HACK
        'memory': 'reqs.instances.memory_guarantee',
        'disk': 'reqs.instances.disk',
        'ssd': 'reqs.instances.ssd',
        'net': 'reqs.instances.net_guarantee',
        'prj': 'tags.prj',
        'ctype': 'tags.ctype',
        'itype': 'tags.itype',
        'metaprj': 'tags.metaprj',
        'mtn_export_to_cauth': 'properties.mtn.export_mtn_to_cauth',
        'mtn_use_mtn_in_config': 'properties.mtn.use_mtn_in_config',
        'mtn_portovm_mtn_addrs': 'properties.mtn.portovm_mtn_addrs',
        'mtn_hbf_slb_name': 'properties.mtn.tunnels.hbf_slb_name',
        'mtn_hbf_parent_macros': 'properties.hbf_parent_macros',
        'mtn_hbf_range': 'properties.hbf_range',
        'tunnel_ipip6_ext_tunnel': 'properties.ipip6_ext_tunnel',
        'tunnel_ipip6_ext_tunnel_v2': 'properties.ipip6_ext_tunnel_v2',
        'volumes': 'reqs.volumes',
        'moved_to_yp': 'yp.moved',
        'moved_to_yp_endpoint_set': 'yp.where.endpoint_set',
        'moved_to_yp_location': 'yp.where.location',
        'dispenser_project_key': 'dispenser.project_key',
    }

    group_owners = card.get('resolved_owners', []) or card.get('owners', [])
    owner = (flask.g.yandex_login in group_owners) or is_admin()
    readonly = not owner or bool(parent)
    alias_parent = app_states().get_trunk_hbf_macro(card['properties']['hbf_parent_macros']).get('parent_macros')

    form = ModifyGroupCardForm(group=group, card=card, instances_info=instances_info, readonly=readonly,
                               my_hbf_macroses=my_hbf_macroses, my_hbf_ranges=my_hbf_ranges, alias_parent=alias_parent)

    if form.submit.data and form.validate_on_submit():
        prefix = 'modify_form-'
        changes = {}
        group_names = [group]
        for field in form:
            key = field.name.split(prefix, 1)[1]
            if key in default_card and key in key_path and default_card[key] != field.data:
                print('CHANGE: {} -> {}'.format(key, field.data))
                value = field.data
                if key == 'cores_per_instance' and round(default_card[key], 2) == field.data:
                    continue
                elif key == 'dispenser_project_key' and not value:
                    print('Empty dispenser_project_key not support.')
                    continue

                if key == 'mtn_hbf_parent_macros' and field.data == '':
                    print('mtn_hbf_parent_macros -> DEFAULT not support yet.')
                    continue

                if key == 'volumes':
                    default_volumes = json.dumps(desimplify_volumes(simplify_volumes(json.loads(default_card[key]))))
                    volumes = json.dumps(desimplify_volumes(json.loads(field.data)))

                    if default_volumes != volumes:
                        changes[key_path[key].replace('.', '!')] = volumes
                else:
                    if key in ['memory', 'disk', 'ssd']:
                        value = '{}Gb'.format(field.data)
                    if key in ['net']:
                        value = '{}Mbit'.format(field.data)
                    if key == 'cores_per_instance':
                        value = int(field.data * 40)
                    if key in ['owners', 'watchers', 'itag', 'prj', 'mtn_hbf_slb_name']:
                        value = '[' + ', '.join(['"{}"'.format(x) for x in field.data]) + ']'
                    changes[key_path[key].replace('.', '!')] = value
            elif key in ['another_groups']:
                group_names = [x for x in field.data]

        if not changes:
            flask.flash('No changes requested. Please change some fields', 'error')
        elif not group_names:
            flask.flash('Select one or more replication groups', 'error')
        else:
            push_task(
                'modify_group_card',
                group_name_raw if len(group_names) > 1 else group,
                {
                    'changes': changes,
                    'group': group_name_raw if len(group_names) > 1 else group,
                    'commit_message': form.data['commit_message'],
                    'group_names': group_names
                },
            )

            flask.flash(
                'Successfully created background request for modifying groups {}'.format(', '.join(group_names)),
                'success'
            )

            return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask_flash_form_error(form)

    else:
        if requests_in_processing:
            flask.flash(flask.Markup(
                '<a href="/queue/essence={}|{},status=executing|enqueued">'
                'This group has modification requests</a>'.format(group, group_name_raw)
            ), 'info')
        form.process()

    return flask.render_template(
        'group_card.html', modify_form=form, readonly=readonly, card=card, group=group, tag=None, tags=tags,
        slb_names=slb_names, hbf_macroses=hbf_macroses, another_groups=another_groups,
        samogon_volumes=DEFAULT_SAMOGON_VOLUMES, default_volumes=DEFAULT_VOLUMES, select_tag='trunk'
    )


@reports_blueprint.route('/group', methods=["GET", "POST"])
@reports_blueprint.route('/<tag>/groups/<group>', methods=["GET", "POST"])
@logger.time_counting()
def tag_group(tag=None, group=None):
    tag = flask.request.args.get('tag', tag)
    group = flask.request.args.get('group', group)

    if tag == 'trunk' or tag is None:
        return flask.redirect(flask.url_for('.trunk_group', group=group))

    if group not in flask.current_app._context.states.get_all_groups(tag):
        return flask.render_template(
            'custom_404.html',
            info='The group {} is not found in {}. Maybe it is not created yet.'.format(
                group, tag
            )
        )

    try:
        card = flask.current_app._context.states.get_group_card(group, tag)
    except requests.ConnectionError:
        return flask.render_template(
            'no_data_info.html', essence=group, essence_type='group',
            info='No group card was found for this {} tag. Maybe the tag is too old.'.format(tag)
        )

    tags = {k: '' for k in flask.current_app._context.states.get_gencfg_tags()}
    tags[tag] = 'selected'

    my_hbf_macroses = [card.get('properties', {}).get('hbf_parent_macros', '')]
    hbf_macroses = {card.get('properties', {}).get('hbf_parent_macros', ''): 'selected'}

    slb_names = {
        x: 'selected'
        for x in (card.get('properties', {}).get('mtn', {}).get('tunnels', {}).get('hbf_slb_name', []) or [])
    }

    instances_info = flask.current_app._context.states.get_group_instances(group, tag)
    alias_parent = app_states().get_trunk_hbf_macro(card['properties']['hbf_parent_macros']).get('parent_macros')

    form = ModifyGroupCardForm(group=group, card=card, instances_info=instances_info, readonly=True,
                               my_hbf_macroses=my_hbf_macroses, my_hbf_ranges=[], alias_parent=alias_parent)
    form.process()

    return flask.render_template(
        'group_card.html', modify_form=form, readonly=True, card=card, group=group, tag=tag, tags=tags,
        slb_names=slb_names, hbf_macroses=hbf_macroses, select_tag=tag
    )


@reports_blueprint.route('/recluster_in_dynamic/<group>', methods=["GET", "POST"])
@logger.time_counting()
def recluster_in_dynamic(group):
    if not is_admin():
        return flask.render_template(
            'no_data_info.html', essence='RECLUSTER IN DYNAMIC', essence_type='group',
            info=DYNAMIC_CLOSED_MESSAGE_2,
        )

    def compute_default_form_fields_from_api(card):
        instances_info = flask.current_app._context.states.get_group_instances(group)

        instances_count = len(instances_info)
        if instances_count:
            avg_power = int(sum(i['power'] for i in instances_info) / instances_count)
        else:
            avg_power = 0

        memory_guarantee = gbs(card['reqs']['instances']['memory_guarantee'])
        hdd_guarantee = gbs(card['reqs']['instances']['disk'])
        ssd_guarantee = gbs(card['reqs']['instances']['ssd'])
        net_guarantee = mbits(card['reqs']['instances']['net_guarantee'])
        volumes = json.dumps(simplify_volumes(card['reqs']['volumes']), indent=4)

        return {
            'target_instances_count': instances_count,
            'target_instance_power': avg_power / 40.,
            'target_instance_memory': memory_guarantee,
            'target_instance_disk': hdd_guarantee,
            'target_instance_ssd': ssd_guarantee,
            'target_instance_net': net_guarantee,
            'volumes': volumes
        }

    flask.flash(flask.Markup(DYNAMIC_CLOSED_MESSAGE_2), 'info')

    parent = None
    if group.endswith('_GUEST'):
        parent = group[0:-6]
        flask.flash(
            'Editing of this GUEST groups is not supported. To change this group, you must change {}'.format(parent),
            'info'
        )

    group_name_raw = group
    for loc in ('MAN_', 'VLA_', 'SAS_', 'MSK_IVA_', 'MSK_MYT_'):
        if group.startswith(loc):
            group_name_raw = group.replace(loc, '')
            break

    requests_in_processing = flask.current_app._context.states.get_requests(
        {
            '$and': [
                {'$or': [{'essence': group}, {'essence': group_name_raw}]},
                {'$or': [{'status': 'enqueued'}, {'status': 'executing'}]}
            ]
        },
        limit=1
    )

    another_groups = {x: '' for x in app_states().get_all_groups() if group_name_raw in x}
    another_groups[group] = 'selected'

    card = flask.current_app._context.states.get_group_card(group)
    group_owners = card.get('resolved_owners', []) or card.get('owners', [])
    owner = (flask.g.yandex_login in group_owners) or is_admin()
    readonly = not owner or bool(parent)

    form = ReclusterInDynamicForm(readonly)
    default_fields = compute_default_form_fields_from_api(card)

    if form.submit.data and form.validate_on_submit():
        prefix = 'recluster_in_dynamic-'
        changed_form_fields = {k: v for k, v in default_fields.iteritems()}
        group_names = [group]
        for field in form:
            key = field.name.split(prefix, 1)[1]
            if key in default_fields and default_fields[key] != field.data:
                changed_form_fields[key] = field.data
            elif key in ['another_groups']:
                print('another_groups: {}'.format(field.data))
                group_names = [x for x in field.data]

        if changed_form_fields == default_fields:
            flask.flash('No changes requested. Please change some fields', 'error')
        elif not group_names:
            flask.flash('Select one or more replication groups', 'error')
        else:
            print('CHANGED: {}'.format(changed_form_fields['volumes']))

            if default_fields['volumes'] == json.dumps(json.loads(changed_form_fields['volumes']), indent=4):
                changed_form_fields.pop('volumes')
            else:
                changed_form_fields['volumes'] =\
                    json.dumps(desimplify_volumes(json.loads(changed_form_fields['volumes'])))
                print('d: {}'.format(default_fields['volumes']))
                print('c: {}'.format(changed_form_fields['volumes']))

            changed_form_fields.update({
                'group_names': group_names,
                'commit_message': form.commit_message.data
            })
            push_task(
                'recluster_in_all_dynamic',
                group_name_raw if len(group_names) > 1 else group,
                changed_form_fields
            )

            flask.flash(
                'Successfully created background request for recluster groups {}'.format(', '.join(group_names)),
                'success'
            )

            return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask_flash_form_error(form)

    else:
        if requests_in_processing:
            flask.flash(flask.Markup(
                '<a href="/queue/essence={}|{},status=executing|enqueued">'
                'This group has modification requests</a>'.format(group, group_name_raw)
            ), 'info')
        populate_form(form, default_fields)

    return flask.render_template('recluster_in_dynamic.html', recluster_form=form, group=group,
                                 another_groups=another_groups, samogon_volumes=DEFAULT_SAMOGON_VOLUMES,
                                 default_volumes=DEFAULT_VOLUMES)


@reports_blueprint.route('/remove_group/<group>', methods=['GET', 'POST'])
@logger.time_counting()
def remove_group(group):
    group_owners = app_states().get_group_card(group).get('resolved_owners', [])
    is_owner = group in flask.g.my_groups or flask.g.yandex_login in group_owners or is_admin()

    if not is_owner:
        return flask.render_template(
            'no_data_info.html', essence=group, essence_type='group', info='Permission denied.'
        )

    count_slaves = len(flask.current_app._context.states.get_group_card(group)['slaves'])
    if count_slaves > 0:
        return flask.render_template(
            'no_data_info.html', essence=group, essence_type='group', info='Group {} has {} slaves. Remove slaves first.'.format(
                group, count_slaves
            )
        )

    try:
        group_online = flask.current_app._context.states.get_group_online(group)
    except Exception:
        return flask.render_template(
            'no_data_info.html', essence=group, essence_type='group', info='API `{}` response error.'.format(
                'https://api.gencfg.yandex-team.ru/online/groups/{}/instances'.format(group)
            )
        )

    rename = {
        'diff_count_hosts': 'Count hosts',
        'new_hosts': 'New hosts',
        'del_hosts': 'Removed hosts',
        'power': 'CPU',
        'porto_limits.memory_guarantee': 'Memory guarantee (porto)',
        'porto_limits.memory_limit': 'Memory limit (porto)'
    }

    form = RemoveGroupForm(not is_owner, flask.g.my_groups, flask.g.yandex_login in ADMINS, is_owner, group)

    if form.submit.data and form.validate_on_submit():
        group = form.group.data
        push_task('remove_group', group, form.data)

        flask.flash('Successfully created background request for removing group {}'.format(group), 'success')

        return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask.flash('There was a problem when submitting the form', 'error')

    else:
        form.process()

    return flask.render_template('remove_group.html', group=group, form=form, group_online=group_online, rename=rename)


@reports_blueprint.route('/create_macros', methods=['GET', 'POST'])
@logger.time_counting()
def create_macros():
    return flask.render_template('no_data_info.html', essence='Create macro', essence_type='macros',
                                 info='Creating macros on the GenCfg side no longer works. '
                                      'Create a macro in <a href="https://racktables.yandex-team.ru/index.php?page=services&tab=projects">RackTables</a>.')


@reports_blueprint.route('/remove_macros/<macros>', methods=['GET', 'POST'])
@logger.time_counting()
def remove_macros(macros):
    my_macroses = [x['name'] for x in flask.current_app._context.states.get_user_hbf_macroses(flask.g.yandex_login)]
    is_owner = macros in my_macroses
    readonly = not is_owner and not is_admin()

    if not is_owner and not is_admin():
        return flask.render_template(
            'no_data_info.html', essence=macros, essence_type='macros', info='Permission denied.'
        )

    form = RemoveMacrosForm(readonly, my_macroses, flask.g.yandex_login in ADMINS, macros)

    if form.submit.data and form.validate_on_submit():
        macros = form.macros.data
        push_task('remove_macros', macros, form.data)

        flask.flash('Successfully created background request for removing group {}'.format(macros), 'success')

        return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask.flash('There was a problem when submitting the form', 'error')

    else:
        form.process()

    return flask.render_template('remove_macros.html', macros=macros, form=form)


@reports_blueprint.route('/build_improxy/', methods=['GET', 'POST'])
@logger.time_counting()
def build_improxy():
    form = BuildImproxyForm()

    if form.submit.data and form.validate_on_submit():
        push_task('build_improxy', 'IMPROXY', form.data)
        flask.flash('Successfully created background request for build improxy', 'success')
        return flask.redirect(flask.url_for('.background_tasks'))
    elif form.is_submitted():
        flask.flash('There was a problem when submitting the form', 'error')
    else:
        form.process()

    return flask.render_template('build_improxy.html', form=form)


@reports_blueprint.route('/background_tasks')
@logger.time_counting()
def background_tasks():
    user_requests = change_requests.find_requests(mongo.mongo_db, {'author': flask.g.yandex_login})

    for user_request in user_requests:
        user_request['sandbox_task_url'] = sandbox_tasks.sandbox_task_url(user_request['sb_task_id'])

        if user_request.get('commit', -1) < 0:
            user_request['commit'] = ''
        user_request['arcadia_commit_url'] = arcadia_commit_url(user_request['commit'])

    # flask.flash(u'В связи с обновлением выполнение заявок приостановлено', 'info')
    return flask.render_template('background_tasks.html', requests=user_requests, is_admin=is_admin())


@reports_blueprint.route('/macroses')
@logger.time_counting()
def macroses():
    if is_admin():
        macroses = flask.current_app._context.states.get_trunk_hbf_macroses()
    else:
        macroses = flask.current_app._context.states.get_user_hbf_macroses(flask.g.yandex_login)

    return flask.render_template('macroses.html', macroses=macroses)


@reports_blueprint.route('/macros/<macros>', methods=["GET", "POST"])
@logger.time_counting()
def card_macros(macros):
    list_macroses = flask.current_app._context.states.get_trunk_hbf_macroses()
    my_macroses = flask.current_app._context.states.get_user_hbf_macroses(flask.g.yandex_login)

    data_macros = None
    for data in list_macroses:
        if data.get('name') == macros:
            data_macros = data
            break

    if not data_macros:
        return flask.abort(404)

    owner = flask.g.yandex_login in data_macros.get('resolved_owners', [])
    readonly = not owner and not is_admin()

    parents = {x['name']: '' for x in my_macroses}
    if is_admin():
        parents = {x['name']: '' for x in list_macroses}
    parents[data_macros.get('parent_macros') or ''] = 'selected'

    default_card = {
        'name': data_macros.get('name'),
        'parent': data_macros.get('parent_macros') or '',
        # 'description': data_macros.get('description'),
        'owners': data_macros.get('owners')
    }

    key_path = {
        # 'description': 'description',
        'owners': 'owners',
        'parent': 'parent'
    }

    form = ModifyMacrosForm(True, [x['name'] for x in my_macroses], flask.g.yandex_login in ADMINS, data_macros)

    if form.submit.data and form.validate_on_submit():
        prefix = 'modify_macros-'
        changes = {}
        for field in form:
            key = field.name.split(prefix, 1)[1]
            if key in default_card and key in key_path and default_card[key] != field.data:
                value = field.data
                if key in ('owners',):
                    value = ','.join(field.data)
                changes[key_path[key].replace('.', '!')] = value

        if not changes:
            flask.flash('No changes requested. Please change some fields', 'error')

        else:
            push_task(
                'modify_macros',
                macros,
                {'changes': changes, 'macros': macros, 'commit_message': form.data['commit_message']},
            )

            flask.flash('Successfully created background request for modifying macros {}'.format(macros), 'success')

            return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask_flash_form_error(form)

    else:
        form.process()

    flask.flash(
        flask.Markup('Editing macros on the Gencfg side no longer works. Edit your macro in <a href="https://racktables.yandex-team.ru/index.php?page=services&tab=projects">RackTables</a>.'),
        'info'
    )
    return flask.render_template(
        'view_macros.html', macros=data_macros['name'], parents=parents, modify_macros=form, readonly=readonly
    )


@reports_blueprint.route('/queue')
@reports_blueprint.route('/queue/<query>')
@logger.time_counting()
def queue(query=''):
    filter_tasks = {}
    if query:
        filter_tasks = {'$and': []}
        parts = [pair.split('=') for pair in query.split(',')]
        for key, values in parts:
            if '|' in values:
                list_values = values.split('|')
                filter_tasks['$and'].append({'$or': [{key: x} for x in list_values]})
            else:
                filter_tasks['$and'].append({key: values})
    print('filter_tasks: {}'.format(filter_tasks))

    mongo_requests = change_requests.find_requests(mongo.mongo_db, filter_tasks, limit=200)

    requests_queue = []
    for mongo_request in mongo_requests:
        active_actions = False
        if mongo_request['author'] == flask.g.yandex_login or flask.g.yandex_login in ADMINS:
            active_actions = True

        if mongo_request.get('commit', -1) < 0:
            mongo_request['commit'] = ''
        requests_queue.append((active_actions, mongo_request))

    # flask.flash(u'В связи с обновлением выполнение заявок приостановлено', 'info')
    return flask.render_template('request_queue.html', requests=requests_queue[:500], is_admin=is_admin())


@reports_blueprint.route('/details/<request_id>')
@logger.time_counting()
def details(request_id):
    mongo_request = change_requests.find_request(mongo.mongo_db, request_id)

    if not mongo_request:
        return flask.abort(404)

    request_info = {
        'essence': mongo_request['essence']['name'],
        'author': mongo_request['author'],
        'type': mongo_request['type'],
        'added': mongo_request['time']['added'].strftime('%m-%d %H:%M'),
        'status': mongo_request['status'],
        'sb_task_id': mongo_request.get('sb_task_id', ''),
        'commit': mongo_request.get('commit', '')
    }

    request_params = {k: v for k, v in mongo_request['params'].items() if k not in ('csrf_token', 'submit')}
    if request_params.get('changes'):
        request_params.update(request_params['changes'])
        request_params.pop('changes')

    return flask.render_template('request_details.html', request_info=request_info, request_params=request_params)


@reports_blueprint.route('/list_types/<types_class>')
@logger.time_counting()
def list_types(types_class):
    if types_class == 'itypes':
        types_data = flask.current_app._context.states.get_trunk_itypes()
    elif types_class == 'ctypes':
        types_data = flask.current_app._context.states.get_trunk_ctypes()
    elif types_class == 'metaprj':
        types_data = flask.current_app._context.states.get_trunk_metaprjs()
    else:
        return flask.abort(404)

    return flask.render_template('list_types.html', types_name=types_class, types_data=types_data)


@reports_blueprint.route('/create_type', methods=["GET", "POST"])
@logger.time_counting()
def create_type():
    form = CreateTypeForm()

    if form.submit.data and form.validate_on_submit():
        form_data = {
            'action': 'add',
            'type_class': form.type_class.data,
            'type_name': form.type_name.data,
            'fields': {
                'description': form.description.data
            },
            'commit_message': form.commit_message.data
        }

        push_task('manipulate_types', form.type_name.data, form_data)

        flask.flash(
            'Successfully created background request for creating type {}'.format(form.type_name.data), 'success'
        )

        return flask.redirect(flask.url_for('.background_tasks'))

    elif form.is_submitted():
        flask_flash_form_error(form)
        flask.flash('There was a problem when submitting the form', 'error')

    else:
        form.process()

    return flask.render_template('create_type.html', form=form)


@reports_blueprint.route('/cancel_request/<request_id>/<redir>', methods=['POST'])
@logger.time_counting()
def cancel_request(request_id, redir):
    request = change_requests.find_request(mongo.mongo_db, request_id)

    if request.get('author') != flask.g.yandex_login and not is_admin():
        return flask.redirect(flask.url_for('.{}'.format(redir)))

    if not is_admin() and request.get('sb_task_id'):
        return flask.redirect(flask.url_for('.{}'.format(redir)))

    change_requests.skip_request(mongo.mongo_db, request['id'])
    return flask.redirect(flask.url_for('.{}'.format(redir)))


@reports_blueprint.route('/copy_request/<request_id>/<redir>', methods=['POST'])
@logger.time_counting()
def copy_request(request_id, redir):
    request = change_requests.find_request(mongo.mongo_db, request_id)

    if request.get('author') != flask.g.yandex_login and not is_admin():
        return flask.redirect(flask.url_for('.{}'.format(redir)))

    change_requests.copy_request(mongo.mongo_db, request['id'], flask.g.yandex_login)
    return flask.redirect(flask.url_for('.{}'.format(redir)))


@reports_blueprint.route('/create_st_ticket/<request_id>/<redir>', methods=['POST'])
@logger.time_counting()
def create_st_ticket(request_id, redir):
    user_request = change_requests.find_request(mongo.mongo_db, request_id)

    if user_request is None:
        return flask.render_template(
            'no_data_info.html', essence=str(request_id), essence_type='request', info='Request not found.'
        )

    if user_request['author'] != flask.g.yandex_login and not is_admin():
        return flask.render_template(
            'no_data_info.html', essence=str(request_id), essence_type='request', info='Permission denied.'
        )

    response = requests.post(
        'https://st-api.yandex-team.ru/v2/issues',
        headers={'Authorization': 'OAuth {}'.format(flask.g.yandex_token)},
        json={
            'queue': 'GENCFG',
            'summary': '{type} request {request_id} finished with status {status}\n\n'.format(type=user_request['type'],
                                                                                              request_id=user_request['id'],
                                                                                              status=user_request['status']),
            'description': 'INFO:\n'
                           'Type: {type}\n'
                           'Author: {author}@\n'
                           'Sandbox task: https://sandbox.yandex-team.ru/task/{sb_task_id}/view\n\n'
                           'PARAMS:\n'
                           '{params}\n\n'
                           'EXCEPTIONS:\n'
                           '{exceptions}\n'.format(type=user_request['type'], author=user_request['author'],
                                                   sb_task_id=user_request['sb_task_id'],
                                                   params=json.dumps(user_request['params'], indent=4),
                                                   exceptions=json.dumps(user_request.get('exceptions', []), indent=4))
        }
    )

    if response.status_code != 201:
        return flask.render_template(
            'no_data_info.html', essence=str(request_id), essence_type='request',
            info=json.dumps(response.json())
        )

    flask.flash(flask.Markup(
        'Ticket was creared. <a href="https://st.yandex-team.ru/{0}">{0}</a>'.format(response.json()['key']),
    ), 'success')
    return flask.redirect(flask.url_for('.{}'.format(redir)))


@reports_blueprint.route('/statistic')
@reports_blueprint.route('/statistic/count/<count>')
@reports_blueprint.route('/statistic/query/<query>')
@reports_blueprint.route('/statistic/count/<count>/query/<query>')
@reports_blueprint.route('/statistic/query/<query>/count/<count>')
@logger.time_counting()
def statistic(count=100, query=None):
    filter_tasks = {}
    if query:
        parts = [pair.split('=') for pair in query.split(',')]
        filter_tasks = {k: v for k, v in parts}

    mongo_requests = list(mongo.mongo_db.background_tasks.find(filter_tasks).sort(
        'added', pymongo.DESCENDING
    ).limit(int(count)))

    colors = {
        'allocate_in_all_dynamic': 'rgb(0, 255, 0)',
        'recluster_in_all_dynamic': 'rgb(0, 255, 0)',
        'modify_group_card': 'rgb(0, 255, 0)',
        'remove_group': 'rgb(0, 255, 0)',
        'create_macros': 'rgb(0, 255, 0)',
        'all_request_types': 'rgb(0, 0, 255)'
    }
    timeline_queue = time_distribution('added', 'finished', mongo_requests)
    timeline_exec = time_distribution('started', 'finished', mongo_requests)
    timeline_count = time_count_distribution('added', mongo_requests)

    return flask.render_template(
        'statistic.html', timeline_queue=timeline_queue, timeline_exec=timeline_exec,
        timeline_count=timeline_count, colors=colors
    )


@reports_blueprint.route('/suggest/<term>')
@reports_blueprint.route('/suggest/', defaults={'term': ''})
@logger.time_counting()
def group_suggest(term):
    q = flask.request.args.get('term', term).strip()

    groups = []
    if '@' not in q or q.startswith('G@'):
        groups = [x for x in flask.current_app._context.states.get_all_groups() if q.replace('G@', '').upper() in x]

    macroses = []
    if '@' not in q or q.startswith('M@'):
        macroses = [
            x['name']
            for x in flask.current_app._context.states.get_trunk_hbf_macroses()
            if q.replace('M@', '').upper() in x['name']
        ]

    all_matched = groups + macroses

    # TODO: use flask.jsonify
    return flask.Response(response=json.dumps(all_matched), mimetype="application/json")


@reports_blueprint.route('/tagsuggest/<term>')
@reports_blueprint.route('/tagsuggest/', defaults={'term': ''})
@logger.time_counting()
def tag_suggest(term):
    q = flask.request.args.get('term', term).strip()

    all_matched = ['trunk']
    for tag in flask.current_app._context.states.get_gencfg_tags(None):
        if q in tag:
            all_matched.append(tag)

    return flask.Response(response=json.dumps(all_matched), mimetype="application/json")


@reports_blueprint.route('/ownersuggest/<query>')
@reports_blueprint.route('/ownersuggest/', defaults={'query': ''})
@logger.time_counting()
def owner_suggest(query):
    query = flask.request.args.get('query', query).strip()
    data = app_states().get_trunk_available_owners()
    separator = ','
    max_size = 40

    terms = query.strip().split(separator)
    prefix = separator.join(terms[:-1])
    term = terms[-1]

    if not term.strip():
        return flask.Response(response=json.dumps([]), mimetype="application/json")

    term_prefix = ' ' * (len(term) - len(term.lstrip()))

    matched = []
    for item in data:
        if len(matched) >= max_size:
            break
        if item.startswith(term.strip()):
            matched.append(item)

    matched = sorted(matched, key=lambda x: len(x))
    for item in data:
        if len(matched) >= max_size:
            break
        if term.strip() in item and item not in matched:
            matched.append(item)

    suggestions = [("{}{}{}{}".format(prefix, separator, term_prefix, x) if prefix else x) for x in matched]

    return flask.Response(response=json.dumps(suggestions), mimetype="application/json")


@reports_blueprint.route('/static/<path:path>')
@logger.time_counting()
def send_static(path):
    response = resource.find('/static/{}'.format(path))
    if path.endswith('.js'):
        mimetype = "application/javascript"
    elif path.endswith('.css'):
        mimetype = "text/css"
    else:
        mimetype = "text/plain"
    return flask.Response(response=response, mimetype=mimetype)


@reports_blueprint.route('/about')
@logger.time_counting()
def about():
    return flask.render_template('about.html', page='About')


@reports_blueprint.route('/status')
@logger.time_counting()
def status():
    last_tag = ''
    gencfg_revisions = flask.current_app._context.states.get_gencfg_revisions()
    for revision in gencfg_revisions:
        if 'tag' in revision:
            last_tag = revision['tag']
            continue
        revision['contains_in'] = last_tag

    return flask.render_template('status.html', page='Status', commits=gencfg_revisions)


@reports_blueprint.route('/favicon.ico')
def favicon():
    path = resource.find('/static/favicon.ico')
    print('FAVICON: {}'.format(type(path)))
    return flask.Response(response=resource.find('/static/favicon.ico'), mimetype='image/png')


def push_task(request_type, essence_name, request_params):
    change_requests.save_request(
        mongo_db=mongo.mongo_db,
        request_type=request_type,
        essence_name=essence_name,
        request_params=request_params,
        login=flask.g.yandex_login
    )


def gbs(in_bytes):
    return round(in_bytes / 1024.0 / 1024.0 / 1024.0, 2)


def mbits(in_bytes):
    return round(in_bytes * 8 / 1024.0 / 1024.0, 2)


def check_tag(tag):
    return re.compile("^(?:trunk|stable-[0-9]{3}-r[0-9]{1,4}|[0-9]{7,})$").match(tag)


def is_admin():
    return flask.g.yandex_login in ADMINS


def get_size_value_str(in_bytes, min_unit=None):
    units = ('B', 'KB', 'MB', 'GB', 'TB')
    unit_index = 0
    value = in_bytes
    while int(value / 1024.) > 0 and unit_index + 1 < len(units):
        value /= 1024.0
        unit_index += 1

    if min_unit and units.index(min_unit) <= unit_index:
        return '{:.2f} {}'.format(value, units[unit_index])
    return None


def app_context():
    return flask.current_app._context


def app_states():
    return app_context().states
