# coding: utf8
import pymongo
import datetime
import collections

import base


MongoStorage = collections.namedtuple('MongoStorage', ['uri', 'replicaset'])

HEARTBEAT_C = MongoStorage(
    uri=','.join([
        'myt0-4012.search.yandex.net:27017',
        'myt0-4019.search.yandex.net:27017',
        'sas1-6063.search.yandex.net:27017',
        'sas1-6136.search.yandex.net:27017',
        'vla1-3984.search.yandex.net:27017',
    ]),
    replicaset='heartbeat_mongodb_c'
)
HEARTBEAT_D = MongoStorage(
    uri=','.join([
        'myt0-4013.search.yandex.net:27017',
        'myt0-4020.search.yandex.net:27017',
        'sas4-4916.search.yandex.net:27017',
        'sas4-4917.search.yandex.net:27017',
        'vla1-6006.search.yandex.net:27017',
    ]),
    replicaset='heartbeat_mongodb_d'
)


@base.retry(5, 1)
def get_collection(db_name, collection_name, heartbeat=None):
    heartbeat = heartbeat or HEARTBEAT_C

    return pymongo.MongoReplicaSetClient(
        heartbeat.uri,
        connectTimeoutMS=15000,
        replicaSet=heartbeat.replicaset,
        w='majority',
        wtimeout=15000,
    )[db_name][collection_name]


@base.retry(5, 1)
def find_one(collection, query, sort_key=None, sort_value=None):
    sort_key = sort_key or '$natural'
    sort_value = sort_value or pymongo.ASCENDING
    requests = collection.find(query).sort(sort_key, sort_value).limit(1)
    request = requests[0] if requests.count() else None
    return request


@base.retry(5, 1)
def find_all(collection, query, sort_key=None, sort_value=None, limit=100):
    sort_key = sort_key or '$natural'
    sort_value = sort_value or pymongo.ASCENDING
    requests = collection.find(query).sort(sort_key, sort_value).limit(limit)
    requests = list(requests) if requests.count() else None
    return requests


@base.retry(5, 1)
def update_one(collection, query, updates, upsert=True):
    collection.update_one(query, {"$set": updates}, upsert=upsert)


@base.retry(5, 1)
def get_last_tag():
    collection = get_collection('topology_commits', 'tags')
    tags = find_all(collection, {}, sort_key='commit', sort_value=-1)
    return {k: v for k, v in tags[0].iteritems() if k not in ('_id',)} if tags else None


@base.retry(5, 1)
def get_last_tested_commit():
    collection = get_collection('topology_commits', 'commits')
    commits = find_all(
        collection,
        {'$or': [{'test_passed': True}, {'test_passed': False}]},
        sort_key='commit', sort_value=-1
    )
    return {k: v for k, v in commits[0].iteritems() if k not in ('_id',)} if commits else None


@base.retry(5, 1)
def get_list_request(query=None):
    collection = get_collection('ggdb', 'gencfg_gui_requests')
    return find_all(collection, query, sort_key='time.added', sort_value=pymongo.DESCENDING)


@base.retry(5, 1)
def get_first_request(request_type=None):
    query = {'$and': [
        {'$or': [
            {'status': 'enqueued'},
            {'status': 'executing'}
        ]}
    ]}
    if request_type:
        query['$and'].append({'type': request_type})

    requests = get_list_request(query)
    return requests[-1] if requests else None


@base.retry(5, 1)
def get_request_delay(request_type=None):
    request = get_first_request(request_type)
    request_create_time = request['time']['added'] if request else None
    time_now = datetime.datetime.now()

    if request_create_time is None:
        return 0
    return (time_now - request_create_time).total_seconds()
