# coding: utf8
import base
import datetime

import logger


GROUP_SUCCESS = ('SUCCESS',)
GROUP_RELEASED = ('RELEASED',)
GROUP_SUCCESSFUL = GROUP_SUCCESS + GROUP_RELEASED
GROUP_ERROR = ('EXCEPTION', 'TIMEOUT', 'NOT_RELEASED', 'FAILURE', 'STOPPED')
GROUP_FINISHED = GROUP_SUCCESSFUL + GROUP_ERROR
SANDBOX_API_URL = 'http://sandbox.yandex-team.ru/api/v1.0'

collector_logger = logger.LoggerWrapper.get_logger('collector')


def _get_datetime_from_timestamp(timestamp):
    result = ''
    for fmt in ('%Y-%m-%dT%H:%M:%S.%ZZ', '%Y-%m-%dT%H:%M:%S.%zZ', '%Y-%m-%dT%H:%M:%S.%fZ', '%Y-%m-%dT%H:%M:%SZ'):
        try:
            result = datetime.datetime.strptime(timestamp, fmt)
            result = result + datetime.timedelta(hours=3)
            break
        except Exception:
            pass
    return result


def get_info_by_task(sb_task):
    result = {'info': {}, 'input': {}}

    for key in ('id', 'type', 'description', 'status', 'owner'):
        result['info'][key] = sb_task.get(key, '')

    if sb_task.get('status', '') in GROUP_SUCCESSFUL:
        result['info']['status_group'] = 'SUCCESS'
    elif sb_task.get('status', '') in GROUP_ERROR:
        result['info']['status_group'] = 'ERROR'
    else:
        result['info']['status_group'] = 'NOT_FINISHED'

    execution = sb_task.get('time', {'created': '', 'updated': ''})
    result['info']['created'] = _get_datetime_from_timestamp(execution['created'])
    result['info']['updated'] = _get_datetime_from_timestamp(execution['updated'])

    for key, value in sb_task.get('input_parameters', {}).iteritems():
        result['input'][key] = value

    return result


def get_list_tasks(task_type, group_statuses=None, limit=10):
    json_data = base.http_get(base.make_url(
        SANDBOX_API_URL, 'task', limit=limit, children='true', order='-id', type=task_type
    ))

    list_tasks = [] if not json_data else [get_info_by_task(task) for task in json_data.get('items', [])]

    result = []
    for task in list_tasks:
        if group_statuses and task['info']['status'] not in group_statuses:
            continue
        result.append(task)

    collector_logger.info('LIST TASKS ({}): {}'.format(
        task_type,
        ', '.join(str(x['info']['id']) for x in result)
    ))
    return result


def get_last_task(task_type, group_statuses=None):
    list_tasks = get_list_tasks(task_type, group_statuses, 10)
    last_task = list_tasks[0] if list_tasks else None

    if last_task is None:
        # retry with longer history
        list_tasks = get_list_tasks(task_type, group_statuses, 200)
        last_task = list_tasks[0] if list_tasks else None
        if last_task is None:
            raise ValueError('failed to get the last task')

    collector_logger.info('LAST TASK ({}): {}'.format(
        task_type, last_task['info']['id'] if last_task else None
    ))
    return last_task


def get_status_from_task(task, wait_release=False):
    success_statuses = GROUP_RELEASED if wait_release else GROUP_SUCCESS
    return task['info']['status'] in success_statuses
