# coding: utf8
import re
import datetime
import subprocess
import xml.etree.ElementTree as ET

import base


GENCFG_DB_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/data/gencfg_db'
GENCFG_CODE_URL = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/gencfg'
GENCFG_TAGS_URL = 'svn+ssh://arcadia.yandex.ru/arc/tags/gencfg'
GENCFG_TAG_URL = 'svn+ssh://arcadia.yandex.ru/arc/tags/gencfg/{}'
TAG_PATTERN = re.compile(r'^stable-(?P<branch>\d+)-r(?P<tag>\d+)$')


def _parse_svn_log(output):
    result = []
    for line in output.split('\n'):
        if not line:
            continue

        parts = line.split(' | ')

        if len(parts) != 4:
            continue

        commit, user, time, _ = parts
        commit = int(commit[1:])
        date, time, _ = time.split(' ', 2)

        result.append(
            (commit, user, datetime.datetime.strptime('{}T{}'.format(date, time), '%Y-%m-%dT%H:%M:%S'))
        )

    return result


@base.retry(5, 1)
def _get_last_commits(url, limit):
    output = subprocess.check_output(['svn', 'log', url, '-l', str(limit)])
    return sorted(_parse_svn_log(output), key=lambda x: x[0], reverse=True)


def get_last_db_commits(limit=10):
    return _get_last_commits(GENCFG_DB_URL, limit)


def get_last_code_commits(limit=10):
    return _get_last_commits(GENCFG_CODE_URL, limit)


def get_last_merge_commits(limit=20):
    commits = get_last_db_commits(limit)
    commits.extend(get_last_code_commits(limit))

    commits.sort(key=lambda x: x[0], reverse=True)

    if len(commits) > limit:
        return commits[0:limit]
    return commits


@base.retry(5, 1)
def get_list_last_tags():
    xml_tags = subprocess.check_output(['svn', 'ls', '--xml', GENCFG_TAGS_URL])
    tags_tree = ET.fromstring(xml_tags)

    filtered_tags = []
    for entry in tags_tree[0]:
        if not TAG_PATTERN.match(entry[0].text):
            continue
        filtered_tags.append((entry[0].text, int(entry[1].attrib['revision'])))

    filtered_tags.sort(key=lambda x: x[1], reverse=True)

    return filtered_tags


@base.retry(5, 1)
def get_commits_in_tag(tag_name, limit=10):
    output = subprocess.check_output(['svn', 'log', '-l', str(limit), GENCFG_TAG_URL.format(tag_name)])
    return sorted(_parse_svn_log(output), key=lambda x: x[0], reverse=True)


def get_commits_after_tag(last_tag_name, limit=10):
    commits_in_tag = get_commits_in_tag(last_tag_name, 1)
    last_commit_in_tag = commits_in_tag[0] if commits_in_tag else 0
    all_last_commits = get_last_merge_commits(100)
    commits_after_tag = filter(lambda x: x[0] > last_commit_in_tag[0], all_last_commits)

    if len(commits_after_tag) > limit:
        return commits_after_tag[0:limit]
    return commits_after_tag


def get_commits_not_in_tag(limit=10):
    list_last_tags = get_list_last_tags()
    last_tag_name = list_last_tags[0][0] if list_last_tags else None
    print(last_tag_name)
    if last_tag_name:
        return get_commits_after_tag(last_tag_name, limit)
    else:
        return get_last_merge_commits(limit)
