# coding: utf8
import time
import functools
import traceback
import threading

import logger
import telegram_bot_api as tba


tbot_logger = logger.LoggerWrapper.get_logger('tbot')


def telegram_response_validator(raise_on_fail=True):
    def decorator(func):
        @functools.wraps(func)
        def wrapper(self, *args, **kwargs):
            raise_on_fail = True
            try:
                if 'raise_on_fail' in kwargs:
                    raise_on_fail = kwargs['raise_on_fail']
                    kwargs.pop('raise_on_fail')
                return func(self, *args, **kwargs)
            except Exception as e:
                tbot_logger.error('{} - {}: {}'.format(func.__name__, type(e).__name__, e))
                if raise_on_fail:
                    raise
        return wrapper
    return decorator


class TelegramBot(object):
    def __init__(self, token):
        self.token = token

    @telegram_response_validator()
    def get_me(self):
        return tba.get_me(self.token)

    @telegram_response_validator()
    def send_message(self, chat_id, text):
        return tba.send_message(self.token, chat_id, text)

    @telegram_response_validator(raise_on_fail=False)
    def get_updates(self, offset):
        return tba.get_updates(self.token, offset)

    def update_processing(self, update, callback, raise_on_error=False):
        try:
            if not update.get('message') or not update.get('update_id'):
                return

            offset = update.update_id + 1
            callback(update.message)
            return offset

        except Exception:
            tbot_logger.traceback(traceback.format_exc(limit=8))
            if raise_on_error:
                raise

    def checking_updates(self, callback, delay=10, raise_on_error=False):
        offset = 0
        new_delay = delay
        while True:
            updates = self.get_updates(offset) or []
            for i, update in enumerate(updates):
                new_offset = self.update_processing(update, callback, raise_on_error)
                new_delay = delay if not new_offset else int(new_delay / 2)
                offset = new_offset or offset
            time.sleep(new_delay)

    def async_checking_updates(self, callback, delay=3, raise_on_error=False):
        process = threading.Thread(target=self.checking_updates, args=(callback, delay, raise_on_error))
        process.start()
        return process

    def working(self, worker, delay=10, raise_on_error=False):
        while True:
            try:
                worker()
            except Exception:
                tbot_logger.traceback(traceback.format_exc(limit=8))
                if raise_on_error:
                    raise
            time.sleep(delay)

    def async_working(self, worker, delay=10, raise_on_error=False):
        process = threading.Thread(target=self.working, args=(worker, delay, raise_on_error))
        process.start()
        return process
