# coding: utf8
import datetime

import sdk.tasks as tasks


class GencfgStateNotificator(tasks.TBotTask):
    delay_sections = ('test', 'build')
    state_sections = ('TEST_CONFIG_GENERATOR_2', 'BUILD_CONFIG_GENERATOR')

    class Parameters(tasks.TBotTask.Parameters):
        pass

    class Context(tasks.TBotTask.Context):
        monitoring = {}
        messages = []
        error = False

    def send_bad_notify(self, text):
        self.send_notifications_to_followers(
            '{}\nhttps://yasm.yandex-team.ru/panel/GencfgMonitoring/'.format(
                text
            )
        )

    def send_good_notify(self, text):
        self.send_notifications_to_followers(text)

    def on_prepare(self):
        self.Context.monitoring = self.get_gencfg_monitoring_state()
        self.Context.messages = []

    def to_date(self, date_string):
        return datetime.datetime.strptime(date_string, '%Y-%m-%d %H:%M:%S')

    @property
    def gm_states(self):
        return self.Context.monitoring['states']

    def gm_state(self, item):
        return self.gm_states[item]

    def gm_state_value(self, item):
        return self.gm_states[item]['value']

    def gm_state_updated(self, item):
        return self.to_date(self.gm_states[item]['updated'])

    @property
    def db_states(self):
        return self.state['states']

    def db_state(self, item):
        return self.db_states[item]

    def db_state_value(self, item):
        return self.db_states[item]['value']

    def db_state_updated(self, item):
        return self.db_states[item]['updated']

    @property
    def gm_delays(self):
        return self.Context.monitoring['delays']

    def gm_delay(self, item):
        return self.gm_delays[item]

    def gm_delay_value(self, item):
        return int(self.gm_delays[item]['value'])

    def gm_delay_updated(self, item):
        return self.to_date(self.gm_delays[item]['updated'])

    @property
    def db_delays(self):
        return self.state['delays']

    def db_delay(self, item):
        return self.db_delays[item]

    def db_delay_value(self, item):
        return int(self.db_delays[item]['value'])

    def db_delay_updated(self, item):
        return self.db_delays[item]['updated']

    def db_max_delay(self, item):
        return self.settings['delays'][item]

    def is_delay_actuality(self, item):
        return self.db_delay_updated(item) >= self.gm_delay_updated(item)

    def is_state_actuality(self, item):
        return self.db_state_updated(item) >= self.gm_state_updated(item)

    def is_delay_sections_actuality(self):
        delay_actuality = True
        for delay_section in self.delay_sections:
            if not self.is_delay_actuality(delay_section):
                delay_actuality = False
                break
        return delay_actuality

    def is_state_sections_actuality(self):
        state_actuality = True
        for state_section in self.state_sections:
            if not self.is_state_actuality(state_section):
                state_actuality = False
                break
        return state_actuality

    def db_full_error_state(self):
        for section in self.state_sections:
            if not self.db_state_value(section):
                return True

        for section in self.delay_sections:
            if self.db_delay_value(section) > self.db_max_delay(section):
                return True

        return False

    def db_update_state(self, item, value):
        self.upadte_state({
            'states.{}.value'.format(item): value,
            'states.{}.updated'.format(item): datetime.datetime.utcnow()
        })

    def db_update_delay(self, item, value):
        self.upadte_state({
            'delays.{}.value'.format(item): value,
            'delays.{}.updated'.format(item): datetime.datetime.utcnow()
        })

    def on_execute(self):
        if self.is_delay_sections_actuality() and self.is_state_sections_actuality():
            self.logger.info('INFO IN DB IS ACTUALITY - SKIP PROCESSING')
            return

        self.Context.error = self.db_full_error_state()

        self.check_state_status()
        self.check_delay_status()

        self.logger.info('MESSAGES: {}'.format(self.Context.messages))
        self.logger.info('FULL ERROR: {}'.format(self.Context.error))

        self.send_notify_if_needed()

        self.update_task_state()

    def check_state_status(self):
        for state_section in self.state_sections:
            if not self.gm_state_value(state_section)['ok']:
                self.Context.messages.append('{}: {}\nhttps://sandbox.yandex-team.ru/task/{}/view'.format(
                    state_section,
                    self.gm_state_value(state_section)['status'],
                    self.gm_state_value(state_section)['id']
                ))
            self.logger.info('{}: STATE {}, DB {}'.format(
                state_section,
                self.gm_state_value(state_section)['ok'],
                self.db_state_value(state_section)
            ))

    def check_delay_status(self):
        for delay_section in self.delay_sections:
            if self.gm_delay_value(delay_section) > self.db_max_delay(delay_section):
                self.Context.messages.append('{}: has delay {}m'.format(
                    delay_section.upper(),
                    self.gm_delay_value(delay_section) // 60
                ))
            self.logger.info('{}: STATE {}, DB {}'.format(
                delay_section,
                self.gm_delay_value(delay_section),
                self.db_max_delay(delay_section)
            ))

    def send_notify_if_needed(self):
        if self.Context.error and len(self.Context.messages) == 0:
            self.send_good_notify('Test and Build work normally.')
            self.logger.info('Sent notify about work normally')
        elif not self.Context.error and len(self.Context.messages) > 0:
            self.send_bad_notify('\n\n'.join(self.Context.messages))
            self.logger.info('Sent notify about some errors')

    def update_task_state(self):
        for state_section in self.state_sections:
            if self.gm_state_value(state_section)['ok'] != self.db_state_value(state_section):
                self.db_update_state(state_section, self.gm_state_value(state_section)['ok'])
                self.logger.info('STATE {} - updated'.format(state_section))

        for delay_section in self.delay_sections:
            self.db_update_delay(delay_section, self.gm_delay_value(delay_section))
            self.logger.info('DELAY {} ({}) - updated'.format(delay_section, self.gm_delay_value(delay_section)))
