# vim: ts=4 sw=4 et si smarttab

import os
import hashlib
from doit.tools import create_folder


BUILD_PATH = 'build'
LIB_PATH = os.path.join(BUILD_PATH, 'lib')
PYSITE_PATH = os.path.join(LIB_PATH, 'python3.4', 'site-packages')

DL_PATH = '.downloads'

PYSTICK_PATH = os.path.join(DL_PATH, 'pystick.tgz')
GENISYS_BUNDLE = 'dist/genisys.tgz'

VIRTUALENV_PATH = os.path.join(DL_PATH, 'virtualenv.tgz')
STATSITE_PATH = os.path.join(DL_PATH, 'statsite.tgz')
MONGODB_PATH = os.path.join(DL_PATH, 'mongodb.tgz')

MONGO_TOOLS = [
    '',  # mongo
    'd', 'dump', 'export', 'import', 'oplog', 'stat', 'top'
]

BUNDLE_PYTHON_LIBS = [
    # Required
    'setuptools==18.2',

    # Main deps
    'Flask==0.10.1',
    'Jinja2==2.8',
    'pymongo==3.0.3',
    'requests==2.7.0',
    'certifi==2015.11.20',
    'statsd==3.2.1',            # send stats to statsite
    'beautifulsoup4==4.4.1',    # for checker, html parsing
    'html5lib==0.9999999',      # for checker, html5 parsing, beautifulsoup dependency
    'six==1.10.0',              # beautifulsoup dependency

    # Flask plugins
    'Babel==2.0',
    'Flask-Babel==0.9',
    'Flask-Markdown==0.3',
    'Flask-WTF==0.12',
    'itsdangerous==0.24',
    'Markdown==2.6.2',
    'MarkupSafe==0.23',
    'pytz==2015.4',
    'speaklater==1.3',
    'Werkzeug==0.10.4',
    'WTForms==2.0.2',
]

PY_LAUNCHER_TPL = '''\
#!/bin/sh

script="$0"
while [ -h "$script" ]; do
    script="$(readlink "$script")"
done

fpath="$(cd $(dirname "$script"); pwd)/python3"

if [ ! -x "$fpath" ]; then
    echo "Error: Unable to find python3" >&2
    exit 1
fi

exec $fpath -s -B -tt "${script}.py" "$@"
'''

YANDEX_INTERNAL_ROOT_CA = '''\
-----BEGIN CERTIFICATE-----
MIIFGTCCAwGgAwIBAgIQJMM7ZIy2SYxCBgK7WcFwnjANBgkqhkiG9w0BAQ0FADAf
MR0wGwYDVQQDExRZYW5kZXhJbnRlcm5hbFJvb3RDQTAeFw0xMzAyMTExMzQxNDNa
Fw0zMzAyMTExMzUxNDJaMB8xHTAbBgNVBAMTFFlhbmRleEludGVybmFsUm9vdENB
MIICIjANBgkqhkiG9w0BAQEFAAOCAg8AMIICCgKCAgEAgb4xoQjBQ7oEFk8EHVGy
1pDEmPWw0Wgw5nX9RM7LL2xQWyUuEq+Lf9Dgh+O725aZ9+SO2oEs47DHHt81/fne
5N6xOftRrCpy8hGtUR/A3bvjnQgjs+zdXvcO9cTuuzzPTFSts/iZATZsAruiepMx
SGj9S1fGwvYws/yiXWNoNBz4Tu1Tlp0g+5fp/ADjnxc6DqNk6w01mJRDbx+6rlBO
aIH2tQmJXDVoFdrhmBK9qOfjxWlIYGy83TnrvdXwi5mKTMtpEREMgyNLX75UjpvO
NkZgBvEXPQq+g91wBGsWIE2sYlguXiBniQgAJOyRuSdTxcJoG8tZkLDPRi5RouWY
gxXr13edn1TRDGco2hkdtSUBlajBMSvAq+H0hkslzWD/R+BXkn9dh0/DFnxVt4XU
5JbFyd/sKV/rF4Vygfw9ssh1ZIWdqkfZ2QXOZ2gH4AEeoN/9vEfUPwqPVzL0XEZK
r4s2WjU9mE5tHrVsQOZ80wnvYHYi2JHbl0hr5ghs4RIyJwx6LEEnj2tzMFec4f7o
dQeSsZpgRJmpvpAfRTxhIRjZBrKxnMytedAkUPguBQwjVCn7+EaKiJfpu42JG8Mm
+/dHi+Q9Tc+0tX5pKOIpQMlMxMHw8MfPmUjC3AAd9lsmCtuybYoeN2IRdbzzchJ8
l1ZuoI3gH7pcIeElfVSqSBkCAwEAAaNRME8wCwYDVR0PBAQDAgGGMA8GA1UdEwEB
/wQFMAMBAf8wHQYDVR0OBBYEFKu5xf+h7+ZTHTM5IoTRdtQ3Ti1qMBAGCSsGAQQB
gjcVAQQDAgEAMA0GCSqGSIb3DQEBDQUAA4ICAQAVpyJ1qLjqRLC34F1UXkC3vxpO
nV6WgzpzA+DUNog4Y6RhTnh0Bsir+I+FTl0zFCm7JpT/3NP9VjfEitMkHehmHhQK
c7cIBZSF62K477OTvLz+9ku2O/bGTtYv9fAvR4BmzFfyPDoAKOjJSghD1p/7El+1
eSjvcUBzLnBUtxO/iYXRNo7B3+1qo4F5Hz7rPRLI0UWW/0UAfVCO2fFtyF6C1iEY
/q0Ldbf3YIaMkf2WgGhnX9yH/8OiIij2r0LVNHS811apyycjep8y/NkG4q1Z9jEi
VEX3P6NEL8dWtXQlvlNGMcfDT3lmB+tS32CPEUwce/Ble646rukbERRwFfxXojpf
C6ium+LtJc7qnK6ygnYF4D6mz4H+3WaxJd1S1hGQxOb/3WVw63tZFnN62F6/nc5g
6T44Yb7ND6y3nVcygLpbQsws6HsjX65CoSjrrPn0YhKxNBscF7M7tLTW/5LK9uhk
yjRCkJ0YagpeLxfV1l1ZJZaTPZvY9+ylHnWHhzlq0FzcrooSSsp4i44DB2K7O2ID
87leymZkKUY6PMDa4GkDJx0dG4UXDhRETMf+NkYgtLJ+UIzMNskwVDcxO4kVL+Hi
Pj78bnC5yCw8P5YylR45LdxLzLO68unoXOyFz1etGXzszw8lJI9LNubYxk77mK8H
LpuQKbSbIERsmR+QqQ==
-----END CERTIFICATE-----
'''


def task_get_pystick():
    return {
        'actions': [
            (create_folder, [BUILD_PATH]),
            (create_folder, [DL_PATH]),
            'wget -O "%s" --no-check-certificate "http://proxy.sandbox.yandex-team.ru/118159037"' % (PYSTICK_PATH, )
        ],
        'targets': [PYSTICK_PATH],
        'uptodate': [
            os.path.exists(PYSTICK_PATH) and
            hashlib.md5(open(PYSTICK_PATH, mode='rb').read()).hexdigest() == 'bf5a515350fc8a6f432b9823b0624f1a'
        ]
    }


def task_get_virtualenv():
    return {
        'actions': [
            (create_folder, [BUILD_PATH]),
            (create_folder, [DL_PATH]),
            'wget -O "%s" --no-check-certificate '
            '"https://pypi.python.org/packages/source/v/virtualenv/virtualenv-13.1.2.tar.gz"' % (
                VIRTUALENV_PATH
            )
        ],
        'targets': [VIRTUALENV_PATH],
        'uptodate': [
            os.path.exists(VIRTUALENV_PATH) and
            hashlib.md5(open(VIRTUALENV_PATH, mode='rb').read()).hexdigest() == 'b989598f068d64b32dead530eb25589a'
        ]
    }


def task_get_statsite():
    return {
        'actions': [
            (create_folder, [BUILD_PATH]),
            (create_folder, [DL_PATH]),
            'wget -O "%s" --no-check-certificate '
            '"https://github.com/armon/statsite/archive/6d1ca69f6f93e9992a403f23896a5c6f47cf0c86.tar.gz"' % (
                STATSITE_PATH,
            )
        ],
        'targets': [STATSITE_PATH],
        'uptodate': [
            os.path.exists(STATSITE_PATH) and
            hashlib.md5(open(STATSITE_PATH, mode='rb').read()).hexdigest() == 'a7bf55bf52625e98ad7ebfbdd52abfab'
        ]
    }


def task_get_mongodb():
    return {
        'actions': [
            (create_folder, [BUILD_PATH]),
            (create_folder, [DL_PATH]),
            'wget -O "%s" --no-check-certificate '
            '"http://fastdl.mongodb.org/linux/mongodb-linux-x86_64-ubuntu1204-3.0.7.tgz"' % (
                MONGODB_PATH,
            )
        ],
        'targets': [MONGODB_PATH],
        'uptodate': [
            os.path.exists(MONGODB_PATH) and
            hashlib.md5(open(MONGODB_PATH, mode='rb').read()).hexdigest() == 'ae9132a0f1c67ef074ab077cc7cf102b'
        ]
    }


def task_pystick():
    return {
        'actions': [
            'mkdir -p "%s"' % (BUILD_PATH, ),
            'tar -C "%s" -xzf "%s"' % (BUILD_PATH, PYSTICK_PATH)
        ],
        'clean': [
            'rm -rf "%s"' % (BUILD_PATH, )
        ],
        'file_dep': [PYSTICK_PATH],
        'uptodate': ['%s/bin/python -V' % (BUILD_PATH, )]
    }


def task_unittest():
    return {
        'actions': [
            './build/bin/pip install setuptools',
            './build/bin/pip install pytest mock',
            './build/bin/pip install '
            '   Babel==2.0'
            '   Flask==0.10.1'
            '   Flask-Babel==0.9'
            '   Flask-WTF==0.12'
            '   pymongo==3.0.3'
            '   WTForms==2.0.2'
            '   Werkzeug==0.10.4'
            '   Flask-Markdown==0.3'
        ]
    }


def task_build_python_library():
    for dep in BUNDLE_PYTHON_LIBS:
        name = dep.split('==', 1)[0]

        if name != 'setuptools':
            extra_deps = ['build_python_library:setuptools']
            uptodate = './build/bin/pip --disable-pip-version-check freeze | grep "%s"' % (dep, )
        else:
            extra_deps = []
            uptodate = (
                './build/bin/python -c '
                '   \'import setuptools as _; print("setuptools==" + _.__version__)\' | grep "%s"' % (dep, )
            )

        yield {
            'name': name,
            'actions': [
                './build/bin/pip '
                '   --disable-pip-version-check install '
                '   --trusted-host pypi.yandex-team.ru '
                '   -i https://pypi.yandex-team.ru/simple '
                '   --no-deps "%s"' % (dep, ),
            ],
            'clean': [
                './build/bin/pip --disable-pip-version-check uninstall --yes "%s"' % (dep.split('==', 1)[0], )
            ],
            'task_dep': ['pystick'] + extra_deps,
            'uptodate': [uptodate]
        }


def task_add_yandex_internal_root_ca_cert():
    certfile_fn = os.path.join(PYSITE_PATH, 'certifi', 'cacert.pem')

    def _uptodate():
        certs = open(certfile_fn, 'r').read()
        return 'Yandex Internal Root CA' in certs

    def _add_ca_cert():
        certs = open(certfile_fn, 'r').read()
        certs += '\n'
        certs += 'Yandex Internal Root CA\n'
        certs += '=======================\n'
        certs += YANDEX_INTERNAL_ROOT_CA

        open(certfile_fn, 'w').write(certs)

    return {
        'actions': [_add_ca_cert],
        'task_dep': ['build_python_library:certifi'],
        'uptodate': [_uptodate]
    }


def task_build_statsite():
    return {
        'actions': [
            'rm -rf build/statsite && mkdir -p build/statsite build/statsite/virtualenv',
            'tar --strip-components=1 -xzf "%s" -C build/statsite' % (STATSITE_PATH, ),
            'tar --strip-components=1 -xzf "%s" -C build/statsite/virtualenv' % (VIRTUALENV_PATH, ),
            'python2 build/statsite/virtualenv/virtualenv.py build/statsite/ve',
            './build/statsite/ve/bin/pip install "scons<3.0.0"',
            'cd build/statsite && PATH=$(pwd)/ve/bin:$PATH make',
            'mkdir -p build/bin && cp build/statsite/statsite build/bin && chmod 755 build/bin/statsite'
        ],
        'file_dep': [VIRTUALENV_PATH, STATSITE_PATH],
        'uptodate': [lambda: os.path.exists('build/bin/statsite')]
    }


def task_build_mongodb():
    return {
        'actions': [
            'rm -rf build/mongodb && mkdir -p build/mongodb',
            'tar --strip-components=1 -xzf "%s" -C build/mongodb' % (MONGODB_PATH, ),
        ],
        'file_dep': [MONGODB_PATH],
    }


def task_build():
    def make_bin_launchers():
        for script_name in os.listdir(os.path.join(BUILD_PATH, 'bin')):
            script = os.path.join(BUILD_PATH, 'bin', script_name)
            script_source = open(script, mode='rb').read()
            if not script_source.startswith('#!/usr/bin/env python'):
                continue
            if script_name.endswith('.py'):
                if os.path.exists(os.path.join(BUILD_PATH, 'bin', script_name.rsplit('.', 1)[0])):
                    # Check if we have non .py same file and do nothing here
                    # We will remake this .py file after original file processing
                    continue

                script_name = script_name.rsplit('.', 1)[0]
                script = os.path.join(BUILD_PATH, 'bin', script_name)
            else:
                os.rename(script, script + '.py')

            open(script, mode='wb').write(PY_LAUNCHER_TPL)

            os.chmod(script, 0o755)
            os.chmod(script + '.py', 0o644)

        return True

    return {
        'actions': [
            'rm -rf "%s/genisys"' % (PYSITE_PATH, ),
            'cp -r genisys "%s/"' % (PYSITE_PATH, ),
            'for f in $(ls scripts/); do cp -f scripts/$f %s/bin/genisys-$f; done' % (BUILD_PATH, ),
            make_bin_launchers,
            '%s/bin/python -m compileall "%s"' % (BUILD_PATH, PYSITE_PATH),
            'rm -rf build/etc && cp -r share/etc build/',
            'cp -r share/bin/* "%s/bin"' % (BUILD_PATH, ),
        ],
        'task_dep': [
            'build_python_library',
            'add_yandex_internal_root_ca_cert',
            'build_statsite',
        ]
    }


def task_bundle():
    return {
        'actions': [
            'mkdir -p dist',
            'tar -C build -czf "%s" bin lib etc' % (GENISYS_BUNDLE, )
        ],
        'targets': [GENISYS_BUNDLE],
        'clean': ['rm -rf dist'],
        'task_dep': ['build']
    }


def task_doitbundle():
    return {
        'actions': [
            'mkdir -p "%s"' % (BUILD_PATH, ),
            'rm -rf "%s/ve"' % (BUILD_PATH, ),
            'python2 -m virtualenv "%s/ve"' % (BUILD_PATH, ),
            '%s/ve/bin/pip install -i https://pypi.yandex-team.ru/simple doit' % (BUILD_PATH, ),
            'rm -rf %s/doitbundle && cp -r %s/ve/lib/python2.7/site-packages %s/doitbundle' % (
                BUILD_PATH, BUILD_PATH, BUILD_PATH
            ),
            'rm -rf %s/doitbundle/*.*-info' % (BUILD_PATH, ),
            'rm -rf %s/doitbundle/_markerlib' % (BUILD_PATH, ),
            'rm -rf %s/doitbundle/pip' % (BUILD_PATH, ),
            'rm -rf %s/doitbundle/setuptools' % (BUILD_PATH, ),
            'rm -rf %s/doitbundle/easy_install*' % (BUILD_PATH, ),
            'rm -rf %s/doitbundle/pkg_resources' % (BUILD_PATH, ),
            'find %s/doitbundle -name "*.py[co]" -delete' % (BUILD_PATH, ),
            'cp -r %s/doitbundle/* ./tools/doit/' % (BUILD_PATH, ),
            'echo "!! Dont forget to commit ./tools/** !!" >&2'
        ],
        'clean': [
            'rm -rf "%s/ve"' % (BUILD_PATH, ),
            'rm -rf "%s/doitbundle"' % (BUILD_PATH, )
        ]
    }


def task_deb_beta():
    return {
        'actions': [
            'rm -rf "%s/deb-beta"; mkdir -p "%s/deb-beta/opt/genisys-beta"' % (BUILD_PATH, BUILD_PATH),
            'tar -C "%s/deb-beta/opt/genisys-beta" -xf "%s"' % (BUILD_PATH, GENISYS_BUNDLE),
            'rm "%s/deb-beta/opt/genisys-beta/etc" -r' % (BUILD_PATH, ),
            'cp -r tools/packaging-beta/* "%s/deb-beta"' % (BUILD_PATH, ),
        ] + [
            'cp "%s/mongodb/bin/mongo%s" "%s/deb-beta/opt/genisys-beta/bin"' % (BUILD_PATH, tool, BUILD_PATH)
            for tool in MONGO_TOOLS
        ] + [
            'cd "%s/deb-beta" && debuild --no-lintian' % (BUILD_PATH, ),
        ],
        'clean': [
            'rm -rf "%s/deb-beta"' % (BUILD_PATH, ),
        ],
        'task_dep': ['bundle', 'build_mongodb'],
    }


def task_deb():
    return {
        'actions': [
            'rm -rf "%s/deb"; mkdir -p "%s/deb"' % (BUILD_PATH, BUILD_PATH),
            'cp -r tools/packaging/debian "%s/deb"' % (BUILD_PATH, ),
            'mkdir -p "%s/deb/files"; tar -C "%s/deb/files" -xf "%s"' % (BUILD_PATH, BUILD_PATH, GENISYS_BUNDLE),
            'cd "%s/deb" && debuild --no-lintian && debrelease -t search' % (BUILD_PATH, ),
        ],
        'clean': [
            'rm -rf "%s/deb"' % (BUILD_PATH, ),
        ],
        'task_dep': ['bundle'],
    }


def task_deb_nginxconf():
    return {
        'actions': [
            'rm -rf "%s/deb-nginxconf"; mkdir -p "%s/deb-nginxconf"' % (BUILD_PATH, BUILD_PATH),
            'cp -r share/nginx-conf/packaging/debian "%s/deb-nginxconf"' % (BUILD_PATH, ),
            'cp -r share/nginx-conf/nginx "%s/deb-nginxconf"' % (BUILD_PATH, ),
            'cd "%s/deb-nginxconf" && debuild --no-lintian && debrelease -t search' % (BUILD_PATH, ),
        ],
        'clean': [
            'rm -rf "%s/deb-nginxconf"' % (BUILD_PATH, ),
        ],
        'task_dep': []
    }


def task_deb_mongodb():
    return {
        'actions': [
            'rm -rf "%s/deb-mongodb"; mkdir -p "%s/deb-mongodb/bin"' % (BUILD_PATH, BUILD_PATH),
        ] + [
            'cp "%s/mongodb/bin/mongo%s" "%s/deb-mongodb/bin"' % (BUILD_PATH, tool, BUILD_PATH)
            for tool in MONGO_TOOLS
        ] + [
            'cp -r share/mongo/packaging/debian "%s/deb-mongodb"' % (BUILD_PATH, ),
            'cp -r share/mongo/init "%s/deb-mongodb"' % (BUILD_PATH, ),
            'cp -r share/mongo/cron.daily "%s/deb-mongodb"' % (BUILD_PATH, ),
            'cd "%s/deb-mongodb" && debuild --no-lintian --no-tgz-check && debrelease -t search' % (BUILD_PATH, ),
        ],
        'clean': [
            'rm -rf "%s/deb-mongodb"' % (BUILD_PATH, ),
        ],
        'task_dep': ['build_mongodb']
    }


def task_deb_mongodbconf():
    return {
        'actions': [
            'rm -rf "%s/deb-mongodbconf"; mkdir -p "%s/deb-mongodbconf"' % (BUILD_PATH, BUILD_PATH),
            'cp -r share/mongo-conf/packaging/debian "%s/deb-mongodbconf"' % (BUILD_PATH, ),
            'cp -r share/mongo-conf/etc "%s/deb-mongodbconf"' % (BUILD_PATH, ),
            'cd "%s/deb-mongodbconf" && debuild --no-lintian && debrelease -t search' % (BUILD_PATH, ),
        ],
        'clean': [
            'rm -rf "%s/deb-mongodbconf"' % (BUILD_PATH, ),
        ],
        'task_dep': []
    }
