import traceback
import smtplib
from email.message import EmailMessage
from email.headerregistry import Address

import requests
from jinja2 import Environment, PackageLoader

from genisys.toiler import config


def _get_email_recipients(recps):
    recp_users = set()
    recp_groups = []
    for recp in recps:
        if recp.startswith('group:'):
            recp_groups.append(recp.split(':', 1)[1])
        else:
            recp_users.add(recp)
    extra_errors = []
    for group in recp_groups:
        try:
            resp = requests.get(
                config.STAFF_URI + '/persons',
                params={'official.is_dismissed': 'false',
                        '_fields': 'login',
                        '_limit': config.EMAIL_MAX_RECIPIENTS,
                        '_query': 'department_group.ancestors.url=="{0}" or '
                                  'groups.group.url=="{0}"'.format(
                                      group.replace('"', '\\"')
                                   )},
                headers=config.STAFF_HEADERS, timeout=config.STAFF_TIMEOUT,
                allow_redirects=False
            )
            resp.raise_for_status()
            recp_users.update(sorted(rec['login']
                                     for rec in resp.json()['result']))
        except:
            extra_errors.append('Failed to get users of group %r. %s' %
                                (group, traceback.format_exc()))
    return sorted(recp_users)[:config.EMAIL_MAX_RECIPIENTS], extra_errors


def email(send_to, subject_template, body_template, context):
    send_to, extra_errors = _get_email_recipients(send_to)
    if not send_to:
        raise RuntimeError("no one to send email to! "
                           "extra errors: %r" % (extra_errors, ))
    env = Environment(loader=PackageLoader('genisys.toiler',
                                           'email_templates'),
                      extensions=['jinja2.ext.i18n', 'jinja2.ext.with_'])
    env.globals.update({'config': config})
    env.install_null_translations(newstyle=True)
    body_template = env.get_template(body_template)
    subject_template = env.get_template(subject_template)
    msg = EmailMessage()
    msg['From'] = Address(config.EMAIL_ADDRESS_FROM_ALIAS,
                          config.EMAIL_ADDRESS_FROM_ADDRESS)
    msg['To'] = ", ".join('%s@yandex-team.ru' % (addressee, )
                          for addressee in send_to)
    msg['Subject'] = subject_template.render(context).strip()
    msg.set_content(body_template.render(context, extra_errors=extra_errors))
    with smtplib.SMTP(config.SMTP_SERVER, config.SMTP_PORT) as smtp:
        smtp.send_message(msg)
