import itertools
import operator
import hashlib
import json
from functools import reduce
from collections import defaultdict

from genisys.toiler import base, email, config


def section_vtype_processor(database, record, forced):
    base.update_volatiles_atime(database, 'section', [record['key']])

    if not record['source']['rules']:
        yield "rules list is empty, returning empty result"
        return {'hosts': dict(), 'configs': dict()}, record['meta']

    selectors, postpone = yield from _get_selectors(database, record)

    if record['source']['stype'] == 'sandbox_resource':
        resources, postpone2 = yield from _get_resources(database, record)
        postpone = postpone or postpone2

    if record['mtime'] is None or record['ctime'] >= record['utime']:
        yield "record has NOT been updated since creation / edit"
    else:
        # here we deliberately compare selectors' mtime to section record
        # utime because changing of selector doesn't necessarily mean
        # the resulting calculated value of the section record changed.
        # having section's utime ahead of all linked records' mtime means
        # we have already considered new values of those
        updated_selectors = [key for key, selector in selectors.items()
                             if selector['mtime'] and
                             selector['mtime'] >= record['utime']]
        if not updated_selectors:
            yield "no selectors updated since the record last update"
        else:
            yield "some selectors were updated since the record " \
                  "last update: {}".format(updated_selectors)
        updated_resources = None
        if record['source']['stype'] == 'sandbox_resource':
            # same thing here as above with selectors
            updated_resources = [key for key, resource in resources.items()
                                 if resource['mtime'] >= record['utime']]
            if not updated_resources:
                yield "no resources updated since the record last update"
            else:
                yield "some resources were updated since the record last " \
                      "update: {}".format(updated_resources)
        if not postpone:
            if not updated_selectors and not updated_resources:
                if forced:
                    yield "nothing has changed, but we're forced to do " \
                          "update anyway"
                else:
                    yield "nothing has changed, returning old value"
                    return record['value'], record['meta']

    if postpone:
        yield "updating is being postponed"
        return None, record['meta']

    # preparing arguments for _merge_host_*() functions
    selectors = {key: set(selector['value'])
                 for key, selector in selectors.items()}
    # list of sets of hostnames in order of rules (high to low priority)
    # with None in place where rules have no selector (apply to all hosts)
    hosts = [reduce(operator.and_, [selectors[key]
                                    for key in rule['selector_keys']])
             if rule['selector_keys'] else None
             for rule in record['source']['rules']]
    rule_names = [rule['rule_name'] for rule in record['source']['rules']]
    if record['source']['stype'] == 'sandbox_resource':
        # list of resources info in the same order of rules
        resources = [resources[rule['resource_key']]['value']
                     for rule in record['source']['rules']]
        hosts, configs = yield from _merge_hosts_resources(
            hosts, resources, rule_names
        )
    else:
        configs = [rule['config'] for rule in record['source']['rules']]
        hosts, configs = yield from _merge_hosts_configs(
            hosts, configs, rule_names
        )

    for value in configs.values():
        value['config_hash'] = _get_config_hash(value['config'])

    yield "got {} unique configurations".format(len(configs))

    return {'hosts': hosts, 'configs': configs}, record['meta']

section_vtype_processor.RESULT_TTL = config.SECTION_RESULT_TTL


def _get_selectors(database, record):
    postpone = False
    selector_keys = list(itertools.chain(*(
        rule['selector_keys'] for rule in record['source']['rules']
    )))
    selectors = base.get_volatiles(database, "selector", selector_keys)
    num_needed_selectors = len(set(selector_keys))
    yield "got {} of {} needed selectors".format(len(selectors),
                                                 num_needed_selectors)
    if len(selectors) != num_needed_selectors:
        # this should not happen since we update atime of selectors every
        # time we update section containing those
        yield "selectors {} are missing from volatile collection!".format(
            [key for key in selector_keys if key not in selectors]
        )
        postpone = True

    # count selectors with no mtime (meaning those have not been resolved
    # ever yet).
    no_data_selectors = [key for key, selector in selectors.items()
                         if selector['mtime'] is None]
    if no_data_selectors:
        yield "some selectors have no results available yet: {}".format(
            no_data_selectors
        )
        postpone = True

    if record['meta'].get('reresolve_selectors', True):
        # count selectors with update time less than update time
        # of the section. section volatile ctime changes whenever
        # its source changed (when user edited section / rules).
        outdated_selectors = [key for key, selector in selectors.items()
                              if selector['utime'] is not None
                                 and selector['utime'] < record['ctime']]
        if outdated_selectors:
            yield "some selectors are outdated: {}".format(outdated_selectors)
            postpone = True
    else:
        yield "skipping check of selectors update time"

    broken_selectors = defaultdict(lambda: {'rule_names': set()})
    notified_on_broken_selectors = set(record['meta'].get(
        'notified_on_broken_selectors', ()
    ))
    new_broken_selectors = set()
    for rule in record['source']['rules']:
        for selector_key in rule['selector_keys']:
            selector = selectors.get(selector_key)
            if selector is None:
                if selector_key in notified_on_broken_selectors:
                    continue
                broken_selector = broken_selectors[selector_key]
                broken_selector['rule_names'].add(rule['rule_name'])
                broken_selector.update({
                    'key': selector_key,
                    'status': 'missing',
                    'selector': 'n/a',
                    'proclog': []
                })
                new_broken_selectors.add(selector_key)
                continue
            if selector['last_status'] != 'error':
                if selector_key in notified_on_broken_selectors:
                    # selector is no longer broken
                    yield "selector {} was broken, no it's in status " \
                          "{}".format(selector['key'], selector['last_status'])
                    notified_on_broken_selectors.remove(selector_key)
                continue
            if selector_key in notified_on_broken_selectors:
                # already notified on this
                continue
            fail_duration = selector['ttime'] - (selector['utime']
                                                 or selector['ctime'])
            pcount = selector.get('pcount', 0)
            yield "selector {} has been broken for {}s, postpone " \
                  "count is {}".format(selector['key'], fail_duration, pcount)
            if fail_duration < config.SECTION_MAX_SELECTOR_FAIL_DURATION \
                    or pcount < config.SECTION_MAX_SELECTOR_POSTPONES:
                yield "it's too early to alarm"
                continue
            yield "will notify on it"
            broken_selectors[selector_key]['rule_names'].add(rule['rule_name'])
            broken_selectors[selector_key].update({
                'key': selector_key,
                'status': 'failed',
                'selector': selector['raw_key'],
                'proclog': selector['proclog'],
                'fail_duration': fail_duration,
                'pcount': pcount,
            })
            new_broken_selectors.add(selector_key)

    if new_broken_selectors:
        broken_selectors = sorted(broken_selectors.values(),
                                  key=lambda x: x['key'])
        yield from _notify_on_broken_selectors(database, record,
                                               broken_selectors)
    all_broken_selectors = sorted(
        notified_on_broken_selectors.union(new_broken_selectors)
    )
    if record['meta'].get('notified_on_broken_selectors') \
            or all_broken_selectors:
        record['meta']['notified_on_broken_selectors'] = all_broken_selectors

    if record['meta'].get('notified_on_broken_selectors'):
        yield "some selectors are broken: {}".format(
            record['meta']['notified_on_broken_selectors']
        )
        postpone = True

    return selectors, postpone

def _get_resources(database, record):
    postpone = False
    # here we fetch sandbox releases list just in order to update
    # its access time. we don't use the result, we only want the
    # list of releases to be touched. this is the only place where
    # we do so. when section stops updating so does its releases list
    rkey = record['source']['sandbox_releases_key']
    res = base.get_volatiles(database, "sandbox_releases", [rkey])
    if res:
        # this logging is for debug/informational purpose only
        yield "updated releases list atime"
        if res[rkey]['value'] is None:
            yield "releases list value is not available yet"
    else:
        # this should not happen. continue anyway, it's not our business
        yield "NO releases list is available!"

    resource_keys = [rule['resource_key']
                     for rule in record['source']['rules']]
    resources = base.get_volatiles(database, "sandbox_resource",
                                   resource_keys)
    num_needed_resources = len(set(resource_keys))
    yield "got {} of {} needed resource infos".format(len(resources),
                                                      num_needed_resources)
    if len(resources) != num_needed_resources:
        # this should not happen. saving section again might help
        # (all of volatiles it depends on get reinserted / updated)
        yield "info on resources {} is missing from volatile " \
              "collection!".format([key for key in resource_keys
                                    if key not in resources])
        postpone = True

    no_data_resources = [key for key, resource in resources.items()
                         if resource['mtime'] is None]
    if no_data_resources:
        # we don't demand that resources were updated strictly before
        # we can start updating section, but we still need at least
        # previous values.
        yield "some resources have no results availble yet: {}".format(
            no_data_resources
        )
        postpone = True

    return resources, postpone

def _merge_hosts_configs(hosts, configs, rule_names):
    stack = [(i, ) for i in range(len(configs))]
    hosts_by_combination = {(i, ): i_hosts for i, i_hosts in enumerate(hosts)}
    cfgs_by_combination = {(i, ): i_cfg for i, i_cfg in enumerate(configs)}
    while stack:
        cmb = stack.pop()
        for i in range(cmb[-1] + 1, len(configs)):
            if hosts_by_combination[cmb] is None:
                intersection = hosts[i]
            elif hosts[i] is None:
                intersection = hosts_by_combination[cmb]
            else:
                intersection = hosts_by_combination[cmb].intersection(hosts[i])
                if not intersection:
                    continue
            combination = cmb + (i, )
            hosts_by_combination[combination] = intersection
            cfgs_by_combination[combination] = _overlay_config(
                configs[i], cfgs_by_combination[cmb]
            )
            stack.append(combination)
        yield
    hosts = defaultdict(int)
    configs = {}
    no_selector_cmb = set()
    matched_rules = defaultdict(set)
    for cmb in hosts_by_combination:
        key = sum(1 << idx for idx in cmb)
        configs[key] = cfgs_by_combination[cmb]
        if hosts_by_combination[cmb] is None:
            no_selector_cmb.update(cmb)
            continue
        for host in hosts_by_combination[cmb]:
            hosts[host] |= key
        matched_rules[key].update(cmb)
    # replacing bitmaps, made of rule ids combination, with surrogate keys,
    # so that we do not go over 1<<64 in case of having more than 64 rules.
    # otherwise the resulting value fails to serialize to msgpack, which
    # only allows integers up to unsigned long.
    all_bitmaps = {bitmap: i + 1
                   for i, bitmap in enumerate(sorted(set(hosts.values())))}
    res_configs = {
        all_bitmaps[key]: {
            'config': configs[key],
            'matched_rules': [rule_names[idx]
                              for idx in sorted(matched_rules[key])]}
        for key in hosts.values()
    }
    hosts = {host: all_bitmaps[bitmap] for host, bitmap in hosts.items()}
    if no_selector_cmb:
        no_selector_cmb = tuple(sorted(no_selector_cmb))
        res_configs[-1] = {
            'config': cfgs_by_combination[no_selector_cmb],
            'matched_rules': [rule_names[idx] for idx in no_selector_cmb],
        }
    return hosts, res_configs

def _overlay_config(cfg, overlay):
    result = cfg.copy()
    for key in overlay:
        if not key in result:
            result[key] = overlay[key]
            continue
        both_dicts = isinstance(result[key], dict) \
                     and isinstance(overlay[key], dict)
        if not both_dicts:
            result[key] = overlay[key]
            continue
        result[key] = _overlay_config(result[key], overlay[key])
    return result

def _merge_hosts_resources(hosts, resources, rule_names):
    res_hosts = {}
    res_configs = {}
    default_config_idx = None
    for i, i_hosts in enumerate(hosts):
        if i_hosts is None:
            for j_hosts in hosts[i:]:
                if j_hosts is None:
                    continue
                for host in j_hosts:
                    res_hosts.setdefault(host, -1)
            default_config_idx = i
            break
        for host in i_hosts:
            if host not in res_hosts:
                res_hosts[host] = i
        yield
    all_config_keys = set(res_hosts.values())
    res_configs = {
        key: {'config': resources[key],
              'matched_rules': [rule_names[key]]}
        for key in all_config_keys if key >= 0
    }
    if default_config_idx is not None:
        res_configs[-1] = {
            'config': resources[default_config_idx],
            'matched_rules': [rule_names[default_config_idx]]
        }
    return res_hosts, res_configs

def _get_config_hash(config):
    serialized = json.dumps(config, sort_keys=True).encode('utf8')
    return hashlib.sha1(serialized).hexdigest()

def _notify_on_broken_selectors(database, record, broken_selectors):
    yield "sending notifications on newly broken " \
          "selectors: {}".format(broken_selectors)
    secpath = record['raw_key']
    all_owners = set(record['meta']['owners'])
    if secpath:
        ancestor_paths = secpath.split('.')
        ancestor_keys = [base.volatile_key_hash('.'.join(ancestor_paths[:i]))
                         for i in range(len(ancestor_paths) + 1)]
        secrecs = base.get_volatiles(database, 'section', ancestor_keys,
                                     with_values=False)
        for section in secrecs.values():
            all_owners.update(section['meta']['owners'])
    all_owners = sorted(all_owners)
    yield "all owners of section {} are {}".format(secpath, all_owners)
    email.email(send_to=all_owners,
                subject_template='broken_selectors_subj.txt',
                body_template='broken_selectors.txt',
                context={'broken_selectors': broken_selectors,
                         'secrec': record})
