import socket
import os

import flask
from pymongo import MongoClient
from flask.ext.babel import Babel
from flask_wtf.csrf import CsrfProtect

from genisys.web.model import MongoStorage
from genisys.web import route
from genisys.web import auth
from genisys.web import filters
from genisys.web import stats


def wrap_in_force_scheme_middleware(app, scheme):
    def wrapped(environ, start_response):
        environ['wsgi.url_scheme'] = scheme
        return app(environ, start_response)
    return wrapped


def make_app():
    app = flask.Flask('genisys.web')

    app.config.from_object('genisys.web.config')
    app.config.from_envvar('GENISYS_WEB_CONFIG')

    stats.set_up(app)

    auth.set_up(app)

    mongoclient = MongoClient(
        app.config['MONGODB_URI'],
        serverSelectionTimeoutMS=app.config['MONGODB_SERVER_SELECTION_TIMEOUT']
    )
    app.storage = MongoStorage(mongoclient, app.config['MONGODB_DB_NAME'])

    app.jinja_env.add_extension('jinja2.ext.i18n')
    app.jinja_env.add_extension('jinja2.ext.with_')
    Babel(app)
    csrf = CsrfProtect(app)

    app.jinja_env.filters.update(filters.jinja_filters)
    app.jinja_env.tests.update(filters.jinja_tests)
    app.jinja_env.globals.update(filters.jinja_globals)

    route.setup_routes(app, csrf)

    forced_scheme = app.config.get('FORCE_URL_SCHEME')
    if forced_scheme:
        app.wsgi_app = wrap_in_force_scheme_middleware(app.wsgi_app,
                                                       forced_scheme)

    hostname = socket.gethostname()
    pid = os.getpid()
    addr = app.config['WSGI_LISTEN_ADDRESS']
    app.logger.info('running on %r host %r, pid %d', addr, hostname, pid)

    app.jinja_env.globals.update({'node_info': {
        'hostname': hostname,
        'pid': os.getpid(),
        'wsgi_listen_address': addr,
    }})

    @app.errorhandler(404)
    def page_not_found(error):
        html = flask.render_template('err_404.html')
        return app.response_class(html, status=404, content_type='text/html')

    @app.errorhandler(Exception)
    def server_error(error):
        app.logger.exception(error)
        html = flask.render_template('err_500.html', hostname=hostname,
                                     addr=addr, pid=pid)
        return app.response_class(html, status=500)

    return app
